#include "log_viewer.h"

#include <library/cpp/build_info/build_info.h>
#include <library/cpp/svnversion/svnversion.h>
#include <library/cpp/eventlog/events_extension.h>
#include <library/cpp/eventlog/dumper/common.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/colorizer/colors.h>

#include <util/string/split.h>
#include <util/string/cast.h>
#include <util/string/builder.h>

namespace NInfra::NViewLog {

void TViewSelectedFields::ProcessEvent(const TEvent* event, TDataUnit& dataUnit) const {
    for (auto& processor : EventProcessors_) {
        processor(event, dataUnit);
    }
}

void TViewSelectedFields::PrintDataUnit(const TDataUnit& dataUnit) const {
    if (dataUnit.size() < 2) {
        return;
    }

    NColorizer::TColors colors;
    for (const auto& [key, vector] : dataUnit) {
        if (key != "Frame") {
            for (const auto& value : vector) {
                Cout << colors.BoldColor() <<  key << ":" << colors.OldColor()
                     << (key.find('/') != TString::npos ? " " : "\n") << value << "\n";
            }
        } else {
            Cout << colors.GreenColor() << colors.BoldColor() << key << "\t" << vector.back() << "\n" << colors.OldColor();
        }
    }
}

THolder<NLastGetopt::TOptsParseResult> TViewSelectedFields::AddAndParseOpts(NLastGetopt::TOpts& opts, int argc, const char* argv[]) {
    TString selectedFields;

    opts.AddLongOption(
            'f', "fields",
            "Select which particular fields or events to print."
            "Example: TPacketInfo,TLookupResult/RecordsNumber\n"
            "To print for each frame only TPacketInfo event and RecordsNumber field in TLookupResult event.")
        .Optional()
        .StoreResult(&selectedFields);

    THolder<NLastGetopt::TOptsParseResult> parsedOpts = MakeHolder<NLastGetopt::TOptsParseResult>(&opts, argc, argv);

    CreateEventProcessors(selectedFields);

    return parsedOpts;
}

void TViewSelectedFields::CreateEventProcessors(const TString& selectedFields) {
    TVector<TString> descriptors;
    StringSplitter(selectedFields).Split(',').SkipEmpty().Collect(&descriptors);

    NProtoBuf::TEventFactory* factory = NProtoBuf::TEventFactory::Instance();

    EventProcessors_.push_back([](const NInfra::TEvent* event, TDataUnit& data) {
        data.emplace("Frame", TVector<TString>(1, ToString(event->FrameId())));
    });

    for (const TString& descriptor : descriptors) {
        EventProcessors_.push_back(TViewSelectedFields::CreateEventProcessor(descriptor, factory));
    }
}

TEventProcessor TViewSelectedFields::CreateEventProcessor(const TString& eventDescriptor, NProtoBuf::TEventFactory* fac) const {
    size_t pos = eventDescriptor.find('/');
    const TString eventName = eventDescriptor.substr(0, pos);
    const TString fieldPath = (pos == TString::npos ? "" : eventDescriptor.substr(pos + 1));
    ui64 allowedId = fac->IdByName(eventName);

    return [allowedId, eventDescriptor, fieldPath](const TEvent* event, TDataUnit& dataUnit) {
        if (event->Id() != allowedId) {
            return;
        }

        const TString timestampString = TString::Join("Timestamp: ", ToString(event->Timestamp()));
        const TString fieldString = GetEventFieldAsString(event, fieldPath);
        const TString result = (eventDescriptor.find("/") != TString::npos
                                ? TString::Join(fieldString, "\n", timestampString)
                                : TString::Join(timestampString, "\n", fieldString));
        dataUnit[eventDescriptor].push_back(result);
    };
}

void TViewContainsEvent::ProcessEvent(const TEvent* event, TDataUnit& dataUnit) const {
    for (auto& processor : EventProcessors_) {
        processor(event, dataUnit);
    }
}

bool TViewContainsEvent::FilterFrame(const TDataUnit& dataUnit) const {
    for (auto& frameFilter : FrameFilters_) {
        if (!frameFilter(dataUnit)) {
            return false;
        }
    }
    return true;
}

void TViewContainsEvent::PrintDataUnit(const TDataUnit& dataUnit) const {
    if (dataUnit.size() < 2) {
        return;
    }
    NColorizer::TColors colors;

    const auto frameIt = dataUnit.find("Frame");
    Cout << colors.GreenColor() << colors.BoldColor() << frameIt->first << "\t" << frameIt->second.back() << "\n" << colors.OldColor();

    const auto& eventsVector = dataUnit.find("Events")->second;
    for (const auto& value : eventsVector) {
        Cout << value << "\n";
    }
}

THolder<NLastGetopt::TOptsParseResult> TViewContainsEvent::AddAndParseOpts(NLastGetopt::TOpts& opts, int argc, const char* argv[]) {
    THashMap<TString, TString> presetFilters;
    InitializePresetFilters(presetFilters);

    TString containsEvent;
    opts.AddLongOption(
            'c', "contains-event",
            "Only print frames that contain events whose particular field's value equals given value.\n"
            "Example: TPacketInfo/ResourceType/SomeResourceType,TLookupResult/RecordsNumber/1\n")
        .Optional()
        .StoreResult(&containsEvent);

    TString presetFilter;
    if (presetFilters.size()) {
        TStringBuilder filters;
        for (const auto& [key, value] : presetFilters) {
            filters << key << " " << value << "\n";
        }
        opts.AddLongOption(
                'p', "preset-filter",
                TString::Join("Use a preset frame filter.\n", filters))
            .Optional()
            .StoreResult(&presetFilter);
    }

    THolder<NLastGetopt::TOptsParseResult> parsedOpts = MakeHolder<NLastGetopt::TOptsParseResult>(&opts, argc, argv);

    if (presetFilter) {
        auto it = presetFilters.find(presetFilter);
        if (it == presetFilters.end()) {
            Cout << "Preset filters are:" << Endl;
            for (const auto& [key, value] : presetFilters) {
                Cout << key << " " << value << Endl;
            }
            ythrow yexception() << "Invalid preset filter index " << presetFilter << ".\n";
        }
        containsEvent = presetFilters[presetFilter];
    }

    InitializeProcessorsAndFilters(containsEvent);

    return parsedOpts;
}

void TViewContainsEvent::InitializeProcessorsAndFilters(const TString& containsEvent) {
    TVector<TString> descriptors;
    StringSplitter(containsEvent).Split(',').SkipEmpty().Collect(&descriptors);

    NProtoBuf::TEventFactory* factory = NProtoBuf::TEventFactory::Instance();

    EventProcessors_.push_back([](const NInfra::TEvent* event, TDataUnit& data) {
        data.emplace("Frame", TVector<TString>(1, ToString(event->FrameId())));
    });

    EventProcessors_.push_back([factory](const NInfra::TEvent* event, TDataUnit& data) {
        const TString timestampString = TString::Join("Timestamp: ", ToString(event->Timestamp()));
        const TString fieldString = GetEventFieldAsString(event, "");
        const TStringBuf eventName = factory->NameById(event->Id());

        TStringBuilder builder;
        builder << timestampString << "\n" << eventName << "\n" << fieldString;
        data["Events"].push_back(builder);
    });

    for (const TString& descriptor : descriptors) {
        EventProcessors_.push_back(TViewContainsEvent::CreateEventProcessor(descriptor, factory));
        FrameFilters_.push_back(TViewContainsEvent::CreateFrameFilter(descriptor));
    }
}

TEventProcessor TViewContainsEvent::CreateEventProcessor(const TString& eventDescriptor, NProtoBuf::TEventFactory* fac) const {
    size_t fieldPathDelimPosition = eventDescriptor.find('/');
    size_t valueDelimPosition  = eventDescriptor.find('/', fieldPathDelimPosition + 1);

    Y_ENSURE(fieldPathDelimPosition != TString::npos);
    Y_ENSURE(valueDelimPosition != TString::npos);

    const TString eventName = eventDescriptor.substr(0, fieldPathDelimPosition);
    const TString fieldPath = eventDescriptor.substr(fieldPathDelimPosition + 1, valueDelimPosition - fieldPathDelimPosition - 1);
    const TString value = eventDescriptor.substr(valueDelimPosition + 1);

    Y_ENSURE(eventName);
    Y_ENSURE(fieldPath);

    ui64 allowedId = fac->IdByName(eventName);

    return [allowedId, eventDescriptor, fieldPath, value](const TEvent* event, TDataUnit& dataUnit) {
        if (event->Id() != allowedId) {
            return;
        }

        const TString fieldString = GetEventFieldAsString(event, fieldPath);
        if (fieldString == value) {
            dataUnit[eventDescriptor].push_back("Matched");
        }
    };
}

TFrameFilter TViewContainsEvent::CreateFrameFilter(const TString& descriptor) const {
    return [descriptor](const TDataUnit& dataUnit) {
        auto it = dataUnit.find(descriptor);
        if (it == dataUnit.end()) {
            return false;
        }
        return it->second.back() == "Matched";
    };
}

void TLogViewer::Run() {
    ProcessLog();
    ProcessData();
    Print();
}

void TLogViewer::ProcessLog() {
    TEventIterator it(IterOptions_);

    while (const TEvent* event = it.SafeNext()) {
        size_t index = GetIndex(event->FrameId());
        TDataUnit& dataUnit = Data_[index];

        Viewlog_->ProcessEvent(event, dataUnit);
    }

    TData tempData;
    for (const auto& dataUnit : Data_) {
        if (Viewlog_->FilterFrame(dataUnit)) {
            tempData.push_back(dataUnit);
        }
    }
    Data_ = std::move(tempData);
}

void TLogViewer::Print() const {
    for (const auto& dataUnit : Data_) {
        Viewlog_->PrintDataUnit(dataUnit);
    }
}

size_t TLogViewer::GetIndex(ui64 frameId) {
    const auto [it, emplaced] = FrameIdToIndexMap_.try_emplace(frameId, Data_.size());
    if (emplaced) {
        Data_.push_back(TDataUnit());
    }

    return it->second;
}

TString GetEventFieldAsString(const TEvent* event, const TString& fieldPath) {
    if (!fieldPath) {
        // The last symbol is '\n', so it's deleted.
        return event->GetData().pop_back();
    }

    const NProtoBuf::Message* message = event->GetProto();

    Y_ENSURE(message);

    const google::protobuf::Descriptor* descriptor = message->GetDescriptor();
    const google::protobuf::Reflection* reflection = message->GetReflection();

    Y_ENSURE(descriptor);
    Y_ENSURE(reflection);

    TVector<TStringBuf> parts = StringSplitter(fieldPath).Split('/').SkipEmpty().ToList<TStringBuf>();
    TStringBuf lastPart(parts.back());
    parts.pop_back();

    for (auto part : parts) {
        auto fieldDescriptor = descriptor->FindFieldByName(TString(part));
        if (!fieldDescriptor) {
            ythrow yexception() << "Cannot deduce field \"" << part << "\".";
        }
        message = &reflection->GetMessage(*message, fieldDescriptor);
        descriptor = message->GetDescriptor();
        reflection = message->GetReflection();
    }

    const google::protobuf::FieldDescriptor* fieldDescriptor = descriptor->FindFieldByName(TString(lastPart));

    TString fieldValue = "";

    switch (fieldDescriptor->type()) {
        case google::protobuf::FieldDescriptor::Type::TYPE_DOUBLE:
            fieldValue = ToString(reflection->GetDouble(*message, fieldDescriptor));
            break;
        case google::protobuf::FieldDescriptor::Type::TYPE_FLOAT:
            fieldValue = ToString(reflection->GetFloat(*message, fieldDescriptor));
            break;
        case google::protobuf::FieldDescriptor::Type::TYPE_BOOL:
            fieldValue = ToString(reflection->GetBool(*message, fieldDescriptor));
            break;
        case google::protobuf::FieldDescriptor::Type::TYPE_INT32:
            fieldValue = ToString(reflection->GetInt32(*message, fieldDescriptor));
            break;
        case google::protobuf::FieldDescriptor::Type::TYPE_UINT32:
            fieldValue = ToString(reflection->GetUInt32(*message, fieldDescriptor));
            break;
        case google::protobuf::FieldDescriptor::Type::TYPE_INT64:
            fieldValue = ToString(reflection->GetInt64(*message, fieldDescriptor));
            break;
        case google::protobuf::FieldDescriptor::Type::TYPE_UINT64:
            fieldValue = ToString(reflection->GetUInt64(*message, fieldDescriptor));
            break;
        case google::protobuf::FieldDescriptor::Type::TYPE_STRING:
            fieldValue = ToString(reflection->GetString(*message, fieldDescriptor));
            break;
        case google::protobuf::FieldDescriptor::Type::TYPE_ENUM:
            {
                const NProtoBuf::EnumValueDescriptor* enumValueDescriptor = reflection->GetEnum(*message, fieldDescriptor);
                fieldValue = ToString(enumValueDescriptor->name());
            }
            break;
        default:
            throw yexception() << "GetEventFieldAsString for type " << fieldDescriptor->type_name() << " is not implemented.";
    }

    return fieldValue;
}

int ViewLogMode(int argc, const char* argv[], THolder<IViewLog> viewlog) {
    NLastGetopt::TOpts opts;

    opts.SetFreeArgsNum(1);
    opts.SetFreeArgTitle(0, "<eventlog>", "Event log file");

    TString start;
    TString end;
    NInfra::TIteratorOptions options;

    opts.AddLongOption(
            's', "start-time")
        .Optional()
        .StoreResult(&start);

    opts.AddLongOption(
            'e', "end-time")
        .Optional()
        .StoreResult(&end);

    opts.AddLongOption(
            'v', "version",
            "Print program version")
        .NoArgument()
        .Optional()
        .Handler([]() {
            Cout << GetProgramSvnVersion() << Endl;
            Cout << GetBuildInfo() << Endl;
            exit(0);
        });

    THolder<NLastGetopt::TOptsParseResult> parsedOpts(viewlog->AddAndParseOpts(opts, argc, argv));

    options.SetFileName(parsedOpts->GetFreeArgs()[0])
        .SetStartTime(ParseTime(start, NInfra::TIteratorOptions::MIN_START_TIME))
        .SetEndTime(ParseTime(end, NInfra::TIteratorOptions::MAX_END_TIME));

    TLogViewer viewer(std::move(viewlog), options);
    viewer.Run();

    return EXIT_SUCCESS;
}

int ViewLog(int argc, const char* argv[]) {
    TModChooser modChooser;

    for (const auto& mode : Singleton<TViewLogFactory>()->GetModes()) {
        modChooser.AddMode(mode.Name(), mode.MainFunction(), mode.Description());
    }

    try {
        return modChooser.Run(argc, argv);
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}

} // NInfra::NViewLog
