#pragma once

#include "event_iterator.h"

#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/generic/map.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/singleton.h>
#include <library/cpp/getopt/small/last_getopt.h>

#include <functional>

#define REGISTER_VIEWER(name, TViewLog, description) \
    static NInfra::NViewLog::TViewLogRegistrator name##_registrator( \
            #name, \
            [](int argc, const char** argv) { \
                return ViewLogMode(argc, argv, MakeHolder<TViewLog>()); \
            }, \
            description);

namespace NInfra::NViewLog {

using TDataUnit = TMap<TString, TVector<TString>>;
using TData = TVector<TDataUnit>;
using TEventProcessor = std::function<void(const TEvent*, TDataUnit&)>;
using TEventProcessors = TVector<TEventProcessor>;
using TFrameFilter = std::function<bool(const TDataUnit&)>;
using TFrameFilters = TVector<TFrameFilter>;

class IViewLog {
public:
    virtual ~IViewLog() = default;

    virtual void ProcessEvent(const TEvent* event, TDataUnit& dataUnit) const = 0;

    virtual bool FilterFrame(const TDataUnit& dataUnit) const = 0;

    virtual void ProcessData(TData& data) const = 0;

    virtual void PrintDataUnit(const TDataUnit&) const = 0;

    virtual THolder<NLastGetopt::TOptsParseResult> AddAndParseOpts(NLastGetopt::TOpts& opts, int argc, const char* argv[]) {
        return MakeHolder<NLastGetopt::TOptsParseResult>(&opts, argc, argv);
    }
};

class TViewLogMode {
public:
    TViewLogMode(const TString& name,
                 const std::function<int (int, const char**)>& mainFunction,
                 const TString& description)
        : Name_(name)
        , MainFunction_(mainFunction)
        , Description_(description) {}

    inline const TString& Name() const {
        return Name_;
    }

    inline const std::function<int (int, const char**)> MainFunction() const {
        return MainFunction_;
    }

    inline const TString& Description() const {
        return Description_;
    }

private:
    TString Name_;
    std::function<int (int, const char**)> MainFunction_;
    TString Description_;
};

class TViewLogFactory {
public:
    void Add(TViewLogMode&& mode) {
        if (!Names_.contains(mode.Name())) {
            Names_.insert(mode.Name());
            Factory_.push_back(std::move(mode));
        } else {
            Cdbg << "TViewLogMode " << mode.Name() << " already exists.";
        }
    }

    const TVector<TViewLogMode>& GetModes() const {
        return Factory_;
    }

private:
    TVector<TViewLogMode> Factory_;
    THashSet<TString> Names_;
};

class TViewLogRegistrator {
public:
    inline TViewLogRegistrator(const TString& name,
                               std::function<int (int, const char**)> function,
                               const TString& description) {
        Singleton<TViewLogFactory>()->Add(TViewLogMode(name, function, description));
    }
};

int ViewLogMode(int argc, const char* argv[], THolder<IViewLog> viewlog);

int ViewLog(int argc, const char* argv[]);

class TViewSelectedFields: public IViewLog {
public:
    void ProcessEvent(const TEvent* event, TDataUnit& dataUnit) const override;

    bool FilterFrame(const TDataUnit& dataUnit) const override {
        Y_UNUSED(dataUnit);
        return true;
    }

    void ProcessData(TData& data) const override {
        Y_UNUSED(data);
    }

    void PrintDataUnit(const TDataUnit& dataUnit) const override;

    THolder<NLastGetopt::TOptsParseResult> AddAndParseOpts(NLastGetopt::TOpts& opts, int argc, const char* argv[]) override;

private:
    void CreateEventProcessors(const TString& selectedFields);

    TEventProcessor CreateEventProcessor(const TString& descriptor, NProtoBuf::TEventFactory* factory) const;

    TEventProcessors EventProcessors_;
};

REGISTER_VIEWER(select_fields,
                TViewSelectedFields,
                "Select, which particular fields or events to print.")

class TViewContainsEvent: public IViewLog {
public:
    void ProcessEvent(const TEvent* event, TDataUnit& dataUnit) const override;

    bool FilterFrame(const TDataUnit& dataUnit) const override;

    void ProcessData(TData& data) const override {
        Y_UNUSED(data);
    }

    void PrintDataUnit(const TDataUnit& dataUnit) const override;

    THolder<NLastGetopt::TOptsParseResult> AddAndParseOpts(NLastGetopt::TOpts& opts, int argc, const char* argv[]) override;

private:
    virtual void InitializePresetFilters(THashMap<TString, TString>& presetFilters) const {
        Y_UNUSED(presetFilters);
    }

    void InitializeProcessorsAndFilters(const TString& containsEvent);

    TEventProcessor CreateEventProcessor(const TString& descriptor, NProtoBuf::TEventFactory* factory) const;

    TFrameFilter CreateFrameFilter(const TString& descriptor) const;

    TEventProcessors EventProcessors_;
    TFrameFilters FrameFilters_;
};

REGISTER_VIEWER(contains_event,
                TViewContainsEvent,
                "Print frames, that contain certain events and whose particular fields' values match.")

class TLogViewer {
public:
    TLogViewer(THolder<IViewLog> viewlog, TIteratorOptions options)
        : Viewlog_(std::move(viewlog))
        , IterOptions_(std::move(options)) {}

    void Run();
private:
    // Iterate over eventlog and process events.
    void ProcessLog();

    void ProcessData() {
        Viewlog_->ProcessData(Data_);
    }

    void Print() const;

    size_t GetIndex(ui64 frameId);

private:
    THolder<IViewLog> Viewlog_;
    TIteratorOptions IterOptions_;
    TData Data_;
    THashMap<ui64, size_t> FrameIdToIndexMap_;
};

TString GetEventFieldAsString(const NInfra::TEvent* event, const TString& fieldName);

} // NInfra::NViewLog
