#include "logger.h"

#include <infra/libs/unified_agent_log_backend/unified_agent_log_backend.h>

#include <library/cpp/eventlog/eventlog_int.h>
#include <library/cpp/logger/file.h>
#include <library/cpp/logger/rotating_file.h>

#include <util/string/cast.h>
#include <util/system/fs.h>

namespace NInfra {

TLogger::TLogger(TLoggerConfig cfg, std::function<void()> onWriteFail)
    : Format_((cfg.GetBackend() == TLoggerConfig_ELogBackend_FILE || cfg.GetBackend() == TLoggerConfig_ELogBackend_ROTATING_FILE)
        ? TLoggerConfig_ELogFormat_PROTO
        : TLoggerConfig_ELogFormat_JSON)
    , Level_(MakeAtomicShared<TAtomic>(FromString<ELogPriority>(cfg.GetLevel())))
{
    if (cfg.GetDuplicateToUnifiedAgent()) {
        Y_ENSURE(cfg.HasUnifiedAgentLogBackendConfig(), "Duplicating to unified agent but its log backend config is missing");
        if (cfg.GetUnifiedAgentLogBackendConfig().GetMeta().find("_log_name") == cfg.GetUnifiedAgentLogBackendConfig().GetMeta().end()) {
            (*cfg.MutableUnifiedAgentLogBackendConfig()->MutableMeta())["_log_name"] = cfg.GetPath();
        }
    }

    switch(cfg.GetBackend()) {
        case TLoggerConfig_ELogBackend_STDERR:
            Backend_ = MakeHolder<TStreamLogBackend>(&Cerr);
            break;
        case TLoggerConfig_ELogBackend_STDOUT:
            Backend_ = MakeHolder<TStreamLogBackend>(&Cout);
            break;
        case TLoggerConfig_ELogBackend_FILE:
            Backend_ = MakeHolder<TFileLogBackend>(cfg.GetPath());
            break;
        case TLoggerConfig_ELogBackend_ROTATING_FILE:
            if (cfg.GetRotatePath() != cfg.GetPath() + ".1") {
                NFs::Rename(cfg.GetRotatePath(), cfg.GetPath() + ".1");
            }
            Backend_ = MakeHolder<TRotatingFileLogBackend>(cfg.GetPath(), cfg.GetMaxLogSizeBytes(), cfg.GetRotatedFilesCount());
            break;
        case TLoggerConfig_ELogBackend_UNIFIED_AGENT:
            Y_ENSURE(!cfg.GetDuplicateToUnifiedAgent(), "Backend is already unified agent, no need to duplicate");
            Y_ENSURE(cfg.HasUnifiedAgentLogBackendConfig(), "Backend is unified agent but its config is missing");
            Backend_ = MakeHolder<NUnifiedAgentLogBackend::TUnifiedAgentLogBackend>(cfg.GetUnifiedAgentLogBackendConfig());
            break;
        default:
            throw yexception() << "Unknown log backend for NInfra::TLogger: '" << TLoggerConfig_ELogBackend_Name(cfg.GetBackend()) << "'";
    }

    if (cfg.GetDuplicateToUnifiedAgent()) {
        UnifiedAgentLogBackend_ = MakeHolder<NUnifiedAgentLogBackend::TUnifiedAgentLogBackend>(cfg.GetUnifiedAgentLogBackendConfig(), Backend_.Get());
        Logger_ = new TLog(MakeHolder<TThreadedLogBackend>(UnifiedAgentLogBackend_.Get(), cfg.GetQueueSize(), std::move(onWriteFail)));
    } else {
        Logger_ = new TLog(MakeHolder<TThreadedLogBackend>(Backend_.Get(), cfg.GetQueueSize(), std::move(onWriteFail)));
    }

    if (cfg.GetDuplicateToErrorBooster()) {
        if (!cfg.GetErrorBoosterClientConfig().HasFile() || cfg.GetErrorBoosterClientConfig().GetFile().empty()) {
            cfg.MutableErrorBoosterClientConfig()->SetFile(cfg.GetPath());
        }
        ErrorBoosterClient_ = new NErrorBoosterClient::TErrorBoosterClient(cfg.GetErrorBoosterClientConfig());
    }
}

} // namespace NInfra
