#include "event_iterator.h"
#include "logger.h"
#include "log_printer.h"
#include "test_common.h"

#include <infra/libs/logger/protos/events.ev.pb.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/string/vector.h>
#include <util/system/tempfile.h>

using namespace NInfra;
using namespace NInfra::NTestCommon;

namespace {

// It's important, that priorities are in ELogPriority order!
const TVector<TString> PRIORITIES = {"EMERG", "ALERT", "CRITICAL_INFO", "ERROR", "WARNING", "NOTICE", "INFO", "DEBUG", "RESOURCES"};

NInfra::TEventIterator* CreateIterator(const NInfra::TIteratorOptions& options) {
    return new NInfra::TEventIterator(options);
}

THashSet<TString> ReadEventLog(const TString& path) {
    THashSet<TString> result;
    NInfra::TIteratorOptions options;

    THolder<NInfra::TEventIterator> iter = THolder(CreateIterator(options.SetFileName(path)));
    const NInfra::TEvent* event;

    for (;;) {
        try {
            event = iter->Next();

            if (event) {
                if (event->Id() == NLogEvent::TTestMessage::ID) {
                    result.insert(event->Get<NLogEvent::TTestMessage>()->GetData());
                } else if (event->Id() == NLogEvent::TOtherTestMessage::ID) {
                    result.insert("NLogEvent::TOtherTestMessage");
                }
            } else {
                break;
            }
        } catch (...) {
            Cout.Flush();
            Cerr << "Error occured: " << CurrentExceptionMessage() << Endl;
        }
    }

    return result;
}

TVector<std::pair<ui64, TString>> ReadTimestamps(const TString& path, ui64 startTime = TIteratorOptions::MIN_START_TIME, ui64 endTime = TIteratorOptions::MAX_END_TIME) {
    TVector<std::pair<ui64, TString>> result;
    NInfra::TIteratorOptions options;
    options.SetFileName(path)
    .SetStartTime(startTime)
    .SetEndTime(endTime);

    THolder<NInfra::TEventIterator> iter = THolder(CreateIterator(options));
    const NInfra::TEvent* event;

    for (;;) {
        try {
            event = iter->Next();

            if (event) {
                if (event->Id() == NLogEvent::TTestMessage::ID) {
                    result.push_back({event->Timestamp(), event->Get<NLogEvent::TTestMessage>()->GetData()});
                }
            } else {
                break;
            }
        } catch (...) {
            Cout.Flush();
            Cerr << "Error occured: " << CurrentExceptionMessage() << Endl;
        }
    }

    return result;
}

TVector<std::pair<ui64, TString>> FilterTimestamps(const TVector<std::pair<ui64, TString>>& pairs, ui64 startTime, ui64 endTime) {
    TVector<std::pair<ui64, TString>>  result;

    for (const auto& elem : pairs) {
        if (elem.first >= startTime && elem.first <= endTime) {
            result.push_back(elem);
        }
    }

    return result;
}

TVector<ui64> GetEventsByEvList(const TString& path, const TVector<TString>& evList, bool includeExcludeFlag) {
    TVector<ui64> result;
    NInfra::TIteratorOptions options;
    options.IncludeExcludeFlag = includeExcludeFlag;
    options.SetFileName(path)
           .SetEventList(evList);

    THolder<NInfra::TEventIterator> iter = THolder(CreateIterator(options));
    const NInfra::TEvent* event = iter->Next();

    while (event) {
        result.push_back(event->Id());

        event = iter->Next();
    }

    return result;
}

TVector<ui64> GetEventsByFrameList(const TString& path, const TVector<ui64>& frameList) {
    TVector<ui64> result;
    NInfra::TIteratorOptions options;
    options.SetFileName(path)
           .SetFrameList(frameList);

    THolder<NInfra::TEventIterator> iter = THolder(CreateIterator(options));
    const NInfra::TEvent* event = iter->Next();

    while (event) {
        result.push_back(event->FrameId());

        event = iter->Next();
    }

    return result;
}

TVector<ui32> GetAllowedEvents(const TString& path, const std::function<bool(const TEvent*)>& isAllowed) {
    TVector<ui32> result;
    NInfra::TIteratorOptions options;
    options.SetFileName(path).SetEventPredicate(isAllowed);

    THolder<NInfra::TEventIterator> iter = THolder(CreateIterator(options));
    const NInfra::TEvent* event = iter->Next();

    while (event) {
        result.push_back(event->Id());

        event = iter->Next();
    }

    return result;
}

TVector<ui32> GetEventsByMaxLogPriority(const TString& path, ELogPriority maxLogPriority) {
    TVector<ui32> result;
    NInfra::TIteratorOptions options;
    options.SetFileName(path).SetMaxLogPriority(maxLogPriority);

    THolder<NInfra::TEventIterator> iter = THolder(CreateIterator(options));
    const NInfra::TEvent* event = iter->Next();

    while (event) {
        result.push_back(event->Id());

        event = iter->Next();
    }

    return result;
}

void LogAllTTestMessages(TLogger& logger, const TString handlePref = "") {
    auto frame = logger.SpawnFrame();

    for (const auto& priority : PRIORITIES) {
        frame->LogEvent(FromString<ELogPriority>(priority), NLogEvent::TTestMessage(handlePref + priority));
    }
}

TVector<std::pair<TString, TString>> ReadSourceLocations(const TString& path) {
    TVector<std::pair<TString, TString>> result;
    NInfra::TIteratorOptions options;
    options.SetFileName(path);

    THolder<NInfra::TEventIterator> iter = THolder(CreateIterator(options));
    const NInfra::TEvent* event;

    for (;;) {
        try {
            event = iter->Next();

            if (event) {
                if (event->Id() == NLogEvent::TTestMessage::ID) {
                    TStringBuilder sourceString;
                    if (TMaybe<TSourceLocation> sourceLocation = event->SourceLocation(); sourceLocation.Defined()) {
                        sourceString << *sourceLocation;
                    }
                    result.emplace_back(
                        event->Get<NLogEvent::TTestMessage>()->GetData(),
                        sourceString
                    );
                }
            } else {
                break;
            }
        } catch (...) {
            Cout.Flush();
            Cerr << "Error occured: " << CurrentExceptionMessage() << Endl;
        }
    }

    return result;
}

} // namespace

Y_UNIT_TEST_SUITE(Logger) {
    Y_UNIT_TEST(ParseLevel) {
        TString logFile = "eventlog";
        TTempFile tmp(logFile);

        for (const auto& priority : PRIORITIES) {
            UNIT_ASSERT_NO_EXCEPTION(TLogger(CreateLoggerConfig(priority, logFile)));
        }

        UNIT_ASSERT_EXCEPTION(TLogger(CreateLoggerConfig("42", logFile)), yexception);
    }

    Y_UNIT_TEST(SetLevel) {
        TString logFile = "eventlog_SetLevel";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("ERROR", logFile));

        TVector<TString> required;

        for (size_t i = 0; i < PRIORITIES.size(); ++i) {
            logger.SetLevel(FromString<ELogPriority>(PRIORITIES[i]));

            LogAllTTestMessages(logger, ToString(i));

            for (const auto& priority : PRIORITIES) {
                if (FromString<ELogPriority>(priority) <= FromString<ELogPriority>(PRIORITIES[i])) {
                    required.push_back(ToString(i) + priority);
                }
            }
        }

        logger.CloseLog();

        const THashSet<TString> events = ReadEventLog(logFile);

        UNIT_ASSERT_VALUES_EQUAL(required.size(), events.size());

        for (const TString& str : required) {
            UNIT_ASSERT(events.contains(str));
        }
    }

    Y_UNIT_TEST(SetLevelOnCreatedFrame) {
        TString logFile = "eventlog_SetLevelOnCreatedFrame";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("ERROR", logFile));

        {
            auto frame = logger.SpawnFrame();
            logger.SetLevel(ELogPriority::TLOG_ERR);
            frame->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TTestMessage("ERR_INFO"));
            frame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TTestMessage("ERR_ERR"));
            logger.SetLevel(ELogPriority::TLOG_INFO);
            frame->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TTestMessage("INFO_INFO"));
        }

        logger.CloseLog();

        const THashSet<TString> events = ReadEventLog(logFile);
        UNIT_ASSERT_VALUES_EQUAL(2, events.size());
    }

    Y_UNIT_TEST(EmptyMessage) {
        TString logFile = "eventlog_EmptyMessage";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("INFO", logFile));

        {
            auto frame = logger.SpawnFrame();
            frame->LogEvent(ELogPriority::TLOG_INFO, NLogEvent::TOtherTestMessage());
        }

        logger.CloseLog();

        const THashSet<TString> events = ReadEventLog(logFile);

        const TVector<TString> required{
            "NLogEvent::TOtherTestMessage"
        };

        UNIT_ASSERT_VALUES_EQUAL(required.size(), events.size());

        for (const TString& str : required) {
            UNIT_ASSERT(events.contains(str));
        }
    }

    Y_UNIT_TEST(RotatingLog) {
        TString logFile = "eventlog_RotatingLog";

        TTempFile tmp(logFile);
        TTempFile tmpRotated(TStringBuilder{} << logFile << ".1");

        TLoggerConfig conf = CreateLoggerConfig("ERROR", logFile);
        conf.SetBackend(TLoggerConfig_ELogBackend_ROTATING_FILE);
        conf.SetMaxLogSizeBytes(5);
        conf.SetRotatedFilesCount(1);
        TLogger logger(conf);

        logger.SetLevel(ELogPriority::TLOG_INFO);

        LogAllTTestMessages(logger, "some_data");

        logger.CloseLog();

        const THashSet<TString> events = ReadEventLog(logFile);
        UNIT_ASSERT_VALUES_EQUAL(1, events.size());
        UNIT_ASSERT(events.contains("some_dataINFO"));

        const THashSet<TString> rotatedEvents = ReadEventLog(TStringBuilder{} << logFile << ".1");
        UNIT_ASSERT_VALUES_EQUAL(1, events.size());
        UNIT_ASSERT(rotatedEvents.contains("some_dataNOTICE"));
    }

    Y_UNIT_TEST(OnWriteFail) {
        TString logFile = "eventlog";
        TTempFile tmp(logFile);

        bool wasWriteError = false;
        auto onError = [&wasWriteError]() {
            wasWriteError = true;
        };
        TLoggerConfig config = CreateLoggerConfig("DEBUG", logFile);
        config.SetQueueSize(4);
        auto logger = TLogger(config, std::move(onError));

        auto writeToLog = [&logger](int cnt) {
            for (int i = 0; i < cnt; ++i) {
                logger.SpawnFrame()->LogEvent(ELogPriority::TLOG_DEBUG, NLogEvent::TTestMessage("text"));
            }
        };

        UNIT_ASSERT_NO_EXCEPTION(writeToLog(1));
        UNIT_ASSERT(!wasWriteError);

        // spawn queue overflow
        UNIT_ASSERT_NO_EXCEPTION(writeToLog(1000));
        UNIT_ASSERT(wasWriteError);
    }
}

Y_UNIT_TEST_SUITE(EventlogIterator) {

    Y_UNIT_TEST(StartTime_EndTime) {
        TString logFile = "eventlog_Timestamps";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("DEBUG", logFile));
        const size_t eventsCount = 20;

        {
            auto frame = logger.SpawnFrame();

            for (size_t i = 0; i < eventsCount; ++i) {
                Sleep(TDuration::MicroSeconds(TIteratorOptions::MAX_OVERLAP / eventsCount * 1.5));
                frame->LogEvent(NLogEvent::TTestMessage(ToString(i) * i));
            }
        }

        logger.CloseLog();

        auto allTimestamps = ReadTimestamps(logFile);

        for (const auto& start : allTimestamps) {
            for (const auto& end : allTimestamps) {
                UNIT_ASSERT_VALUES_EQUAL(
                    ReadTimestamps(logFile, start.first, end.first)
                    , FilterTimestamps(allTimestamps, start.first, end.first)
                );
            }
        }
    }

    Y_UNIT_TEST(ExcludeEventList) {
        TString logFile = "eventlog_ExcludeEventList";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("DEBUG", logFile));

        size_t otherEventsCnt = 5;
        size_t testEventsCnt = 4;
        size_t startSyncCycleEventsCnt = 3;

         {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < otherEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_EMERG, NLogEvent::TOtherTestMessage());
            }
        }

        {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < testEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_EMERG, NLogEvent::TTestMessage());
            }
        }

        {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < startSyncCycleEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_EMERG, NLogEvent::TStartSyncCycle());
            }
        }

        logger.CloseLog();

        TVector<ui64> events = GetEventsByEvList(logFile, {"TStartSyncCycle", "33001"}, false);

        UNIT_ASSERT_VALUES_EQUAL(events.size(), otherEventsCnt + testEventsCnt);

        for (ui64 id : events) {
            if (id == NLogEvent::TTestMessage::ID) {
                --testEventsCnt;
            }

            if (id == NLogEvent::TOtherTestMessage::ID) {
                --otherEventsCnt;
            }
        }

        UNIT_ASSERT_VALUES_EQUAL(otherEventsCnt, 0);
        UNIT_ASSERT_VALUES_EQUAL(testEventsCnt, 0);
    }

    Y_UNIT_TEST(IncludeEventList) {
        TString logFile = "eventlog_IncludeEventList";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("DEBUG", logFile));

        size_t otherEventsCnt = 5;
        size_t testEventsCnt = 4;
        size_t startSyncCycleEventsCnt = 3;

         {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < otherEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_EMERG, NLogEvent::TOtherTestMessage());
            }
        }

        {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < testEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_EMERG, NLogEvent::TTestMessage());
            }
        }

        {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < startSyncCycleEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_EMERG, NLogEvent::TStartSyncCycle());
            }
        }

        logger.CloseLog();

        TVector<ui64> events = GetEventsByEvList(logFile, {"TStartSyncCycle", "33001"}, true);

        UNIT_ASSERT_VALUES_EQUAL(events.size(), startSyncCycleEventsCnt);

        for (ui64 id : events) {
            if (id == NLogEvent::TStartSyncCycle::ID) {
                --startSyncCycleEventsCnt;
            }
        }

        UNIT_ASSERT_VALUES_EQUAL(startSyncCycleEventsCnt, 0);
    }

    Y_UNIT_TEST(FrameList) {
        TString logFile = "eventlog_FrameList";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("DEBUG", logFile));

        size_t otherTestMessageCnt = 5;
        size_t testMessageCnt = 4;
        size_t frameCnt = 5;

        for (size_t i = 0; i < frameCnt; ++i) {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < otherTestMessageCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_EMERG, NLogEvent::TOtherTestMessage());
            }
            for (size_t i = 0; i < testMessageCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_EMERG, NLogEvent::TTestMessage());
            }
        }

        logger.CloseLog();
        TVector<ui64> frameList = {1, 2, 5};
        TVector<ui64> events = GetEventsByFrameList(logFile, frameList);

        size_t totalNeedFrameCnt = frameList.size() * (otherTestMessageCnt + testMessageCnt);
        UNIT_ASSERT_VALUES_EQUAL(events.size(), totalNeedFrameCnt);

        for (ui64 frameId : events) {
            if (IsIn(frameList, frameId)) {
                --totalNeedFrameCnt;
            }
        }

        UNIT_ASSERT_VALUES_EQUAL(totalNeedFrameCnt, 0);
    }

    Y_UNIT_TEST(EventAllowedPredicate) {
        TString logFile = "eventlog_Predicate";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("DEBUG", logFile));

        size_t testEventsCnt = 4;
        size_t startSyncCycleEventsCnt = 3;

        {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < testEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_EMERG, NLogEvent::TTestMessage());
            }
        }

        {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < startSyncCycleEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_EMERG, NLogEvent::TStartSyncCycle());
            }
        }

        logger.CloseLog();

        TVector<ui32> events = GetAllowedEvents(logFile, [](const TEvent* event) {
            return event->Id() == NLogEvent::TStartSyncCycle::ID;
        });

        UNIT_ASSERT_VALUES_EQUAL(events, TVector<ui32>(startSyncCycleEventsCnt, NLogEvent::TStartSyncCycle::ID));
    }

    Y_UNIT_TEST(EventAllowedByMaxLogPriority) {
        TString logFile = "eventlog_EventAllowedByMaxLogPriority";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("DEBUG", logFile));

        size_t otherEventsCnt = 5;
        size_t testEventsCnt = 4;

        {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < otherEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_DEBUG, NLogEvent::TOtherTestMessage());
            }
        }

        {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < testEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_CRIT, NLogEvent::TTestMessage());
            }
        }

        {
            auto frame = logger.SpawnFrame();
            for (size_t i = 0; i < otherEventsCnt; ++i) {
                frame->LogEvent(ELogPriority::TLOG_DEBUG, NLogEvent::TOtherTestMessage());
            }
        }

        logger.CloseLog();

        TVector<ui32> events = GetEventsByMaxLogPriority(logFile, ELogPriority::TLOG_CRIT);

        UNIT_ASSERT_VALUES_EQUAL(events, TVector<ui32>(testEventsCnt, NLogEvent::TTestMessage::ID));
    }

    Y_UNIT_TEST(CrashedEventLogData) {
        TString logFile = "eventlog_CrashedEventLogData";

        THashSet<TString> events;

        TVector<size_t> handlePosByBytePos;

        TString serializedEvents;

        {
            TFile file(logFile, EOpenModeFlag::RdWr | EOpenModeFlag::CreateAlways);

            TLogger logger(CreateLoggerConfig("RESOURCES", logFile));

            auto frame = logger.SpawnFrame();

            for (size_t i = 0; i < PRIORITIES.size(); ++i) {
                size_t beginPos = file.GetLength();
                frame->LogEvent(FromString<ELogPriority>(PRIORITIES[i]), NLogEvent::TTestMessage(PRIORITIES[i]));
                logger.ReopenLog();
                size_t endPos = file.GetLength();
                handlePosByBytePos.resize(handlePosByBytePos.size() + endPos - beginPos);

                for (size_t j = beginPos; j < endPos; ++j) {
                    handlePosByBytePos[j] = i;
                }
            }
            events = ReadEventLog(logFile);

            logger.CloseLog();

            {
                TFileInput input(logFile);

                serializedEvents = input.ReadAll();
            }
        }

        // one byte is missing
        for (size_t crashedBytePos = 129; crashedBytePos < serializedEvents.size(); ++crashedBytePos) {
            TString serializedCrashedEvents = serializedEvents.substr(0, crashedBytePos) + serializedEvents.substr(crashedBytePos + 1);
            TTempFile tmp(logFile);

            TFileOutput output(logFile);
            output << serializedCrashedEvents << Flush;

            auto someEvents = ReadEventLog(logFile);

            for (const auto& event : events) {
                if (!someEvents.contains(event)) {
                    size_t handlePos = FromString<ELogPriority>(event);
                    UNIT_ASSERT(handlePos == handlePosByBytePos[crashedBytePos]);
                }
            }
        }

        // two bytes are missing
        for (size_t firstCrashedBytePos = 0; firstCrashedBytePos < PRIORITIES.size(); ++firstCrashedBytePos) {
            for (size_t secondCrashedBytePos = firstCrashedBytePos + 1; secondCrashedBytePos < PRIORITIES.size(); ++secondCrashedBytePos) {
                TString serializedCrashedEvents;
                for (size_t i = 0; i < serializedEvents.size(); ++i) {
                    if (i != firstCrashedBytePos && i != secondCrashedBytePos) {
                        serializedCrashedEvents += serializedEvents[i];
                    }
                }

                TTempFile tmp(logFile);

                TFileOutput output(logFile);
                output << serializedCrashedEvents << Flush;

                auto someEvents = ReadEventLog(logFile);

                for (const auto& event : events) {
                    if (!someEvents.contains(event)) {
                        size_t handlePos = FromString<ELogPriority>(event);
                        UNIT_ASSERT(
                            handlePos == handlePosByBytePos[firstCrashedBytePos]
                            || handlePos == handlePosByBytePos[secondCrashedBytePos]
                        );
                    }
                }
            }
        }
    }

    Y_UNIT_TEST(MacroSourceLocation) {
        TString logFile = "eventlog_SourceLocation";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("DEBUG", logFile));

        auto logFrame = logger.SpawnFrame();

        TSourceLocation firstEventLocation = __LOCATION__;
        INFRA_LOG_INFO(NLogEvent::TTestMessage("1"));
        ++firstEventLocation.Line;

        TSourceLocation secondEventLocation = __LOCATION__;
        INFRA_LOG_INFO(NLogEvent::TTestMessage("2"));
        ++secondEventLocation.Line;

        logger.CloseLog();

        auto readSourceLocations = ReadSourceLocations(logFile);
        Sort(readSourceLocations.begin(), readSourceLocations.end());

        TVector<std::pair<TString, TString>> expected = {
            {"1", TStringBuilder() << firstEventLocation},
            {"2", TStringBuilder() << secondEventLocation},
        };

        UNIT_ASSERT_VALUES_EQUAL(readSourceLocations, expected);
    }
}

Y_UNIT_TEST_SUITE(EventlogCachedIterator) {
    Y_UNIT_TEST(PrevEvent) {
        TString logFile = "eventlog_PrevEvent";

        TTempFile tmp(logFile);

        TLogger logger(CreateLoggerConfig("DEBUG", logFile));
        const size_t eventsCount = 50;

        {
            auto frame = logger.SpawnFrame();

            for (size_t i = 0; i < eventsCount; ++i) {
                frame->LogEvent(NLogEvent::TTestMessage(ToString(i)));
            }
        }

        logger.CloseLog();

        TVector<std::pair<ui64, TString>> result;
        NInfra::TIteratorOptions options;
        options.SetFileName(logFile);

        THolder<NInfra::TCachedEventIterator> iter = MakeHolder<TCachedEventIterator>(options);
        const NInfra::TEvent* event;
        TStack<TString> messages;

        for (;;) {
            try {
                event = iter->Next();

                if (event) {
                    messages.push(event->GetData());
                } else {
                    break;
                }
            } catch (...) {
                Cout.Flush();
                Cerr << "Error occured: " << CurrentExceptionMessage() << Endl;
            }
        }

        messages.pop();

        for (;;) {
            try {
                event = iter->Prev();

                if (event) {
                    UNIT_ASSERT(messages);
                    UNIT_ASSERT_EQUAL(messages.top(), event->GetData());
                    messages.pop();
                } else {
                    break;
                }
            } catch (...) {
                Cout.Flush();
                Cerr << "Error occured: " << CurrentExceptionMessage() << Endl;
            }
        }
        UNIT_ASSERT(!messages);
    }
}

Y_UNIT_TEST_SUITE(LogPrinter) {
    Y_UNIT_TEST(PrintLastNEvents) {
        const TString logFileName = "test-eventlog";
        const TString outFileName = "test-output";
        const TString resultFilePrefix = "test-result";

        TIteratorOptions options;
        options.SetFileName(logFileName);

        TString fileContents;
        TString correctFileContents;
        for (int i = 0; i < 5; ++i) {
            {
                TFileInput input(resultFilePrefix + ToString((1u << i) - 1));
                correctFileContents = input.ReadAll();
            }
            {
                TFileOutput output(outFileName);
                NInfra::NPrivate::PrintLastNEvents(options, (1u << i) - 1, output, false, false, true, false);
            }
            {
                TFileInput input(outFileName);
                fileContents = input.ReadAll();
            }

            UNIT_ASSERT_STRINGS_EQUAL(fileContents, correctFileContents);
        }
    }

    Y_UNIT_TEST(PrintHumanReadableAndTimestampDiff) {
        const TString logFileName = "test-eventlog";
        const TString outFileName = "test-output";
        const TString resultFile = "human-readable-result";

        TIteratorOptions options;
        options.SetFileName(logFileName);

        TVector<TString> fileContents;
        TVector<TString> correctFileContents;
        {
            TFileInput input(resultFile);
            correctFileContents = SplitString(input.ReadAll(), "\n");
        }
        {
            TFileOutput output(outFileName);
            NInfra::NPrivate::PrintLastNEvents(options, 100, output, true, true, true, false);
        }
        {
            TFileInput input(outFileName);
            fileContents = SplitString(input.ReadAll(), "\n");
        }

        UNIT_ASSERT_EQUAL(fileContents.size(), correctFileContents.size());
        for (size_t i = 0; i < fileContents.size(); ++i) {
            TVector<TString> fileParts = SplitString(fileContents[i], "\t");
            TVector<TString> correctFileParts = SplitString(correctFileContents[i], "\t");

            UNIT_ASSERT_EQUAL(fileParts.size(), correctFileParts.size());

            // locale-aware comparison
            UNIT_ASSERT_EQUAL_C(
                TInstant::ParseIso8601(fileParts[0])
                , TInstant::ParseIso8601(correctFileParts[0])
                , fileParts[0] + "!= " + correctFileParts[0]
            );

            for (size_t j = 1; j < fileParts.size(); ++j) {
                UNIT_ASSERT_STRINGS_EQUAL(fileParts[j], correctFileParts[j]);
            }
        }
    }
}
