#pragma once

#include "logger.h"

#include <infra/libs/lifetime_histogram/lifetime_histogram.h>
#include <infra/libs/sensors/sensor.h>

#include <google/protobuf/descriptor.h>

#include <util/datetime/cputimer.h>
#include <util/datetime/base.h>
#include <util/string/cast.h>

namespace NInfra {

namespace {

class TFrameBiographer {
public:
    template <typename TStartEvent, typename TFinishEvent>
    TFrameBiographer(
        TLogFramePtr logFrame,
        NLifetimeHistogram::TLifetimeHistogram lifetimeHistogram,
        ELogPriority logPriority,
        TStartEvent&& startEvent,
        TFinishEvent&& finishEvent
    )
        : LogFrame_(logFrame)
        , LifetimeHistogram_(lifetimeHistogram)
        , LogPriority_(logPriority)
        , FinishEventId_(finishEvent.ID)
        , FinishEvent_(MakeHolder<TFinishEvent>(std::move(finishEvent)))
    {
        LogFrame_->LogEvent(LogPriority_, std::move(startEvent));
    }

    TFrameBiographer(TFrameBiographer&&) noexcept = default;

    ~TFrameBiographer() {
        LifetimeHistogram_.Stop();
        const ui64 durationTime = LifetimeHistogram_.GetDuration().GetRef();

        if (FinishEvent_) {
            const NProtoBuf::Reflection* reflection = FinishEvent_->GetReflection();
            for (const TString& timeSpentFieldName : {"TimeSpentMcs", "TimeSpent"}) {
                if (auto* field = FinishEvent_->GetDescriptor()->FindFieldByName(timeSpentFieldName);
                    field && field->type() == google::protobuf::FieldDescriptor::TYPE_UINT64)
                {
                    reflection->SetUInt64(FinishEvent_.Get(), field, durationTime);
                    break; // Expected that only one field is declared
                }
            }
            LogFrame_->LogEvent(LogPriority_, *FinishEvent_, FinishEventId_);
        }
    }

private:
    TLogFramePtr LogFrame_;
    NLifetimeHistogram::TLifetimeHistogram LifetimeHistogram_;
    ELogPriority LogPriority_;
    ui32 FinishEventId_;
    THolder<NProtoBuf::Message> FinishEvent_;
};

} // anonymous namespace

struct TFrameBioChapter {
    TLogFramePtr Frame;
    TFrameBiographer Bio;
};

template <
    ELogPriority LOG_PRIORITY = ELogPriority::TLOG_INFO,
    typename TStartEvent,
    typename TFinishEvent
>
TFrameBioChapter CreateBioChapterForFrame(
    NInfra::TLogFramePtr logFrame,
    TStartEvent&& startEvent,
    TFinishEvent&& finishEvent,
    NLifetimeHistogram::TLifetimeHistogram frameTimeHistogram = NLifetimeHistogram::TLifetimeHistogram()
) {
    return TFrameBioChapter{
        .Frame = logFrame,
        .Bio = TFrameBiographer{
            logFrame,
            frameTimeHistogram,
            LOG_PRIORITY,
            std::move(startEvent),
            std::move(finishEvent)
        }
    };
}


template <
    ELogPriority LOG_PRIORITY = ELogPriority::TLOG_INFO,
    typename TStartEvent,
    typename TFinishEvent
>
TFrameBioChapter CreateBiographedLoggerFrame(
    NInfra::TLogger& logger,
    TStartEvent&& startEvent,
    TFinishEvent&& finishEvent,
    NLifetimeHistogram::TLifetimeHistogram frameTimeHistogram = NLifetimeHistogram::TLifetimeHistogram()
) {
    return CreateBioChapterForFrame(
        logger.SpawnFrame(),
        std::move(startEvent),
        std::move(finishEvent),
        frameTimeHistogram
    );
}

} // namespace NInfra
