#include "logger.h"

#include <infra/libs/logger/yt_logger/protos/events.ev.pb.h>

#include <library/cpp/logger/priority.h>

#include <util/stream/format.h>
#include <util/stream/printf.h>
#include <util/stream/str.h>
#include <util/string/cast.h>

#include <thread>

namespace NInfra {

////////////////////////////////////////////////////////////////////////////////

namespace {

ELogPriority ConvertLogLevel(NYT::ILogger::ELevel level) {
    switch (level) {
        case NYT::ILogger::FATAL:
            return ELogPriority::TLOG_EMERG;
        case NYT::ILogger::ERROR:
            return ELogPriority::TLOG_ERR;
        case NYT::ILogger::INFO:
            return ELogPriority::TLOG_INFO;
        case NYT::ILogger::DEBUG:
            return ELogPriority::TLOG_DEBUG;
    }
}

} // anonymous namespace

////////////////////////////////////////////////////////////////////////////////

TYtLoggerPtr InitYtLogger(TLoggerConfig config, std::function<void()> onWriteFail) {
    TYtLoggerPtr logger = MakeIntrusive<TYtLogger>(std::move(config), std::move(onWriteFail));
    NYT::SetLogger(logger);
    return logger;
}

////////////////////////////////////////////////////////////////////////////////

TYtLogger::TYtLogger(TLoggerConfig config, std::function<void()> onWriteFail)
    : TLogger(std::move(config), std::move(onWriteFail))
    , LogFrame_(SpawnFrame())
{
}

void TYtLogger::Log(
    NYT::ILogger::ELevel level,
    const TSourceLocation& sourceLocation,
    const char* format,
    va_list args
) {
    ELogPriority logPriority = ConvertLogLevel(level);
    if (!LogFrame_->AcceptLevel(logPriority)) {
        return;
    }

    NLogEvent::TYtEvent event;
    event.SetSource(TString::Join(sourceLocation.File, ":", ToString(sourceLocation.Line)));
    size_t threadId = std::hash<std::thread::id>{}(std::this_thread::get_id());
    event.SetThreadId(ToString(Hex(threadId, HF_ADDX)));
    TStringOutput stream(*event.MutableMessage());
    Printf(stream, format, args);
    LogFrame_->LogEvent(logPriority, std::move(event));
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NInfra
