#include "memory_lock.h"
#include <infra/libs/memory_lock/protos/events.ev.pb.h>
#include <util/system/mlock.h>
#include <util/generic/yexception.h>

namespace NInfra::NMemoryLock {

constexpr TStringBuf SENSOR = "memory_lock";
constexpr TStringBuf TYPE_LABEL = "type";
constexpr TStringBuf TYPE_LABEL_FAIL = "Fail";

void LockSelfMemory(
    const TMemoryLockConfig& lockConfig
    , NInfra::TLogFramePtr logFrame
    , const TMaybe<NInfra::TSensorGroup>& sensorGroup
    , bool throwOnFail
    , const std::function<void(const yexception&)>& onFail
    , const std::function<void()>& onSuccess
) {
    if (sensorGroup) {
        auto memoryLockDescriptor = ELockMemory_descriptor();
        for (int i = 0; i < memoryLockDescriptor->value_count(); ++i) {
            TString labelName = memoryLockDescriptor->value(i)->name();
            if (!labelName.empty()) {
                NInfra::TIntGaugeSensor(*sensorGroup, SENSOR, {{TYPE_LABEL, labelName}}).Set(0);
            }
        }
        NInfra::TIntGaugeSensor(*sensorGroup, SENSOR, {{TYPE_LABEL, TYPE_LABEL_FAIL}}).Set(0);
    }
    try {
        TServiceMemoryLockSuccess ev;
        switch (lockConfig.GetLockType()) {
        case None:
            break;
        case Startup:
            LockAllMemory(LockCurrentMemory);
            ev.AddMlockFlags("LockCurrentMemory");
            break;
        case All:
            LockAllMemory(LockCurrentMemory | LockFutureMemory);
            ev.AddMlockFlags("LockCurrentMemory");
            ev.AddMlockFlags("LockFutureMemory");
            break;
        }
        logFrame->LogEvent(ELogPriority::TLOG_INFO, ev);
        if (sensorGroup) {
            NInfra::TIntGaugeSensor(*sensorGroup, SENSOR, {{TYPE_LABEL, ELockMemory_Name(lockConfig.GetLockType())}}).Set(1);
        }
        if (onSuccess) {
            onSuccess();
        }
    } catch (const yexception& e) {
        TServiceMemoryLockError ev(CurrentExceptionMessage());
        logFrame->LogEvent(ELogPriority::TLOG_ERR, ev);
        if (sensorGroup) {
            NInfra::TIntGaugeSensor(*sensorGroup, SENSOR, {{TYPE_LABEL, TYPE_LABEL_FAIL}}).Set(1);
        }
        if (onFail) {
            onFail(e);
        }
        if (throwOnFail) {
            std::rethrow_exception(std::exception_ptr(std::current_exception()));
        }
    }
}

}
