
from infra.libs.monitoring_lib.docs_builder.basic.description import Description
from infra.libs.monitoring_lib.juggler.meta import Urls as MetaUrls

import copy

DEFAULT_DESCRIPTION_HEADER = "unknown alert"
DEFAULT_MEANING = u"Тут еще нет описания :("
DEFAULT_URLS_ORDER = ["solomon", "yasm_alert", "juggler", "wiki", "docs", "nanny", "qloud"]

TAB2 = "  "


def add_tab(text: str) -> str:
    if text:
        return "{}{}".format(TAB2, f"\n{TAB2}".join(text.splitlines()))
    else:
        return text


class AlertDescription(Description):
    class Case:
        def __init__(self, cause: str, solution: str = ":(", example: str = None):
            self.cause = cause
            self.solution = solution
            self.example = example

        @property
        def markdown(self) -> str:
            md_items = [
                "---",
                f"{self.cause}",
                f"**Решение**. {self.solution}",
            ]
            if self.example:
                md_items.append(f"**Пример**. {self.example}")
            return "\n\n".join(md_items) + "\n"

    class Cases:
        def __init__(self, cases: list['Case'], under_cut: bool = True):
            self.cases = cases
            self.under_cut = under_cut

        @property
        def markdown(self) -> str:
            md_items = []
            if self.under_cut:
                md_items.append(u"\n{% cut \"Возможные причины и решения\" %}\n")
            for case in self.cases:
                md_items.append(f"{add_tab(case.markdown)}\n")
            if self.under_cut:
                md_items.append("\n{% endcut %}\n")
            return "\n".join(md_items)

    class Urls:
        def __init__(self, urls: MetaUrls, urls_order: list = DEFAULT_URLS_ORDER, under_cut: bool = False):
            self.urls_order = urls_order
            self.under_cut = under_cut
            self.raw_urls = copy.deepcopy(urls)

        def add_urls(self, urls: MetaUrls):
            self.raw_urls.add(urls)

        def markdown_list(self, urls_type: str, urls) -> list:
            md_items = []
            md_items.append(f"\n- {urls_type.capitalize()}\n")
            for url in urls:
                md_items.append(f"  * [{url.title}]({url.url})")
            return md_items

        @property
        def markdown(self) -> str:
            urls_with_type = {}
            other_urls = []
            for url in self.raw_urls.ordered_urls:
                if "skip_docs" in url.meta:
                    continue
                if "type" in url.meta:
                    urls_with_type.setdefault(url.meta["type"], []).append(url)
                else:
                    other_urls.append(url)

            md_items = []
            if self.under_cut:
                md_items.append("\n{% cut \"Ссылки\" %}\n")
            md_items.append("\n{% list tabs %}\n")
            for urls_type in self.urls_order:
                if urls_type in urls_with_type:
                    md_items += self.markdown_list(urls_type, urls_with_type[urls_type])
            for urls_type, urls in urls_with_type.items():
                if urls_type not in self.urls_order:
                    md_items += self.markdown_list(urls_type, urls)
            if other_urls:
                md_items += self.markdown_list("other", other_urls)
            md_items.append("\n{% endlist %}\n")
            if self.under_cut:
                md_items.append("\n{% endcut %}\n")
            return "\n".join(md_items)

    class Knowledgeable:
        def __init__(self, authors: list[str] = None, knowledgeable: list[str] = None, under_cut: bool = True):
            self.authors = authors
            self.knowledgeable = knowledgeable
            self.under_cut = under_cut

        @property
        def markdown(self) -> str:
            md_items = []
            if self.under_cut:
                md_items.append(u"\n{% cut \"К кому можно обратиться\" %}\n")
            if self.authors:
                md_items.append(u"**Aвторы**: {}".format(", ".join(f"[{author}@](https://staff.yandex-team.ru/{author})" for author in self.authors)))
            if self.knowledgeable:
                md_items.append(u"**Информированные**: {}".format(
                    ", ".join(f"[{knowledgeable}@](https://staff.yandex-team.ru/{knowledgeable})" for knowledgeable in self.knowledgeable)
                ))
            if self.under_cut:
                md_items.append("\n{% endcut %}\n")
            return "\n".join(md_items)

    def __init__(
        self,
        alert_name: str = DEFAULT_DESCRIPTION_HEADER,
        meaning: str = DEFAULT_MEANING,
        cases: Cases = None,
        urls: Urls = None,
        knowledgeable: Knowledgeable = None,
    ):
        super().__init__(alert_name)
        self.meaning = meaning
        self.cases = cases
        self.urls = urls
        self.knowledgeable = knowledgeable

    def add_urls(self, urls: MetaUrls):
        if not self.urls:
            self.urls = self.Urls(urls)
        else:
            self.urls.add_urls(urls)

    @property
    def markdown(self) -> str:
        md_items = [self.meaning]
        if self.cases:
            md_items.append(self.cases.markdown)
        if self.knowledgeable:
            md_items.append(self.knowledgeable.markdown)
        if self.urls:
            md_items.append(self.urls.markdown)
        return "\n".join(md_items)


class JuggleCheckDescription(AlertDescription):
    def fill(self, juggler_check):
        if not self.header or self.header == DEFAULT_DESCRIPTION_HEADER:
            self.header = juggler_check.service

        if juggler_check.optional_args.meta.urls:
            self.add_urls(juggler_check.optional_args.meta.urls)


class SolomonAlertDescription(AlertDescription):
    pass
