from abc import ABCMeta, abstractmethod

import os

from library.python import find_root
from pathlib import Path


class DocsBuilder(metaclass=ABCMeta):
    def __init__(self, path: Path):
        self.path = Path(find_root.detect_root(os.getcwd())) / path
        if not self.path.exists():
            self.path.mkdir(parents=True)

    @abstractmethod
    def build(self):
        pass


class BasicDocsBuilder(DocsBuilder):
    def __init__(self, path_to_docs: str, pages: dict, path_in_docs: str, docs_name: str = None):
        assert(path_in_docs)
        super().__init__(Path(path_to_docs) / path_in_docs)

        self.pages = {}
        self.pages_for_item_type = {}

        for page, item_types in pages.items():
            self.pages[page.name] = page
            for item_type in item_types:
                self.pages_for_item_type.setdefault(item_type, []).append(page.name)

        self.path_in_docs = path_in_docs
        self.docs_name = docs_name

    def get_full_link(self, link: str) -> str:
        if self.docs_name:
            return f"https://docs.yandex-team.ru/{self.docs_name}/{self.path_in_docs}/{link}"
        return ""

    def put_item_in_page(self, item, page_name, once=True) -> dict[str]:
        return {page_name: self.get_full_link(self.pages[page_name].put_item(item))}

    def put_item(self, item, once=True) -> dict[str]:
        links = {}
        found = False
        for item_type, pages in self.pages_for_item_type.items():
            if isinstance(item, item_type):
                found = True
                for page_name in pages:
                    links |= self.put_item_in_page(item, page_name, once)

        if not found:
            raise ValueError(f"there is no default page for {type(item)}")

        return links

    def build(self):
        for name, page in self.pages.items():
            with (self.path / f"{name}.md").open('w') as f:
                f.write(page.markdown)


class NullDocsBuilder(DocsBuilder):
    def __init__(self, *args, **kwards):
        pass

    def put_item(self, *args, **kwards) -> dict[str]:
        return []

    def build(self):
        pass
