from abc import ABCMeta, abstractmethod


class Description(metaclass=ABCMeta):
    def __init__(self, header: str):
        self.header = header

    @property
    @abstractmethod
    def markdown(self) -> str:
        pass


class Section(metaclass=ABCMeta):
    def __init__(self, description_type: type):
        self.description_type = description_type

    @abstractmethod
    def put_item(self, item, anchor: str):
        pass

    @property
    @abstractmethod
    def markdown(self) -> str:
        pass


class VanillaSection(Section):
    def __init__(self, description_type: type, header: str = None):
        super().__init__(description_type)
        self.description_type = description_type
        self.header = header
        self.items = []
        self.description_headers = set()

    def put_item(self, item, anchor: str, once=False):
        if type(item) is self.description_type:
            self.items.append((item, anchor))
        else:
            raise ValueError("incorrect description type")

    @property
    def markdown(self) -> str:
        md_items = []
        if self.header and self.items:
            md_items.append(f"## {self.header}")
        for item, anchor in self.items:
            md_items.append(f"\n### **{item.header}** {{#{anchor}}}\n")
            md_items.append(item.markdown)
        return "\n".join(md_items)


class Page(metaclass=ABCMeta):
    def __init__(self, name: str, title: str, sections: list[Section]):
        self.name = name
        self.title = title
        self.sections = {}
        for section in sections:
            self.sections[section.description_type] = section
        self.anchor_counter = 0

    @abstractmethod
    def put_item(self, item, once=True) -> str:
        pass

    @property
    @abstractmethod
    def markdown(self) -> str:
        pass


class VanillaPage(Page):
    def __init__(self, name: str, title: str, sections: list[Section] = []):
        super().__init__(name, title, sections)
        self.anchors = {}

    def put_item(self, item, once=True) -> str:
        if type(item) not in self.sections:
            self.sections[type(item)] = VanillaSection(type(item))

        item_key = (type(item), item.header)

        if not once or item_key not in self.anchors:
            self.anchor_counter += 1
            self.sections[type(item)].put_item(item, self.anchor_counter)
            self.anchors[item_key] = f"{self.name}#{self.anchor_counter}"
        return self.anchors[item_key]

    @property
    def markdown(self) -> str:
        md_items = [f"# {self.title}"]
        for section in self.sections.values():
            md_items.append(section.markdown)
        return "\n".join(md_items)
