import copy
import operator
from dataclasses import dataclass, field


@dataclass
class Url:
    title: str
    url: str
    meta: dict = field(default_factory=lambda: dict())

    def __hash__(self):
        return hash((self.title, self.url) + tuple(sorted(self.meta.items())))

    @property
    def data(self) -> dict:
        data_element = {"title": self.title, "url": self.url}
        if "type" in self.meta:
            data_element["type"] = self.meta["type"]
        return data_element


class Urls:
    def __init__(self, urls: list[Url]):
        self.ordered_urls = copy.deepcopy(urls)
        self.urls_set = set(self.ordered_urls)

    @property
    def data(self) -> list:
        data = []
        for url in self.ordered_urls[::-1]:
            if "skip_juggler" in url.meta:
                continue
            data.append(url.data)
        return data

    def add(self, other_urls: 'Urls'):
        for url in other_urls.ordered_urls:
            if url not in self.urls_set:
                self.ordered_urls.append(url)
                self.urls_set.add(url)

    def __ior__(self, other_urls: 'Urls'):
        self.add(other_urls)
        return self

    def __or__(self, other_args: 'Urls'):
        result = Urls(self.ordered_urls)
        return operator.__ior__(result, other_args)


class Meta:
    def __init__(self, urls: Urls = None, other_field: dict = None):
        self.urls = urls
        if not other_field:
            self.other_field = {}
        else:
            self.other_field = other_field

    @property
    def data(self) -> dict:
        data = {}
        if self.urls:
            data["urls"] = self.urls.data
        if self.other_field:
            data |= self.other_field
        return data

    def add_urls(self, other_urls: Urls):
        if not self.urls:
            self.urls = copy.deepcopy(other_urls)
        else:
            self.urls |= other_urls


class UrlHandler:
    def __init__(self, handler: 'UrlHandler' = None):
        self.next_handler = handler

    def __call__(self, url: Url):
        if self.next_handler:
            self.next_handler(url)


class UrlsWithHandlers(Urls):
    def __init__(self, urls: list[Url], handlers: list[UrlHandler]):
        self.handlers = copy.deepcopy(handlers)
        curls = copy.deepcopy(urls)
        for url in curls:
            for handler in self.handlers:
                handler(url)
        super().__init__(curls)

    def add(self, other_urls: 'Urls'):
        curls = copy.deepcopy(other_urls)
        for url in curls.ordered_urls:
            for handler in self.handlers:
                handler(url)
        super().add(curls)


HOSTNAME_TO_TYPE = {
    f"{service}.yandex-team.ru": service
    for service in ["solomon", "juggler", "wiki", "docs", "nanny", "qloud", "abc", "deploy"]
} | {
    "st.yandex-team.ru": "tracker",
    "yasm.yandex-team.ru": "yasm_alert",
}


class TypeUrlHandler(UrlHandler):
    def __init__(self, handler: 'UrlHandler' = None, hostname_to_prefix: dict = HOSTNAME_TO_TYPE):
        super().__init__(handler)
        self.hostname_to_prefix = hostname_to_prefix

    def __call__(self, url: Url):
        for hostname, urltype in self.hostname_to_prefix.items():
            if url.url.find(hostname) != -1:
                url.meta.setdefault("type", urltype)
                break
        super().__call__(url)
