from infra.libs.monitoring_lib.docs_builder.alert.builder import AlertDocsBuilder
from infra.libs.monitoring_lib.docs_builder.alert.description import JuggleCheckDescription
from infra.libs.monitoring_lib.docs_builder.basic.builder import NullDocsBuilder
from infra.libs.monitoring_lib.juggler.check import JugglerCheck
from infra.libs.monitoring_lib.juggler.meta import Url, Urls
from infra.libs.monitoring_lib.project.url_generators import UrlGenerator
from infra.libs.monitoring_lib.util.api import Api

import copy
from dataclasses import dataclass, field


class Project:
    def __init__(self, name: str, api: Api):
        self.name = name
        self.api = api

    def apply(self):
        pass


class ProjectWithDocs(Project):
    def __init__(self, name: str, api: Api, docs_builder):
        super().__init__(name, api)
        self.docs_builder = docs_builder

    def add_description(self, description, *args, **kwargs) -> list[str]:
        return self.docs_builder.put_item(description, *args, **kwargs)

    def apply(self):
        self.docs_builder.build()
        super().apply()


class AlertProject(ProjectWithDocs):
    @dataclass
    class Options:
        general_urls: Urls = None
        url_generators_by_juggler_check: list[UrlGenerator] = field(default_factory=lambda: list)

    def __init__(self, name: str, api: Api, docs_options: AlertDocsBuilder.Options = None, options: Options = Options()):
        if docs_options:
            super().__init__(name, api, AlertDocsBuilder(docs_options))
        else:
            super().__init__(name, api, NullDocsBuilder())

        self.options = options

        self.juggler_checks = []
        self.solomon_alerts = []

    def set_urls(self, juggler_check):
        if self.options.general_urls:
            juggler_check.optional_args.meta.add_urls(self.options.general_urls)

        for url_generator in self.options.url_generators_by_juggler_check:
            juggler_check.optional_args.meta.add_urls(url_generator.generate(self, juggler_check))

    def update_juggler_check(self, juggler_check, juggler_check_description=None):
        assert(isinstance(juggler_check, JugglerCheck))

        if not juggler_check_description:
            juggler_check_description = JuggleCheckDescription()
        else:
            juggler_check_description = copy.deepcopy(juggler_check_description)

        self.set_urls(juggler_check)

        juggler_check_description.fill(juggler_check)
        links = self.add_description(juggler_check_description)

        if links:
            link = links["alerts"]
            juggler_check.optional_args.meta.add_urls(Urls(
                urls=[Url(title="Описание алерта", url=link, meta={"type": "docs"})]
            ))

        self.juggler_checks.append(juggler_check)

    def apply(self):
        for juggler_check in self.juggler_checks:
            self.api.juggler_upsert_check(juggler_check)
        super().apply()
