import json
import logging
import requests

from dataclasses import dataclass, field, InitVar

from juggler_sdk import JugglerApi as JugglerSdkApi, Check as JugglerSdkCheck

from infra.libs.monitoring_lib.juggler.check import JugglerCheck


class SolomonApi:
    @dataclass
    class Options:
        project_id: str
        token: str

    def __init__(self, options: Options):
        self.options = options
        self.headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'Authorization': 'OAuth {}'.format(options.token),
        }

    def put_item(self, item, name, data):

        url = 'http://solomon.yandex.net/api/v2/projects/{}/{}/{}'.format(self.options.project_id, item, name)

        resp = requests.get(url, headers=self.headers, timeout=10)

        logging.debug(json.dumps(resp.json(), indent=4))
        update_func = requests.put
        old_version = 0
        if resp.status_code == 404:
            update_func = requests.post
            url = 'http://solomon.yandex.net/api/v2/projects/{}/{}'.format(self.options.project_id, item)
        else:
            resp.raise_for_status()
            old_version = resp.json()['version']

        data['version'] = old_version
        logging.debug(json.dumps(data, indent=4))
        resp = update_func(url, headers=self.headers, json=data, timeout=10)
        logging.debug(json.dumps(resp.json(), indent=4))
        resp.raise_for_status()

        logging.info('Success creation of {} with version {}'.format(name, old_version + 1))


class JugglerApi:
    @dataclass
    class Options:
        project_id: str
        token: str
        mark: str = None

        def __post_init__(self):
            if not self.mark:
                self.mark = self.project_id + '__monitoring_tool__'

    def __init__(self, options: Options):
        self.options = options
        self.juggler = JugglerSdkApi(
            'http://juggler-api.search.yandex.net',
            oauth_token=self.options.token,
            project=self.options.project_id,
            mark=self.options.mark,
        )

    def put_item(self, juggler_check: JugglerCheck):
        result = self.juggler.upsert_check(JugglerSdkCheck(**juggler_check.data)).diff.to_dict()
        logging.info('Created juggler check: {} {}'.format(juggler_check.service, juggler_check.host))
        logging.debug('Diff:\n{}'.format(json.dumps(result, indent=4)))


@dataclass
class Api:
    juggler_api_options: InitVar[JugglerApi.Options] = None
    solomon_api_options: InitVar[SolomonApi.Options] = None
    juggler: JugglerApi = field(init=False, default=None)
    solomon: SolomonApi = field(init=False, default=None)

    def __post_init__(self, juggler_api_options, solomon_api_options):
        if juggler_api_options:
            self.juggler = JugglerApi(juggler_api_options)
        if solomon_api_options:
            self.solomon = SolomonApi(solomon_api_options)

    def juggler_upsert_check(self, juggler_check: JugglerCheck):
        self.juggler.put_item(juggler_check)
