#include "multi_log_backend.h"

#include <library/cpp/logger/log.h>
#include <library/cpp/logger/stream.h>

#include <util/generic/vector.h>
#include <util/generic/yexception.h>

namespace NInfra::NMultiLogBackend {

class TMultiLogBackend::TImpl {
public:
    inline TImpl(TVector<THolder<TLogBackend>> slaves)
        : Slaves_(std::move(slaves))
    {
    }

    inline void WriteData(const TLogRecord& rec) {
        for (const auto& backend : Slaves_) {
            if (backend) {
                backend->WriteData(rec);
            }
        }
    }

    inline void ReopenLog() {
        for (const auto& backend: Slaves_) {
            if (backend) {
                backend->ReopenLog();
            }
        }
    }

    inline void ReopenLogNoFlush() {
        for (const auto& backend: Slaves_) {
            if (backend) {
                backend->ReopenLogNoFlush();
            }
        }
    }

    inline size_t QueueSize() const {
        size_t queueSize = 0;
        for (const auto& backend: Slaves_) {
            if (backend) {
                try {
                    queueSize = Max(queueSize, backend->QueueSize());
                } catch (const yexception&) {
                }
            }
        }
        return queueSize;
    }

private:
    const TVector<THolder<TLogBackend>> Slaves_;
};

TMultiLogBackend::TMultiLogBackend(TVector<THolder<TLogBackend>> slaves)
    : Impl_(new TImpl(std::move(slaves)))
{
}

TMultiLogBackend::~TMultiLogBackend() {
}

void TMultiLogBackend::WriteData(const TLogRecord& rec) {
    Impl_->WriteData(rec);
}

void TMultiLogBackend::ReopenLog() {
    Impl_->ReopenLog();
}

void TMultiLogBackend::ReopenLogNoFlush() {
    Impl_->ReopenLogNoFlush();
}

size_t TMultiLogBackend::QueueSize() const {
    return Impl_->QueueSize();
}

}
