#pragma once

#include <util/generic/bt_exception.h>
#include <util/generic/variant.h>
#include <util/string/builder.h>
#include <util/system/type_name.h>

#define OUTCOME_TRYX(...) ({ auto &&res = (__VA_ARGS__); if (res.IsError()) return res.Error(); res.Success(); })
#define OUTCOME_TRYV(...) ({ auto &&res = (__VA_ARGS__); if (res.IsError()) return res.Error(); })

namespace {
struct TEmptySuccess {
};
}

namespace NPrivate {
// specialize these classes for useful output
template<class E /*Error*/>
class SuccessCastErrorMessageGetter {
public:
    static TString GetMessage(const E* const) {
        return "";
    }
};

template<class S /*Success*/>
class ErrorCastErrorMessageGetter {
public:
    static TString GetMessage(const S* const) {
        return "";
    }
};
}

/**
    Wrapper over std::variant which stores either Success or Error
*/
template<class S /* Success */, class E /* Error */>
class TExpected : private std::variant<S, E> {
public:
    TExpected(const S& success)
            : std::variant<S, E>(success) {
    }

    TExpected(const E& error)
            : std::variant<S, E>(error) {
    }

    TExpected(S&& success)
            : std::variant<S, E>(std::forward<S>(success)) {
    }

    TExpected(E&& error)
            : std::variant<S, E>(std::forward<E>(error)) {
    }

    //! throws exception if unable to cast
    S& Success() {
        S* result = std::get_if<S>(this);
        Y_ENSURE_EX(result, TWithBackTrace<yexception>() << "Unable to cast TExpected<" << TypeName<S>() << ", " << TypeName<E>() << "> to Success."
            << NPrivate::SuccessCastErrorMessageGetter<E>::GetMessage(std::get_if<E>(this)));
        return *result;
    }

    //! throws exception if unable to cast
    const S& Success() const {
        const S* result = std::get_if<S>(this);
        Y_ENSURE_EX(result, TWithBackTrace<yexception>() << "Unable to cast TExpected<" << TypeName<S>() << ", " << TypeName<E>() << "> to Success."
            << NPrivate::SuccessCastErrorMessageGetter<E>::GetMessage(std::get_if<E>(this)));
        return *result;
    }

    bool IsSuccess() const noexcept {
        return std::holds_alternative<S>(*this);
    }

    //! throws exception if unable to cast
    E& Error() {
        E* result = std::get_if<E>(this);
        Y_ENSURE_EX(result, TWithBackTrace<yexception>() << "Unable to cast TExpected<" << TypeName<S>() << ", " << TypeName<E>() << "> to Error."
            << NPrivate::ErrorCastErrorMessageGetter<S>::GetMessage(std::get_if<S>(this)));
        return *result;
    }

    //! throws exception if unable to cast
    const E& Error() const {
        const E* result = std::get_if<E>(this);
        Y_ENSURE_EX(result, TWithBackTrace<yexception>() << "Unable to cast TExpected<" << TypeName<S>() << ", " << TypeName<E>() << "> to Error."
            << NPrivate::ErrorCastErrorMessageGetter<S>::GetMessage(std::get_if<S>(this)));
        return *result;
    }

    bool IsError() const noexcept {
        return std::holds_alternative<E>(*this);
    }

    //! true on Success
    template<typename CS = S, typename CE = E>
    constexpr explicit operator std::enable_if_t<!std::is_same_v<CS, bool> && !std::is_same_v<CE, bool>, bool>() const noexcept {
        return IsSuccess();
    }

    //! throws exception if unable to cast
    template<typename CS = S>
    explicit operator std::enable_if_t<!std::is_same_v<CS, bool>, S>() {
        return Success();
    }

    //! throws exception if unable to cast
    template<typename CE = E>
    explicit operator std::enable_if_t<!std::is_same_v<CE, bool>, E>() {
        return Error();
    }

    /**
        Create TExpected with default Success
    */
    static TExpected<S, E> DefaultSuccess() {
        return TExpected<S, E>(S());
    }

    /**
        Create TExpected with default Error
    */
    static TExpected<S, E> DefaultError() {
        return TExpected<S, E>(E());
    }

    bool operator==(const TExpected<S, E>& other) const {
        bool thisOk = IsSuccess();
        bool otherOk = other.IsSuccess();

        if (thisOk ^ otherOk) {
            return false;
        }

        if (thisOk) {
            return Success() == other.Success();
        }

        return Error() == other.Error();
    }
};


template<class E /* Error */>
class TExpected<void, E> : private std::variant<TEmptySuccess, E> {
public:
    TExpected(void)
            : std::variant<TEmptySuccess, E>(TEmptySuccess()) {
    }

    TExpected(const E& error)
            : std::variant<TEmptySuccess, E>(error) {
    }

    TExpected(E&& error)
            : std::variant<TEmptySuccess, E>(error) {
    }

    //! throws exception if unable to cast
    void Success() const {
        const TEmptySuccess* result = std::get_if<TEmptySuccess>(this);
        Y_ENSURE_EX(result, TWithBackTrace<yexception>() << "Unable to cast TExpected<void, " << TypeName<E>() << "> to Success."
            << NPrivate::SuccessCastErrorMessageGetter<E>::GetMessage(std::get_if<E>(this)));
    }

    bool IsSuccess() const noexcept {
        return std::holds_alternative<TEmptySuccess>(*this);
    }

    //! throws exception if unable to cast
    E& Error() {
        E* result = std::get_if<E>(this);
        Y_ENSURE_EX(result, TWithBackTrace<yexception>() << "Unable to cast TExpected<void, " << TypeName<E>() << "> to Error.");
        return *result;
    }

    //! throws exception if unable to cast
    const E& Error() const {
        const E* result = std::get_if<E>(this);
        Y_ENSURE_EX(result, TWithBackTrace<yexception>() << "Unable to cast TExpected<void, " << TypeName<E>() << "> to Error.");
        return *result;
    }

    bool IsError() const noexcept {
        return std::holds_alternative<E>(*this);
    }

    template<typename CE = E>
    constexpr explicit operator std::enable_if_t<!std::is_same_v<CE, bool>, bool>() const noexcept {
        return IsSuccess();
    }

    //! throws exception if unable to cast
    template<typename CE = E>
    operator std::enable_if_t<!std::is_same_v<CE, bool>, E>() {
        return Error();
    }

    /**
        Create TExpected with default Success
    */
    static TExpected<void, E> DefaultSuccess() {
        return TExpected<void, E>();
    }

    /**
        Create TExpected with default Error
    */
    static TExpected<void, E> DefaultError() {
        return TExpected<void, E>(E());
    }

    bool operator==(const TExpected<void, E>& other) const {
        bool thisOk = IsSuccess();
        bool otherOk = other.IsSuccess();

        if (thisOk ^ otherOk) {
            return false;
        }

        if (thisOk) {
            return true;
        }

        return Error() == other.Error();
    }
};
