#include "result.h"

#include <library/cpp/testing/unittest/registar.h>

struct TMySuccess {
    int Expected = 0;

    bool operator==(const TMySuccess& other) const {
        return Expected == other.Expected;
    }
};

struct TMyError {
    int Code = 0;
    TString Message = "";

    bool operator==(const TMyError& other) const {
        return Code == other.Code && Message == other.Message;
    }
};

static_assert(std::is_constructible_v<bool, TExpected<TMySuccess, TMyError>>);
static_assert(std::is_constructible_v<bool, TExpected<void, TMyError>>);

// If bool is a template argument, any cast is forbidden to prevent errors
static_assert(!std::is_constructible_v<bool, TExpected<bool, TMyError>>);
static_assert(!std::is_constructible_v<bool, TExpected<TMySuccess, bool>>);
static_assert(!std::is_constructible_v<bool, TExpected<void, bool>>);

Y_UNIT_TEST_SUITE(Expected) {
    Y_UNIT_TEST(SuccessIsTrue) {
        auto result = TExpected<TMySuccess, TMyError>::DefaultSuccess();
        UNIT_ASSERT_EQUAL((bool)result, true);
        UNIT_ASSERT_EQUAL(result.IsSuccess(), true);
        UNIT_ASSERT_EQUAL(result.IsError(), false);
    }

    Y_UNIT_TEST(SuccessIsFalse) {
        auto result = TExpected<TMySuccess, TMyError>::DefaultError();
        UNIT_ASSERT_EQUAL((bool)result, false);
        UNIT_ASSERT_EQUAL(result.IsSuccess(), false);
        UNIT_ASSERT_EQUAL(result.IsError(), true);
    }

    Y_UNIT_TEST(CastExpectedSuccessToSuccess) {
        auto result = TExpected<TMySuccess, TMyError>::DefaultSuccess();
        TMySuccess success;
        UNIT_ASSERT_NO_EXCEPTION(success = (TMySuccess) result);
    }

    Y_UNIT_TEST(CastExpectedSuccessToError) {
        auto result = TExpected<TMySuccess, TMyError>::DefaultSuccess();
        TMyError error;
        UNIT_ASSERT_EXCEPTION(error = (TMyError) result, yexception);
    }

    Y_UNIT_TEST(CastExpectedErrorToError) {
        auto result = TExpected<TMySuccess, TMyError>::DefaultError();
        TMyError error;
        UNIT_ASSERT_NO_EXCEPTION(error = (TMyError) result);
    }

    Y_UNIT_TEST(CastExpectedErrorToSuccess) {
        auto result = TExpected<TMySuccess, TMyError>::DefaultError();
        TMySuccess success;
        UNIT_ASSERT_EXCEPTION(success = (TMySuccess) result, yexception);
    }

    Y_UNIT_TEST(CastSuccessToExpected) {
        TMySuccess success {1};
        TExpected<TMySuccess, TMyError> result = std::move(success);
        UNIT_ASSERT_EQUAL((bool)result, true);
        UNIT_ASSERT_EQUAL(((TMySuccess)result).Expected, 1);
    }

    Y_UNIT_TEST(CastErrorToExpected) {
        TMyError error {1, "msg"};
        TExpected<TMySuccess, TMyError> result = std::move(error);
        UNIT_ASSERT_EQUAL((bool)result, false);
        UNIT_ASSERT_EQUAL(((TMyError)result).Code, 1);
        UNIT_ASSERT_EQUAL(((TMyError)result).Message, "msg");
    }

    Y_UNIT_TEST(TryXSuccess) {
        auto action = []() -> TExpected<TMySuccess, TMyError> {
            return OUTCOME_TRYX(TExpected<TMySuccess, TMyError>::DefaultSuccess());
        };
        UNIT_ASSERT_EQUAL((bool)action(), true);
    }

    Y_UNIT_TEST(TryXError) {
        auto action = []() -> TExpected<TMySuccess, TMyError> {
            return OUTCOME_TRYX(TExpected<TMySuccess, TMyError>::DefaultError());
        };
        UNIT_ASSERT_EQUAL((bool)action(), false);
    }

    Y_UNIT_TEST(TryVSuccess) {
        auto action = []() -> TExpected<void, TMyError> {
            OUTCOME_TRYV(TExpected<void, TMyError>::DefaultSuccess());
            return TExpected<void, TMyError>::DefaultSuccess();
        };
        UNIT_ASSERT_EQUAL((bool)action(), true);
    }

    Y_UNIT_TEST(TryVError) {
        auto action = []() -> TExpected<void, TMyError> {
            OUTCOME_TRYV(TExpected<void, TMyError>::DefaultError());
            return TExpected<void, TMyError>::DefaultSuccess();
        };
        UNIT_ASSERT_EQUAL((bool)action(), false);
    }

    Y_UNIT_TEST(TryXCastExpectedOnError) {
        auto action = []() -> TExpected<TMySuccess, TMyError> {
            return OUTCOME_TRYX(TExpected<TMySuccess, TMyError>(TMyError {1, "msg"}));
        };
        auto result = action();
        UNIT_ASSERT_EQUAL((bool)result, false);
        UNIT_ASSERT_EQUAL(((TMyError)result).Code, 1);
        UNIT_ASSERT_EQUAL(((TMyError)result).Message, "msg");
    }

    Y_UNIT_TEST(TryVCastExpectedOnError) {
        auto action = []() -> TExpected<TMySuccess, TMyError> {
            OUTCOME_TRYV(TExpected<void, TMyError>(TMyError {1, "msg"}));
            return TExpected<TMySuccess, TMyError>::DefaultSuccess();
        };
        auto result = action();
        UNIT_ASSERT_EQUAL((bool)result, false);
        UNIT_ASSERT_EQUAL(((TMyError)result).Code, 1);
        UNIT_ASSERT_EQUAL(((TMyError)result).Message, "msg");
    }

    Y_UNIT_TEST(ExpectedSBoolOnSuccess) {
        TExpected<TMySuccess, bool> result = TMySuccess{1};

        // operator bool is removed
        UNIT_ASSERT_EQUAL(result.IsSuccess(), true);
        UNIT_ASSERT_EQUAL(result.IsError(), false);
        UNIT_ASSERT_EQUAL(((TMySuccess)result).Expected, 1);
    }

    Y_UNIT_TEST(ExpectedSBoolOnError) {
        for (int resultValue = 0; resultValue < 2; ++resultValue) {
            TExpected<TMySuccess, bool> result = (bool)resultValue;
            // operator bool is removed
            UNIT_ASSERT_EQUAL(result.IsSuccess(), false);
            UNIT_ASSERT_EQUAL(result.IsError(), true);
            UNIT_ASSERT_EQUAL(result.Error(), (bool)resultValue);
        }
    }

    Y_UNIT_TEST(ExpectedBoolEOnSuccess) {
        for (int resultValue = 0; resultValue < 2; ++resultValue) {
            TExpected<bool, TMyError> result = (bool)resultValue;
            // operator bool is removed
            UNIT_ASSERT_EQUAL(result.IsSuccess(), true);
            UNIT_ASSERT_EQUAL(result.IsError(), false);
            UNIT_ASSERT_EQUAL(result.Success(), (bool)resultValue);
        }
    }

    Y_UNIT_TEST(ExpectedBoolEOnError) {
        TExpected<bool, TMyError> result = TMyError{1, "msg"};
        // operator bool is removed
        UNIT_ASSERT_EQUAL(result.IsSuccess(), false);
        UNIT_ASSERT_EQUAL(result.IsError(), true);
        UNIT_ASSERT_EQUAL(((TMyError)result).Code, 1);
        UNIT_ASSERT_EQUAL(((TMyError)result).Message, "msg");
    }

    Y_UNIT_TEST(ExpectedVoidBoolOnSuccess) {
        TExpected<void, bool> result = TExpected<void, bool>::DefaultSuccess();
        // operator bool is removed
        UNIT_ASSERT_EQUAL(result.IsSuccess(), true);
        UNIT_ASSERT_EQUAL(result.IsError(), false);
    }

    Y_UNIT_TEST(ExpectedVoidBoolOnError) {
        for (int resultValue = 0; resultValue < 2; ++resultValue) {
            TExpected<void, bool> result = (bool)resultValue;
            // operator bool is removed
            UNIT_ASSERT_EQUAL(result.IsSuccess(), false);
            UNIT_ASSERT_EQUAL(result.IsError(), true);
            UNIT_ASSERT_EQUAL(result.Error(), (bool)resultValue);
        }
    }

    Y_UNIT_TEST(ExpectedSEOperatorEq) {
        TVector<TExpected<TMySuccess, TMyError>> results = {
            TMySuccess{1},
            TMySuccess{2},
            TMyError{1, ""},
            TMyError{2, "other"},
        };
        for (size_t i = 0; i < results.size(); ++i) {
            for (size_t j = 0; j < results.size(); ++j) {
                if (i == j) {
                    UNIT_ASSERT(results[i] == results[j]);
                } else {
                    UNIT_ASSERT(!(results[i] == results[j]));
                }
            }
        }
    }

    Y_UNIT_TEST(ExpectedSBoolOperatorEq) {
        TVector<TExpected<TMySuccess, bool>> results = {
            TMySuccess{1},
            TMySuccess{2},
            true,
            false,
        };
        for (size_t i = 0; i < results.size(); ++i) {
            for (size_t j = 0; j < results.size(); ++j) {
                if (i == j) {
                    UNIT_ASSERT(results[i] == results[j]);
                } else {
                    UNIT_ASSERT(!(results[i] == results[j]));
                }
            }
        }
    }

    Y_UNIT_TEST(ExpectedBoolEOperatorEq) {
        TVector<TExpected<bool, TMyError>> results = {
            true,
            false,
            TMyError{1, ""},
            TMyError{2, "other"},
        };
        for (size_t i = 0; i < results.size(); ++i) {
            for (size_t j = 0; j < results.size(); ++j) {
                if (i == j) {
                    UNIT_ASSERT(results[i] == results[j]);
                } else {
                    UNIT_ASSERT(!(results[i] == results[j]));
                }
            }
        }
    }

    Y_UNIT_TEST(ExpectedVoidEOperatorEq) {
        TVector<TExpected<void, TMyError>> results = {
            TExpected<void, TMyError>::DefaultSuccess(),
            TMyError{1, ""},
            TMyError{2, "other"},
        };
        for (size_t i = 0; i < results.size(); ++i) {
            for (size_t j = 0; j < results.size(); ++j) {
                if (i == j) {
                    UNIT_ASSERT(results[i] == results[j]);
                } else {
                    UNIT_ASSERT(!(results[i] == results[j]));
                }
            }
        }
    }

    Y_UNIT_TEST(ExpectedVoidBoolOperatorEq) {
        TVector<TExpected<void, bool>> results = {
            TExpected<void, bool>::DefaultSuccess(),
            true,
            false,
        };
        for (size_t i = 0; i < results.size(); ++i) {
            for (size_t j = 0; j < results.size(); ++j) {
                if (i == j) {
                    UNIT_ASSERT(results[i] == results[j]);
                } else {
                    UNIT_ASSERT(!(results[i] == results[j]));
                }
            }
        }
    }
}
