#pragma once

#include "sensor_group.h"
#include "sensor_registry.h"

#include <util/string/builder.h>

namespace NInfra {

class TSensorError: public yexception {
};

class ISensor {
public:
    ISensor(bool throwOnError = false)
        : ThrowOnError_(throwOnError)
    {
    }

    virtual ~ISensor() = default;

    virtual void Set(i64) {
        if (ThrowOnError_) {
            throw TSensorError() << GetSensorErrorMessage("Set");
        }
    }

    virtual void Set(double) {
        if (ThrowOnError_) {
            throw TSensorError() << GetSensorErrorMessage("Set");
        }
    }

    virtual void Inc() {
        if (ThrowOnError_) {
            throw TSensorError() << GetSensorErrorMessage("Inc");
        }
    }

    virtual void Add(i64) {
        if (ThrowOnError_) {
            throw TSensorError() << GetSensorErrorMessage("Add");
        }
    }

    virtual void Add(double) {
        if (ThrowOnError_) {
            throw TSensorError() << GetSensorErrorMessage("Add");
        }
    }

    virtual void Start() {
        if (ThrowOnError_) {
            throw TSensorError() << GetSensorErrorMessage("Start");
        }
    }

    virtual void Update() {
        if (ThrowOnError_) {
            throw TSensorError() << GetSensorErrorMessage("Update");
        }
    }

    virtual void Reset() {
        if (ThrowOnError_) {
            throw TSensorError() << GetSensorErrorMessage("Reset");
        }
    }

private:
    const bool ThrowOnError_;

    TString GetSensorErrorMessage(const TString& method) {
        return TStringBuilder() << "Sensor of type \"" << Type() << "\" does not support method \"" << method << "\"";
    }

    virtual TStringBuf Type() const = 0;
};

class TIntGaugeSensor: public ISensor {
public:
    TIntGaugeSensor(TSensorGroup group, TStringBuf name, const TVector<std::pair<TStringBuf, TStringBuf>>& labels = {})
        : Backend_(SensorRegistry().IntGauge(NPrivate::CreateLabels(group.AddLabels(labels), name)))
    {
    }

    void Add(i64 x) override {
        Backend_->Add(x);
    }

    void Set(i64 x) override {
        Backend_->Set(x);
    }

private:
    NMonitoring::TIntGauge* Backend_;
    static constexpr TStringBuf Type_ = "IntGauge";

    TStringBuf Type() const override {
        return Type_;
    }
};

class TGaugeSensor: public ISensor {
public:
    TGaugeSensor(TSensorGroup group, TStringBuf name, const TVector<std::pair<TStringBuf, TStringBuf>>& labels = {})
        : Backend_(SensorRegistry().Gauge(NPrivate::CreateLabels(group.AddLabels(labels), name)))
    {
    }

    void Add(double x) override {
        Backend_->Add(x);
    }

    void Set(double x) override {
        Backend_->Set(x);
    }

private:
    NMonitoring::TGauge* Backend_;
    static constexpr TStringBuf Type_ = "Gauge";

    TStringBuf Type() const override {
        return Type_;
    }
};

class TRateSensor: public ISensor {
public:
    TRateSensor(TSensorGroup group, TStringBuf name, const TVector<std::pair<TStringBuf, TStringBuf>>& labels = {})
        : Backend_(SensorRegistry().Rate(NPrivate::CreateLabels(group.AddLabels(labels), name)))
    {
    }

    void Inc() override {
        Backend_->Inc();
    }

    void Add(i64 x) override {
        Backend_->Add(x);
    }

    ui64 Get() const {
        return Backend_->Get();
    }

private:
    NMonitoring::TRate* Backend_;
    static constexpr TStringBuf Type_ = "Rate";

    TStringBuf Type() const override {
        return Type_;
    }
};

class THistogramCounterSensor : public ISensor {
public:
    THistogramCounterSensor(TSensorGroup group, TStringBuf name, NMonitoring::IHistogramCollectorPtr&& collector, const TVector<std::pair<TStringBuf, TStringBuf>>& labels = {})
        : Backend_(SensorRegistry().HistogramCounter(NPrivate::CreateLabels(group.AddLabels(labels), name), std::move(collector)))
    {
    }

    void Add(i64 x) override {
        Backend_->Record(x);
    }

    NMonitoring::IHistogramSnapshotPtr TakeSnapshot() const {
        return Backend_->TakeSnapshot();
    }

private:
    NMonitoring::THistogram* Backend_;

    static constexpr TStringBuf Type_ = "HistogramCounter";

    TStringBuf Type() const override {
        return Type_;
    }
};

class THistogramRateSensor : public ISensor, public TAtomicRefCount<THistogramRateSensor> {
public:
    THistogramRateSensor(TSensorGroup group, TStringBuf name, NMonitoring::IHistogramCollectorPtr&& collector, const TVector<std::pair<TStringBuf, TStringBuf>>& labels = {})
        : Backend_(SensorRegistry().HistogramRate(NPrivate::CreateLabels(group.AddLabels(labels), name), std::move(collector)))
    {
    }

    void Add(i64 x) override {
        Backend_->Record(x);
    }

    NMonitoring::IHistogramSnapshotPtr TakeSnapshot() const {
        return Backend_->TakeSnapshot();
    }

private:
    NMonitoring::THistogram* Backend_;

    static constexpr TStringBuf Type_ = "HistogramRate";

    TStringBuf Type() const override {
        return Type_;
    }
};

class TDurationSensor: public ISensor {
public:
    TDurationSensor(const TSensorGroup& group, TStringBuf name)
        : Start_(TInstant::Now())
        , Backend_(SensorRegistry().Gauge(NPrivate::CreateLabels(group, name)))
    {
    }

    void Start() override {
        Start_ = TInstant::Now();
    }

    void Update() override {
        Backend_->Set((TInstant::Now() - Start_).MicroSeconds() / 1000.);
    }

    void Reset() override {
        Backend_->Set(0);
    }

private:
    TInstant Start_;
    NMonitoring::TGauge* Backend_;
    static constexpr TStringBuf Type_ = "Duration";

    TStringBuf Type() const override {
        return Type_;
    }
};

} // namespace NInfra
