#pragma once

#include "reply.h"
#include "request.h"
#include "str_iface.h"

#include <infra/libs/service_iface/str_iface.h>

#include <util/generic/hash.h>
#include <util/generic/vector.h>
#include <util/string/builder.h>


namespace NInfra {
    /*
* This router contatins functions, which use for http and grpc requests
* Handle function calls functions from Callbacks, which can throw an exception
*/

    namespace NServiceAttributes {
        const TString RemoteAddressAttributeName();
    }

    struct TRouterResponse {
        TString Content;
        TAttributes Attributes;
    };

    class IRequestRouter {
    public:
        virtual bool Has(TStringBuf path) const = 0;

        virtual TRouterResponse Handle(TStringBuf path, const TString& postData, const TVector<std::pair<TString, TString>>& headers) const = 0;

        virtual ~IRequestRouter() = default;
    };

    using TRequestRouterPtr = TSimpleSharedPtr<IRequestRouter>;

    template <typename TService>
    class TRequestRouter: public IRequestRouter {
    public:
        using TCallback = std::function<TRouterResponse(TStringBuf, const TString&, const TAttributes& attrs)>;

        TRequestRouter(TService& service)
            : Service_(service)
        {
        }

        template <typename TRequest, typename TReply>
        TRequestRouter& Add(const TString& path, const TString& help,
                            void (TService::*Call)(TRequestPtr<typename TRequest::TMessage>, TReplyPtr<typename TReply::TMessage>)) {
            auto callback = [this, Call](TStringBuf parsed, const TString& postData, const TAttributes& attrs) {
                TString response;
                TAttributes replyAttributes;
                (Service_.*Call)(RequestPtr<TRequest>(parsed, postData, attrs), ReplyPtr<TReply>(response, replyAttributes));
                return TRouterResponse{std::move(response), std::move(replyAttributes)};
            };

            Usage_[path] = help;
            Callbacks_[path] = callback;

            return *this;
        }

        bool Has(TStringBuf path) const override {
            return path == "/help" || Callbacks_.contains(path);
        }

        TRouterResponse Handle(TStringBuf path, const TString& postData, const TVector<std::pair<TString, TString>>& headers) const override {
            if (path == "/help") {
                return {GenerateHelp(), TAttributes()};
            }

            TAttributes attributes;
            for (const auto& header : headers) {
                attributes.insert(header);
            }

            return Callbacks_.at(path)(path, postData, attributes);
        }

    private:
        TString GenerateHelp() const {
            TStringBuilder builder;
            builder << "Service API usage:" << Endl;

            for (const auto& pt : Usage_) {
                builder << "    " << pt.first << " - " << pt.second << Endl;
            }

            return builder;
        }

        TService& Service_;
        THashMap<TString, TString> Usage_;
        THashMap<TString, TCallback> Callbacks_;
    };

}
