#pragma once

#include "request.h"
#include "reply.h"

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_value.h>
#include <library/cpp/json/json_writer.h>

namespace NInfra {
    template <typename TMessage>
    class TEmptyRequest: public IRequest<TMessage> {
    public:
        TEmptyRequest(TStringBuf path, const TString&, const TAttributes& attrs)
            : IRequest<TMessage>(path, attrs)
        {
        }

        const TMessage& Get() const override {
            return Default<TMessage>();
        }
    };

    template <typename TMessage>
    class TEmptyReply: public IReply<TMessage> {
    public:
        TEmptyReply(TString&, TAttributes& attrs)
            : IReply<TMessage>(attrs)
        {
        }

        void Set(const TMessage&) override {
        }
    };

    template <typename TMessage,
              bool PRETTY = false,
              NProtobufJson::TProto2JsonConfig::EnumValueMode ENUM_VALUE_MODE = NProtobufJson::TProto2JsonConfig::EnumNumber,
              bool MAP_AS_OBJECT = false>
    class TProto2JsonReply: public IReply<TMessage> {
    public:
        TProto2JsonReply(TString& response, TAttributes& attrs)
            : IReply<TMessage>(attrs)
            , Response_(response)
        {
        }

        void Set(const TMessage& message) override {
            NProtobufJson::TProto2JsonConfig cfg;
            cfg.SetFormatOutput(PRETTY)
               .SetEnumMode(ENUM_VALUE_MODE)
               .SetMapAsObject(MAP_AS_OBJECT);

            Response_ = NProtobufJson::Proto2Json(message, cfg);
        }

    private:
        TString& Response_;
    };

    template <typename TMessage, bool allowUnknownFields = false>
    class TJson2ProtoRequest: public IRequest<TMessage> {
    public:
        TJson2ProtoRequest(TStringBuf path, const TString& postData, const TAttributes& attrs)
            : IRequest<TMessage>(path, attrs)
            , Message_(NProtobufJson::Json2Proto<TMessage>(postData))
        {
            NProtobufJson::TJson2ProtoConfig json2ProtoConfig = NProtobufJson::TJson2ProtoConfig();
            json2ProtoConfig.AllowUnknownFields = allowUnknownFields;
            Message_ = NProtobufJson::Json2Proto<TMessage>(postData, json2ProtoConfig);
        }

        const TMessage& Get() const override {
            return Message_;
        }

    private:
        TMessage Message_;
    };

    template <typename TMessage, bool allowUnknownFields = false>
    class TJsonOrEmpty2ProtoRequest: public IRequest<TMessage> {
    public:
        TJsonOrEmpty2ProtoRequest(TStringBuf path, const TString& postData, const TAttributes& attrs)
            : IRequest<TMessage>(path, attrs)
        {
            if (!postData.empty()) {
                NProtobufJson::TJson2ProtoConfig json2ProtoConfig = NProtobufJson::TJson2ProtoConfig();
                json2ProtoConfig.AllowUnknownFields = allowUnknownFields;
                Message_ = NProtobufJson::Json2Proto<TMessage>(postData, json2ProtoConfig);
            } else {
                Message_ = TMessage();
            }
        }

        const TMessage& Get() const override {
            return Message_;
        }

    private:
        TMessage Message_;
    };

    template <typename TMessage>
    class TRawDataReply: public IReply<TMessage> {
    public:
        TRawDataReply(TString& response, TAttributes& attrs)
            : IReply<TMessage>(attrs)
            , Response_(response)
        {
        }

        void Set(const TMessage& message) override {
            if constexpr (requires(const TMessage& m) { m.GetData(); }) {
                Response_ = message.GetData();
            } else {
                Response_ = message.data();
            }
        }

    private:
        TString& Response_;
    };

    template <typename TMessage>
    class TProtoRequest: public IRequest<TMessage> {
    public:
        TProtoRequest(TStringBuf path, TMessage postData, const TAttributes& attrs)
            : IRequest<TMessage>(path, attrs)
            , Message_(std::move(postData))
        {
        }

        const TMessage& Get() const override {
            return Message_;
        }

    private:
        TMessage Message_;
    };

    template <typename TMessage>
    class TProtoReply: public IReply<TMessage> {
    public:
        TProtoReply(TMessage& response, TAttributes& attrs)
            : IReply<TMessage>(attrs)
            , Response_(response)
        {
        }

        void Set(const TMessage& message) override {
            Response_ = message;
        }

    private:
        TMessage& Response_;
    };

    template <typename TMessage>
    class TSerializedProtoRequest: public IRequest<TMessage> {
    public:
        TSerializedProtoRequest(TStringBuf path, const TString& postData, const TAttributes& attrs)
            : IRequest<TMessage>(path, attrs)
        {
            Y_PROTOBUF_SUPPRESS_NODISCARD Message_.ParseFromString(postData);
        }

        const TMessage& Get() const override {
            return Message_;
        }

    private:
        TMessage Message_;
    };

    template <typename TMessage>
    class TSerializedProtoReply: public IReply<TMessage> {
    public:
        TSerializedProtoReply(TString& response, TAttributes& attrs)
            : IReply<TMessage>(attrs)
            , Response_(response)
        {
        }

        void Set(const TMessage& message) override {
            Y_PROTOBUF_SUPPRESS_NODISCARD message.SerializeToString(&Response_);
        }

    private:
        TString& Response_;
    };

    class TJsonRequest: public IRequest<NJson::TJsonValue> {
    public:
        TJsonRequest(TStringBuf path, const TString& postData, const TAttributes& attrs)
            : IRequest<NJson::TJsonValue>(path, attrs)
        {
            TMemoryInput input(postData);
            Message_ = NJson::ReadJsonTree(&input, /* throw */ false);
        }

        const NJson::TJsonValue& Get() const override {
            return Message_;
        }

    private:
        NJson::TJsonValue Message_;
    };

    template <typename TMessage>
    class TJsonBatch2ProtoRequest: public IRequest<TVector<TMessage>> {
    public:
        TJsonBatch2ProtoRequest(TStringBuf path, const TString& postData, const TAttributes& attrs)
            : IRequest<TVector<TMessage>>(path, attrs)
        {
            NProtobufJson::TJson2ProtoConfig json2ProtoConfig = NProtobufJson::TJson2ProtoConfig();
            json2ProtoConfig.AllowUnknownFields = false;

            TStringStream requestsJson(postData);
            NJson::TJsonValue requests = NJson::ReadJsonTree(&requestsJson, true);
            for (const NJson::TJsonValue& request : requests.GetArray()) {
                TMessage protoRequest;
                NProtobufJson::Json2Proto(request, protoRequest, json2ProtoConfig);
                Requests_.emplace_back(protoRequest);
            }
        }

        const TVector<TMessage>& Get() const override {
            return Requests_;
        }

    private:
        TVector<TMessage> Requests_;
    };

    template <typename TMessage, bool PRETTY = false>
    class TProtoBatch2JsonReply: public IReply<TVector<TMessage>> {
    public:
        TProtoBatch2JsonReply(TString& response, TAttributes& attrs)
            : IReply<TVector<TMessage>>(attrs)
            , Response_(response)
        {
        }

        void Set(const TVector<TMessage>& protoResponses) override {
            NProtobufJson::TProto2JsonConfig cfg;
            cfg.FormatOutput = PRETTY;

            Response_.clear();
            TStringOutput response(Response_);

            NJson::TJsonWriter responseWriter(&response, true);
            responseWriter.OpenArray();
            for (const auto& protoResponse : protoResponses) {
                NProtobufJson::Proto2Json(protoResponse, responseWriter, cfg);
            }
            responseWriter.CloseArray();
        }

    private:
        TString& Response_;
    };
}
