#pragma once

#include <infra/libs/udp_metrics/api/api.pb.h>
#include <infra/libs/udp_metrics/logger/events/events_decl.ev.pb.h>

#include <infra/yp_service_discovery/libs/sdlib/endpointset.h>
#include <infra/yp_service_discovery/libs/sdlib/endpointset.pb.h>

#include <library/cpp/neh/neh.h>
#include <library/cpp/threading/future/async.h>

#include <util/folder/path.h>
#include <util/generic/maybe.h>
#include <util/network/address.h>
#include <util/network/socket.h>
#include <util/random/random.h>
#include <util/string/builder.h>

namespace NUdpMetrics {

template <typename TRequest>
class TSelfBalancingClient<TRequest>::TEndpointsProvider: public NYP::NServiceDiscovery::IEndpointSetProvider {
public:
    void Update(const NYP::NServiceDiscovery::TEndpointSetEx& endpointSet) override {
        TVector<TNetworkAddress> newAddresses;
        newAddresses.reserve(endpointSet.endpoints().size());
        for (const NYP::NServiceDiscovery::NApi::TEndpoint& endpoint : endpointSet.endpoints()) {
            if (endpoint.ip6_address()) {
                newAddresses.emplace_back(endpoint.ip6_address(), endpoint.port());
            } else {
                newAddresses.emplace_back(endpoint.ip4_address(), endpoint.port());
            }
        }

        TWriteGuard guard(AddressesMutex_);
        Addresses_.swap(newAddresses);
    }

    TVector<TNetworkAddress> GetAddresses() const {
        TReadGuard guard(AddressesMutex_);
        return Addresses_;
    }

private:
    TRWMutex AddressesMutex_;
    TVector<TNetworkAddress> Addresses_;
};

template <typename TRequest>
TSelfBalancingClient<TRequest>::TSelfBalancingClient(TSelfBalancingClientConfig config, NInfra::TLogger& logger)
    : Config_(std::move(config))
    , Logger_(logger)
    , EndpointSetManagerLogger_(CreateLogger(Config_.GetEndpointSetManagerLoggerConfig()))
    , EndpointSetManagerStartPool_(MakeHolder<TThreadPool>())
    , ServiceDiscoveryResolver_(MakeAtomicShared<NYP::NServiceDiscovery::TGrpcResolver>(Config_.GetSDConfig()))
    , EndpointSetManager_(NYP::NServiceDiscovery::TSDConfig(Config_.GetSDConfig()), ServiceDiscoveryResolver_)
{
    if (!EndpointSetManagerLogger_.IsNullLog()) {
        EndpointSetManager_.AssignLog(&EndpointSetManagerLogger_);
    }

    for (const auto& endpointSet : Config_.GetEndpointSetKeys()) {
        if (endpointSet.HasFilePath()) {
            NYP::NServiceDiscovery::TEndpointSetKey endpointSetKey(endpointSet.GetFilePath());
            UdpMetricsEndpointsProviders_.push_back(EndpointSetManager_.RegisterEndpointSet<TEndpointsProvider>(std::move(endpointSetKey)));
        } else {
            NYP::NServiceDiscovery::TEndpointSetKey endpointSetKey(endpointSet.GetCluster(), endpointSet.GetId());
            if (endpointSet.GetCluster() == Config_.GetLocalCluster()) {
                UdpMetricsLocalEndpointsProviders_.push_back(EndpointSetManager_.RegisterEndpointSet<TEndpointsProvider>(std::move(endpointSetKey)));
            } else {
                UdpMetricsEndpointsProviders_.push_back(EndpointSetManager_.RegisterEndpointSet<TEndpointsProvider>(std::move(endpointSetKey)));
            }
        }
    }

    NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();

    auto startEndpointSetManager = [this] (NInfra::TLogFramePtr logFrame, const bool isAsync) {
        auto updateCallback = [this] {
            UpdateClient();
        };

        try {
            logFrame->LogEvent(NEventlog::TUdpMetricsClientEndpointSetManagerStart(isAsync));
            EndpointSetManager_.Start(TDuration::Parse(Config_.GetSDConfig().GetUpdateFrequency()), {}, std::move(updateCallback));
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, NEventlog::TUdpMetricsClientEndpointSetManagerStartError(CurrentExceptionMessage()));
            return false;
        }
        logFrame->LogEvent(NEventlog::TUdpMetricsClientEndpointSetManagerStartSuccess());
        UpdateClient();
        return true;
    };

    if (!startEndpointSetManager(logFrame, /* isAsync */ false)) {
        if (Config_.GetAsyncStartOnFailedInitialization()) {
            EndpointSetManagerStartPool_->Start(1);

            NThreading::Async([&startEndpointSetManager, logFrame] {
                while (!startEndpointSetManager(logFrame, /* isAsync */ true)) {
                    const TDuration sleepDuration = TDuration::Seconds(5);
                    logFrame->LogEvent(NEventlog::TUdpMetricsClientSleep(ToString(sleepDuration)));
                    Sleep(sleepDuration);
                }
            }, *EndpointSetManagerStartPool_);
        }
    }
}

template <typename TRequest>
bool TSelfBalancingClient<TRequest>::IncreaseMetrics(const TRequest& request) const {
    TReadGuard guard(Mutex_);
    if (UdpMetricsClient_) {
        return UdpMetricsClient_->IncreaseMetrics(request);
    }
    return false;
}

template <typename TRequest>
void TSelfBalancingClient<TRequest>::ReopenLogs() {
    EndpointSetManagerLogger_.ReopenLog();
}

template <typename TRequest>
TLog TSelfBalancingClient<TRequest>::CreateLogger(const TLoggerConfig& config) const {
    if (config.GetPath().empty()) {
        return TLog();
    }

    return TLog(CreateFilteredOwningThreadedLogBackend(config.GetPath(), FromString<ELogPriority>(config.GetLevel()), config.GetQueueSize()));
}

template <typename TRequest>
bool TSelfBalancingClient<TRequest>::PingEndpoint(const TNetworkAddress& address, NInfra::TLogFramePtr framePtr) const {
    const TString& hostAndPort = PrintHostAndPort(NAddr::TAddrInfo(&*address.Begin()));
    framePtr->LogEvent(NEventlog::TUdpMetricsClientPingEndpoint(hostAndPort));

    NNeh::TMessage request(TStringBuilder() << "http://" << hostAndPort << Config_.GetPingHandle(), TString{});
    NNeh::TResponseRef response;
    try {
        response = NNeh::Request(request, nullptr)->Wait(TDuration::Parse(Config_.GetPingTimeout()));
    } catch (const yexception& e) {
        framePtr->LogEvent(ELogPriority::TLOG_WARNING, NEventlog::TUdpMetricsClientPingEndpointStatus(false, false, e.what()));
        return false;
    }

    bool pingSuccess = response && !response->IsError();
    TString errorMessage = response ? response->GetErrorText() : TString{};
    framePtr->LogEvent(NEventlog::TUdpMetricsClientPingEndpointStatus(true, pingSuccess, std::move(errorMessage)));

    return pingSuccess;
}

template <typename TRequest>
void TSelfBalancingClient<TRequest>::UpdateClient() {
    NInfra::TLogFramePtr framePtr = Logger_.SpawnFrame();

    const auto checkEndpointAlive = [this, framePtr](const TNetworkAddress& address) {
        return PingEndpoint(address, framePtr);
    };

    auto addAliveEndpoints = [&checkEndpointAlive](const TVector<TEndpointsProvider*>& endpointsProviders, TVector<TNetworkAddress>& aliveEndpoints) {
        for (TEndpointsProvider* provider : endpointsProviders) {
            TVector<TNetworkAddress> addresses = provider->GetAddresses();
            std::copy_if(std::make_move_iterator(addresses.begin()), std::make_move_iterator(addresses.end()), std::back_inserter(aliveEndpoints), checkEndpointAlive);
        }
    };

    TVector<TNetworkAddress> udpMetricsAddresses;
    addAliveEndpoints(UdpMetricsLocalEndpointsProviders_, udpMetricsAddresses);

    if (udpMetricsAddresses.empty()) {
        framePtr->LogEvent(NEventlog::TUdpMetricsClientLocalEndpointsEmpty());
        addAliveEndpoints(UdpMetricsEndpointsProviders_, udpMetricsAddresses);
    }

    NEventlog::TUdpMetricsClientAddresses logAddresses;
    for (const auto& address : udpMetricsAddresses) {
        *logAddresses.AddAddresses() = PrintHostAndPort(NAddr::TAddrInfo(&*address.Begin()));
    }
    framePtr->LogEvent(logAddresses);

    if (udpMetricsAddresses.empty()) {
        return;
    }

    TNetworkAddress address = udpMetricsAddresses.at(RandomNumber<size_t>(udpMetricsAddresses.size()));

    const TString& hostAndPort = PrintHostAndPort(NAddr::TAddrInfo(&*address.Begin()));
    framePtr->LogEvent(NEventlog::TUdpMetricsClientUseAddress(hostAndPort));

    TWriteGuard guard(Mutex_);
    UdpMetricsClient_.Reset(MakeHolder<NUdpMetrics::TClient<TRequest>>(std::move(address)));
}

} // namespace NUdpMetrics
