#pragma once

namespace NUdpMetrics::NSensors {

static inline void WriteLabels(NMonitoring::IMetricConsumer* consumer, const NMonitoring::ILabels& labels) {
    consumer->OnLabelsBegin();
    for (auto&& label : labels) {
        if (!label.Value().empty()) {
            consumer->OnLabel(label.Name(), label.Value());
        }
    };
    consumer->OnLabelsEnd();
}

template <typename TStorage, typename TEnabled>
TSensorsRegistry<TStorage, TEnabled>::TSensorsRegistry(const ui64 maxSensorsNumber)
    : SensorsStorage_(maxSensorsNumber)
    , ActualSensors_(maxSensorsNumber)
    , LastTouchedSensorsProcessor_([this]{ ProcessLastTouchedSensors(); }, TDuration::Zero())
{
    Y_ENSURE(maxSensorsNumber > 0, "TSensorsRegistry must be initialized with maxSensorsNumber > 0");
    LastTouchedSensorsProcessor_.Start();
}

template <typename TStorage, typename TEnabled>
THolder<typename TSensorsRegistry<TStorage, TEnabled>::IRateSensorAccessor> TSensorsRegistry<TStorage, TEnabled>::Rate(const NInfra::TSensorGroup& group) {
    return Rate(CreateLabels(group));
}

template <typename TStorage, typename TEnabled>
THolder<typename TSensorsRegistry<TStorage, TEnabled>::IRateSensorAccessor> TSensorsRegistry<TStorage, TEnabled>::Rate(NInfra::TSensorGroup baseGroup, TStringBuf name, const TVector<std::pair<TStringBuf, TStringBuf>>& labels) {
    baseGroup.AddLabels(labels);
    return Rate(NInfra::TSensorGroup{baseGroup, name});
}

template <typename TStorage, typename TEnabled>
THolder<typename TSensorsRegistry<TStorage, TEnabled>::IRateSensorAccessor> TSensorsRegistry<TStorage, TEnabled>::Rate(TLabels labels) {
    auto sensor = SensorsStorage_.Rate(labels);
    LastTouchedLabels_.Enqueue(std::move(labels));
    return sensor;
}

template <typename TStorage, typename TEnabled>
void TSensorsRegistry<TStorage, TEnabled>::RemoveSensor(const NInfra::TSensorGroup& group) {
    RemoveSensor(CreateLabels(group));
}

template <typename TStorage, typename TEnabled>
void TSensorsRegistry<TStorage, TEnabled>::RemoveSensor(TLabels labels) {
    SensorsStorage_.Delete(std::move(labels));
}

template <typename TStorage, typename TEnabled>
void TSensorsRegistry<TStorage, TEnabled>::SetCommonSensorGroup(NInfra::TSensorGroup group) {
    CommonSensorGroup_ = std::move(group);
}

template <typename TStorage, typename TEnabled>
const typename TSensorsRegistry<TStorage, TEnabled>::TLabels& TSensorsRegistry<TStorage, TEnabled>::CommonLabels() const noexcept {
    static TLabels emptyLabels = TLabels();
    return CommonSensorGroup_.Defined() ? CommonSensorGroup_->Labels() : emptyLabels;
}

template <typename TStorage, typename TEnabled>
void TSensorsRegistry<TStorage, TEnabled>::Accept(TInstant time, NMonitoring::IMetricConsumer* consumer) const {
    consumer->OnStreamBegin();

    if (const TLabels& labels = CommonLabels(); !labels.Empty()) {
        NSensors::WriteLabels(consumer, labels);
    }

    Append(time, consumer);
    consumer->OnStreamEnd();
}

template <typename TStorage, typename TEnabled>
void TSensorsRegistry<TStorage, TEnabled>::Append(TInstant time, NMonitoring::IMetricConsumer* consumer) const {
    TVector<TLabels> labelsList;
    ListLabels(labelsList);

    for (const TLabels& labels : labelsList) {
        THolder<ISensorAccessor> accessor = SensorsStorage_.Sensor(labels);
        if (const NMonitoring::EMetricType type = accessor->Type(); type != NMonitoring::EMetricType::UNKNOWN) {
            consumer->OnMetricBegin(type);

            NSensors::WriteLabels(consumer, labels);

            accessor->Accept(time, consumer);

            consumer->OnMetricEnd();
        }
    }
}

template <typename TStorage, typename TEnabled>
void TSensorsRegistry<TStorage, TEnabled>::Print(IOutputStream& outputStream, const ESensorsSerializationType type) const {
    NMonitoring::IMetricEncoderPtr encoder = nullptr;
    switch (type) {
        case ESensorsSerializationType::JSON: {
            encoder = NMonitoring::EncoderJson(&outputStream);
            break;
        }
        case ESensorsSerializationType::SPACK_V1: {
            encoder = NMonitoring::EncoderSpackV1(&outputStream, NMonitoring::ETimePrecision::SECONDS, NMonitoring::ECompression::LZ4);
            break;
        }
        default: {
            ythrow yexception() << "Unknown type for serialization";
        }
    }
    Accept(TInstant::Zero(), encoder.Get());
}

template <typename TStorage, typename TEnabled>
typename TSensorsRegistry<TStorage, TEnabled>::TLabels TSensorsRegistry<TStorage, TEnabled>::CreateLabels(const NInfra::TSensorGroup& group) const {
    TLabels labels = group.Labels();
    const TString name = CommonSensorGroup_.Defined() ? NInfra::TSensorGroup{CommonSensorGroup_.GetRef(), group}.Name() : group.Name();
    labels.Add("sensor", name);
    return labels;
}

template <typename TStorage, typename TEnabled>
void TSensorsRegistry<TStorage, TEnabled>::ListLabels(TVector<TLabels>& result) const {
    TGuard<TMutex> guard(ActualSensorsMutex_);
    result.reserve(ActualSensors_.Size());
    for (typename TCache::TIterator it = ActualSensors_.Begin(); it != ActualSensors_.End(); ++it) {
        result.push_back(it.Key());
    }
}

template <typename TStorage, typename TEnabled>
void TSensorsRegistry<TStorage, TEnabled>::ProcessLastTouchedSensors() {
    TVector<TLabels> labelsList;
    LastTouchedLabels_.DequeueAll(&labelsList);

    for (TLabels& labels : labelsList) {
        TGuard<TMutex> guard(ActualSensorsMutex_);
        if (auto it = ActualSensors_.Find(labels); it == ActualSensors_.End()) {
            TCacheItem value{labels, &SensorsStorage_};
            Y_ENSURE(ActualSensors_.Insert(std::move(labels), std::move(value)));
        }
    }
}

/// noop

template <typename TStorage, typename TEnabled>
NMonitoring::IGauge* TSensorsRegistry<TStorage, TEnabled>::Gauge(NMonitoring::ILabelsPtr) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::TLazyGauge* TSensorsRegistry<TStorage, TEnabled>::LazyGauge(NMonitoring::ILabelsPtr, std::function<double()>) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::TLazyIntGauge* TSensorsRegistry<TStorage, TEnabled>::LazyIntGauge(NMonitoring::ILabelsPtr, std::function<i64()>) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::TLazyCounter* TSensorsRegistry<TStorage, TEnabled>::LazyCounter(NMonitoring::ILabelsPtr, std::function<ui64()>) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::TLazyRate* TSensorsRegistry<TStorage, TEnabled>::LazyRate(NMonitoring::ILabelsPtr, std::function<ui64()>) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::IIntGauge* TSensorsRegistry<TStorage, TEnabled>::IntGauge(NMonitoring::ILabelsPtr) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::ICounter* TSensorsRegistry<TStorage, TEnabled>::Counter(NMonitoring::ILabelsPtr) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::IRate* TSensorsRegistry<TStorage, TEnabled>::Rate(NMonitoring::ILabelsPtr) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::IHistogram* TSensorsRegistry<TStorage, TEnabled>::HistogramCounter(NMonitoring::ILabelsPtr, NMonitoring::IHistogramCollectorPtr) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::IHistogram* TSensorsRegistry<TStorage, TEnabled>::HistogramRate(NMonitoring::ILabelsPtr, NMonitoring::IHistogramCollectorPtr) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::IHistogram* TSensorsRegistry<TStorage, TEnabled>::HistogramCounter(NMonitoring::ILabelsPtr, std::function<NMonitoring::IHistogramCollectorPtr()>) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
NMonitoring::IHistogram* TSensorsRegistry<TStorage, TEnabled>::HistogramRate(NMonitoring::ILabelsPtr, std::function<NMonitoring::IHistogramCollectorPtr()>) {
    return nullptr;
}

template <typename TStorage, typename TEnabled>
void TSensorsRegistry<TStorage, TEnabled>::RemoveMetric(const NMonitoring::ILabels&) noexcept {
}

} // namespace NUdpMetrics::NSensors
