#pragma once

#include "sensors_storage.h"

#include <infra/libs/background_thread/background_thread.h>
#include <infra/libs/sensors/sensor_group.h>

#include <library/cpp/cache/cache.h>
#include <library/cpp/monlib/encode/encoder.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/refcount.h>
#include <util/system/mutex.h>
#include <util/thread/lfqueue.h>

namespace NUdpMetrics::NSensors {

enum class ESensorsSerializationType {
    JSON = 1,
    SPACK_V1 = 2
};

template <typename TStorage, typename = typename std::enable_if_t<std::is_base_of_v<ISensorsStorage, TStorage>>>
class TSensorsRegistry: public NMonitoring::IMetricRegistry {
public:
    using TLabels = NMonitoring::TLabels;

    using ISensorAccessor = typename TStorage::ISensorAccessor;
    using IRateSensorAccessor = typename TStorage::IRateSensorAccessor;

private:
    struct TCounter: TAtomicCounter {
        void IncCount(const TLabels&) {
            Inc();
        }

        void DecCount(const TLabels&) {
            Dec();
        }
    };

    using TQueue = TLockFreeQueue<TLabels, TCounter>;

private:
    using TCacheKey = TLabels;

    struct TCacheItem {
        TLabels Labels;
        TStorage* Storage;
    };

    struct TDeleter {
        static void Destroy(TCacheItem& value) {
            value.Storage->Delete(std::move(value.Labels));
        }
    };

    using TCache = TLRUCache<TCacheKey, TCacheItem, TDeleter>;

public:
    TSensorsRegistry(const ui64 maxSensorsNumber);

    THolder<IRateSensorAccessor> Rate(const NInfra::TSensorGroup& group);
    THolder<IRateSensorAccessor> Rate(NInfra::TSensorGroup baseGroup, TStringBuf name, const TVector<std::pair<TStringBuf, TStringBuf>>& labels = {});

    void RemoveSensor(const NInfra::TSensorGroup& group);

    void SetCommonSensorGroup(NInfra::TSensorGroup group);
    const TLabels& CommonLabels() const noexcept override;

    void Accept(TInstant time, NMonitoring::IMetricConsumer* consumer) const override;
    void Append(TInstant time, NMonitoring::IMetricConsumer* consumer) const override;

    void Print(IOutputStream& stream, const ESensorsSerializationType type) const;

private:
    THolder<IRateSensorAccessor> Rate(TLabels labels);

    void RemoveSensor(TLabels labels);

    TLabels CreateLabels(const NInfra::TSensorGroup& group) const;

    void ListLabels(TVector<TLabels>& result) const;

    void ProcessLastTouchedSensors();

private: /// noop
    NMonitoring::IGauge* Gauge(NMonitoring::ILabelsPtr labels) override;
    NMonitoring::TLazyGauge* LazyGauge(NMonitoring::ILabelsPtr labels, std::function<double()> supplier) override;
    NMonitoring::IIntGauge* IntGauge(NMonitoring::ILabelsPtr labels) override;
    NMonitoring::TLazyIntGauge* LazyIntGauge(NMonitoring::ILabelsPtr labels, std::function<i64()> supplier) override;
    NMonitoring::ICounter* Counter(NMonitoring::ILabelsPtr labels) override;
    NMonitoring::TLazyCounter* LazyCounter(NMonitoring::ILabelsPtr labels, std::function<ui64()> supplier) override;
    NMonitoring::IRate* Rate(NMonitoring::ILabelsPtr labels) override;
    NMonitoring::TLazyRate* LazyRate(NMonitoring::ILabelsPtr labels, std::function<ui64()> supplier) override;
    NMonitoring::IHistogram* HistogramCounter(NMonitoring::ILabelsPtr labels, NMonitoring::IHistogramCollectorPtr collector) override;
    NMonitoring::IHistogram* HistogramRate(NMonitoring::ILabelsPtr labels, NMonitoring::IHistogramCollectorPtr collector) override;
    NMonitoring::IHistogram* HistogramCounter(NMonitoring::ILabelsPtr labels, std::function<NMonitoring::IHistogramCollectorPtr()> collector) override;
    NMonitoring::IHistogram* HistogramRate(NMonitoring::ILabelsPtr labels, std::function<NMonitoring::IHistogramCollectorPtr()> collector) override;
    void RemoveMetric(const NMonitoring::ILabels& labels) noexcept override;

private:
    TMaybe<NInfra::TSensorGroup> CommonSensorGroup_;
    mutable TStorage SensorsStorage_;
    TQueue LastTouchedLabels_;
    TMutex ActualSensorsMutex_;
    TCache ActualSensors_;
    NInfra::TBackgroundThread LastTouchedSensorsProcessor_;
};

} // namespace NUdpMetrics::NSensors

#include "sensors_registry-inl.h"
