#pragma once

#include <util/generic/hash.h>
#include <util/generic/maybe.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/rwlock.h>

#include <array>

template <class K, class V, size_t BucketsNumber>
class TConcurrentHashMap : TNonCopyable {
public:
    class TAccessor {
    public:
        TAccessor() = default;

        TAccessor(TReadGuard&& lock, V* value)
            : ReadLock_(std::move(lock))
            , Value_(value)
        {
        }

        TAccessor(TWriteGuard&& lock, V* value)
            : WriteLock_(std::move(lock))
            , Value_(value)
        {
        }

        TAccessor(TReadGuard&& lock, const V* value)
            : TAccessor(std::move(lock), const_cast<V*>(value))
        {
        }

        TAccessor(TWriteGuard&& lock, const V* value)
            : TAccessor(std::move(lock), const_cast<V*>(value))
        {
        }

        V* Value() const {
            return Value_;
        }

        V* operator->() const {
            return Value_;
        }

        V* operator*() const {
            return Value_;
        }

        explicit operator bool() const {
            return Value_ != nullptr;
        }

    private:
        TMaybe<TReadGuard> ReadLock_;
        TMaybe<TWriteGuard> WriteLock_;
        V* const Value_ = nullptr;
    };

public:
    TConcurrentHashMap(const ui64 reserveSize = 0) {
        static_assert((BucketsNumber & (BucketsNumber - 1)) == 0, "BucketsNumber must be power of 2");

        Reserve(reserveSize);
    }

    void Reserve(const ui64 size) {
        const size_t reserveForBucket = size / BucketsNumber;
        for (size_t i = 0; i < BucketsNumber; ++i) {
            Buckets_[i].reserve(reserveForBucket);
        }
    }

    /**
     * First tries to get an element with read guard.
     * If element is absent, tries to emplace it and return an accessor with write guard.
     */
    template <typename... TArgs>
    std::pair<TAccessor, bool> GetOrEmplace(K&& key, TArgs&&... args) {
        if (TAccessor accessor = Get(key); accessor) {
            return {std::move(accessor), false};
        }
        return Emplace(std::forward<K>(key), std::forward<TArgs>(args)...);
    }

    template <typename... TArgs>
    std::pair<TAccessor, bool> GetOrEmplace(const K& key, TArgs&&... args) {
        if (TAccessor accessor = Get(key); accessor) {
            return {std::move(accessor), false};
        }
        return Emplace(key, std::forward<TArgs>(args)...);
    }

    template <typename... TArgs>
    std::pair<TAccessor, bool> Emplace(K&& key, TArgs&&... args) {
        const size_t bucket = GetBucketForKey(key);
        TWriteGuard guard(BucketMutexes_[bucket]);

        auto [it, inserted] = Buckets_[bucket].try_emplace(std::forward<K>(key), std::forward<TArgs>(args)...);
        if (inserted) {
            AtomicIncrement(Size_);
        }

        return {TAccessor{std::move(guard), &it->second}, inserted};
    }

    template <typename... TArgs>
    std::pair<TAccessor, bool> Emplace(const K& key, TArgs&&... args) {
        const size_t bucket = GetBucketForKey(key);
        TWriteGuard guard(BucketMutexes_[bucket]);

        auto [it, inserted] = Buckets_[bucket].try_emplace(key, std::forward<TArgs>(args)...);
        if (inserted) {
            AtomicIncrement(Size_);
        }

        return {TAccessor{std::move(guard), &it->second}, inserted};
    }

    void Delete(const K& key) {
        const size_t bucket = GetBucketForKey(key);
        TWriteGuard guard(BucketMutexes_[bucket]);
        if (Buckets_[bucket].erase(key)) {
            AtomicDecrement(Size_);
        }
    }

    TAccessor Get(const K& key) const {
        const size_t bucket = GetBucketForKey(key);
        TReadGuard guard(BucketMutexes_[bucket]);
        if (const V* value = Buckets_[bucket].FindPtr(key)) {
            return {std::move(guard), value};
        }
        return {};
    }

    size_t Size() const {
        return AtomicGet(Size_);
    }

private:
    size_t GetBucketForKey(const K& key) const {
        return THash<K>()(key) & (BucketsNumber - 1);
    }

private:
    mutable std::array<TRWMutex, BucketsNumber> BucketMutexes_;
    std::array<THashMap<K, V>, BucketsNumber> Buckets_;
    TAtomic Size_ = 0;
};
