#include "storage.h"

namespace NUdpMetrics::NSensors {

TSensorsStorage::TSensorsStorage(const ui64 reserveSize)
    : Values_(reserveSize)
{
}

size_t TSensorsStorage::Size() const {
    return Values_.Size();
}

THolder<TSensorsStorage::ISensorAccessor> TSensorsStorage::Sensor(TLabels labels) {
    return MakeHolder<TSensorAccessor>(std::move(labels), this);
}

THolder<TSensorsStorage::IRateSensorAccessor> TSensorsStorage::Rate(TLabels labels) {
    return MakeHolder<TRateSensorAccessor>(std::move(labels), this);
}

void TSensorsStorage::Delete(TLabels labels) {
    Values_.Delete(std::move(labels));
}

/// TSensorsStorage::TSensorAccessor

TSensorsStorage::TSensorAccessor::TSensorAccessor(TLabels labels, TSensorsStorage* storage)
    : Labels_(std::move(labels))
    , Storage_(storage)
{
}

NMonitoring::EMetricType TSensorsStorage::TSensorAccessor::Type() const noexcept {
    auto accessor = Storage_->Values_.Get(Labels_);
    return accessor ? accessor->Get()->Type() : NMonitoring::EMetricType::UNKNOWN;
}

void TSensorsStorage::TSensorAccessor::Accept(TInstant time, NMonitoring::IMetricConsumer* consumer) const {
    if (auto accessor = Storage_->Values_.Get(Labels_); accessor) {
        accessor->Get()->Accept(time, consumer);
    }
}

/// TSensorsStorage::TRateSensorAccessor

TSensorsStorage::TRateSensorAccessor::TRateSensorAccessor(TLabels labels, TSensorsStorage* storage)
    : Labels_(std::move(labels))
    , Storage_(storage)
{
}

bool TSensorsStorage::TRateSensorAccessor::Init() noexcept {
    return Storage_->Values_.GetOrEmplace(Labels_, MakeIntrusive<NMonitoring::TRate>(static_cast<ui64>(0))).second;
}

ui64 TSensorsStorage::TRateSensorAccessor::Inc() noexcept {
    if (auto [accessor, inserted] = Storage_->Values_.GetOrEmplace(Labels_, MakeIntrusive<NMonitoring::TRate>(static_cast<ui64>(1))); !inserted) {
        return static_cast<NMonitoring::TRate*>(accessor->Get())->Inc();
    }
    return 1;
}

ui64 TSensorsStorage::TRateSensorAccessor::Add(ui64 n) noexcept {
    if (auto [accessor, inserted] = Storage_->Values_.GetOrEmplace(Labels_, MakeIntrusive<NMonitoring::TRate>(n)); !inserted) {
        return static_cast<NMonitoring::TRate*>(accessor->Get())->Add(n);
    }
    return n;
}

ui64 TSensorsStorage::TRateSensorAccessor::Get() const noexcept {
    return TryGet().GetOrElse(0);
}

void TSensorsStorage::TRateSensorAccessor::Reset() noexcept {
    Storage_->Values_.Delete(Labels_);
}

TMaybe<ui64> TSensorsStorage::TRateSensorAccessor::TryGet() const noexcept {
    if (auto accessor = Storage_->Values_.Get(Labels_); accessor) {
        return static_cast<NMonitoring::TRate*>(accessor->Get())->Get();
    }
    return Nothing();
}

void TSensorsStorage::TRateSensorAccessor::Accept(TInstant time, NMonitoring::IMetricConsumer* consumer) const {
    auto accessor = Storage_->Values_.Get(Labels_);
    if (accessor) {
        accessor->Get()->Accept(time, consumer);
    }
}

} // namespace NUdpMetrics::NSensors
