#include "service.h"

#include <infra/libs/udp_metrics/logger/events/events_decl.ev.pb.h>
#include <infra/libs/udp_metrics/router_api/router_api.h>
#include <infra/libs/udp_metrics/sensors/sensors.h>

#include <infra/libs/sensors/sensor_group.h>
#include <infra/libs/service_iface/request.h>
#include <infra/libs/service_iface/str_iface.h>

#include <library/cpp/string_utils/base64/base64.h>

namespace NUdpMetrics {

namespace {

template <typename TMessage>
TString GetAttributes(const NInfra::TRequestPtr<TMessage> request) {
    TStringBuilder result;
    for (const auto& [key, value] : request->Attributes()) {
        result << key << "=" << value << ";";
    }
    return result;
}

NEventlog::TCrossSection MakeCrossSectionEvent(const TMetricsCrossSectionsConfig::TCrossSectionConfig& crossSection) {
    NEventlog::TCrossSection result;
    result.SetHandlePath(crossSection.GetFetchHandle().GetPath());
    *result.MutableLabels() = crossSection.GetLabels();
    return result;
}

NEventlog::TMetric MakeMetricEvent(const NApi::TMetric& metric) {
    NEventlog::TMetric result;
    result.SetName(metric.GetName());
    for (const auto& it : metric.GetLabels()) {
        auto& labels = (*result.MutableLabels())[it.first];
        for (const NApi::TLabelValue& value : it.second.GetValues()) {
            if (value.HasNotSet()) {
                labels.AddValues()->MutableNotSet();
            } else {
                labels.AddValues()->SetValue(value.GetValue());
            }
        }
    }
    return result;
}

} // anonymous namespace

TMetricsCrossSection::TMetricsCrossSection(TConfig config)
    : Config(std::move(config))
    , Name(Config.GetFetchHandle().GetPath())
    , Labels(Config.GetLabels().begin(), Config.GetLabels().end())
    , SensorsRegistry(MakeHolder<TSensorsRegistry>(Config.GetMaxSensorsNumber()))
{
}

bool TMetricsCrossSection::IsSuitableForMetric(const NApi::TMetric& metric) const {
    for (const TString& label : Labels) {
        if (auto it = metric.GetLabels().find(label); it == metric.GetLabels().end()) {
            return false;
        } else {
            bool hasSetValue = false;
            for (const NApi::TLabelValue& value : it->second.GetValues()) {
                if (value.HasValue()) {
                    hasSetValue = true;
                }
            }
            if (!hasSetValue) {
                return false;
            }
        }
    }

    for (const auto& it : metric.GetLabels()) {
        if (!Labels.contains(it.first)) {
            bool hasNotSet = false;
            for (const NApi::TLabelValue& value : it.second.GetValues()) {
                if (value.HasNotSet()) {
                    hasNotSet = true;
                    break;
                }
            }
            if (!hasNotSet) {
                return false;
            }
        }
    }
    return true;
}

TService::TService(TUdpMetricsServiceConfig config)
    : Config_(std::move(config))
    , UdpService_(Config_.GetUdpServiceConfig(), [this](NUdp::TUdpPacket packet) { this->HandleUdpRequest(std::move(packet)); })
    , HttpService_(Config_.GetHttpServiceConfig(), CreateRouter(*this, Config_.GetMetricsCrossSectionsConfig()))
    , Logger_(Config_.GetLoggerConfig())
    , StatsSensorRegistry_(NSensors::MAX_STATS_SENSORS_NUMBER)
{
    StatsSensorRegistry_.SetCommonSensorGroup(NSensors::COMMON_SENSOR_GROUP);

    UdpService_.SetQueueOverflowCallback([this](NUdp::TUdpPacket) {
        StatsSensorRegistry_.Rate(NSensors::UDP_REQUESTS_QUEUE_OVERFLOW_GROUP)->Inc();
    });

    UdpService_.SetErrorCallback([this](TString error) {
        auto sensorGroup = NSensors::UDP_INTERNAL_ERRORS_GROUP;
        sensorGroup.AddLabel("error", std::move(error));
        StatsSensorRegistry_.Rate(sensorGroup)->Inc();
    });

    const auto& crossSectionConfigs = Config_.GetMetricsCrossSectionsConfig().GetCrossSections();
    MetricsCrossSections_.reserve(crossSectionConfigs.size());
    for (const auto& crossSectionConfig : crossSectionConfigs) {
        MetricsCrossSections_.emplace_back(crossSectionConfig);
    }
}

void TService::Start() {
    UdpService_.Start(Logger_.SpawnFrame());
    HttpService_.Start(Logger_.SpawnFrame());
    Logger_.SpawnFrame()->LogEvent(NEventlog::TUdpMetricsServiceStart());
}

void TService::Wait() {
    UdpService_.Wait(Logger_.SpawnFrame());
    HttpService_.Wait(Logger_.SpawnFrame());
}

void TService::Ping(NInfra::TRequestPtr<NApi::TReqPing>, NInfra::TReplyPtr<NApi::TRspPing> reply) {
    Logger_.SpawnFrame()->LogEvent(NEventlog::TPing());

    NApi::TRspPing result;
    result.SetData("pong");
    reply->Set(result);
}

void TService::SensorsJson(NInfra::TRequestPtr<NApi::TReqSensors> request, NInfra::TReplyPtr<NApi::TRspSensors> reply) {
    auto logFrame = Logger_.SpawnFrame();
    logFrame->LogEvent(NEventlog::TSensors(request->Path(), "json", GetAttributes(request)));

    NApi::TRspSensors result;
    TStringOutput stream(*result.MutableData());

    TStringBuf crossSectionName = request->Path();
    if (const TSensorsRegistry* registry; crossSectionName.ChopSuffix("/json") && (registry = FindRegistry(crossSectionName))) {
        registry->Print(stream, NSensors::ESensorsSerializationType::JSON);
    } else {
        Y_ASSERT(false);
        Y_ENSURE(false, "No sensors registry found by path " << request->Path());
    }

    reply->Set(result);

    if (Config_.GetLogSensorsResponse()) {
        logFrame->LogEvent(NEventlog::TSensorsResponse(Base64Encode(result.GetData())));
    }
}

void TService::Sensors(NInfra::TRequestPtr<NApi::TReqSensors> request, NInfra::TReplyPtr<NApi::TRspSensors> reply) {
    auto logFrame = Logger_.SpawnFrame();
    logFrame->LogEvent(NEventlog::TSensors(request->Path(), "spack_v1", GetAttributes(request)));

    NApi::TRspSensors result;
    TStringOutput stream(*result.MutableData());

    if (const TSensorsRegistry* registry = FindRegistry(request->Path())) {
        registry->Print(stream, NSensors::ESensorsSerializationType::SPACK_V1);
    } else {
        Y_ASSERT(false);
        Y_ENSURE(false, "No sensors registry found by path " << request->Path());
    }

    reply->SetAttribute("Content-Type", "application/x-solomon-spack");
    reply->Set(result);

    if (Config_.GetLogSensorsResponse()) {
        logFrame->LogEvent(NEventlog::TSensorsResponse(Base64Encode(result.GetData())));
    }
}

void TService::ReopenLog(NInfra::TRequestPtr<NApi::TReqReopenLog>, NInfra::TReplyPtr<NApi::TRspReopenLog> reply) {
    Logger_.SpawnFrame()->LogEvent(NEventlog::TReopenLog());

    Logger_.ReopenLog();

    NApi::TRspReopenLog result;
    result.SetData("done");
    reply->Set(result);
}

void TService::Shutdown(NInfra::TRequestPtr<NApi::TReqShutdown> /* request */, NInfra::TReplyPtr<NApi::TRspShutdown> reply) {
    Logger_.SpawnFrame()->LogEvent(NEventlog::TShutdown());

    UdpService_.ShutDown(Logger_.SpawnFrame());
    HttpService_.ShutDown();

    NApi::TRspShutdown result;
    result.SetData("done");
    reply->Set(result);
}

TVector<NInfra::TSensorGroup> TService::MakeSensors(const NApi::TMetric& metric, const TMetricsCrossSection& crossSection) const {
    if (!crossSection.IsSuitableForMetric(metric)) {
        return {};
    }

    NInfra::TSensorGroup baseSensorGroup(metric.GetName());

    TVector<NInfra::TSensorGroup> previousLevel;
    TVector<NInfra::TSensorGroup> currentLevel = {std::move(baseSensorGroup)};
    for (const TString& label : crossSection.Labels) {
        currentLevel.swap(previousLevel);
        currentLevel.clear();

        const NApi::TLabelValues& values = metric.GetLabels().at(label);
        for (const NApi::TLabelValue& value : values.GetValues()) {
            if (value.HasNotSet()) {
                continue;
            }
            for (NInfra::TSensorGroup previousLevelSensorGroup : previousLevel) {
                previousLevelSensorGroup.AddLabel(label, value.GetValue());
                currentLevel.push_back(std::move(previousLevelSensorGroup));
            }
        }
    }

    return currentLevel;
}

NInfra::TRequestPtr<NApi::TReqIncreaseMetrics> TService::ParseUdpPacket(NUdp::TUdpPacket packet, NInfra::TLogFramePtr /* logFrame */) const {
    return NInfra::RequestPtr<NInfra::TSerializedProtoRequest<NApi::TReqIncreaseMetrics>>("", packet.Data, NInfra::TAttributes{});
}

void TService::HandleUdpRequest(NUdp::TUdpPacket packet) {
    NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
    logFrame->LogEvent(ELogPriority::TLOG_DEBUG, NEventlog::TUdpRequest(packet.Address.GetIp(), packet.Address.GetPort(), packet.Data.size()));

    StatsSensorRegistry_.Rate(NSensors::UDP_INCREASE_METRICS_REQUESTS_GROUP)->Inc();

    NInfra::TRequestPtr<NApi::TReqIncreaseMetrics> request;
    try {
        request = ParseUdpPacket(std::move(packet), logFrame);
    } catch (...) {
        logFrame->LogEvent(ELogPriority::TLOG_DEBUG, NEventlog::TUdpPacketParseError(CurrentExceptionMessage()));
        return;
    }

    if (request) {
        IncreaseMetrics(request, nullptr, logFrame);
    } else {
        logFrame->LogEvent(ELogPriority::TLOG_DEBUG, NEventlog::TUdpPacketParseError("ParseUdpPacket returned uninitialized request"));
    }
}

void TService::IncreaseMetrics(NInfra::TRequestPtr<NApi::TReqIncreaseMetrics> request, NInfra::TReplyPtr<NApi::TRspIncreaseMetrics>) {
    IncreaseMetrics(request, nullptr, Logger_.SpawnFrame());
}

void TService::IncreaseMetrics(NInfra::TRequestPtr<NApi::TReqIncreaseMetrics> request, NInfra::TReplyPtr<NApi::TRspIncreaseMetrics>, NInfra::TLogFramePtr logFrame) {
    const NApi::TReqIncreaseMetrics& increaseMetricsRequest = request->Get();

    for (const NApi::TMetric& metric : increaseMetricsRequest.GetMetrics()) {
        for (const TMetricsCrossSection& crossSection : MetricsCrossSections_) {
            TVector<NInfra::TSensorGroup> sensors = MakeSensors(metric, crossSection);
            if (!sensors.empty()) {
                for (const NInfra::TSensorGroup& sensor : sensors) {
                    crossSection.SensorsRegistry->Rate(sensor)->Inc();
                }
            } else {
                if (logFrame->AcceptLevel(ELogPriority::TLOG_DEBUG)) {
                    logFrame->LogEvent(ELogPriority::TLOG_DEBUG, NEventlog::TCreateCrossSectionSensorError(MakeCrossSectionEvent(crossSection.Config), MakeMetricEvent(metric)));
                }
            }
        }
    }
}

TService::TSensorsRegistry* TService::FindRegistry(const TStringBuf name) const {
    if (name == "/sensors") {
        return const_cast<TService::TSensorsRegistry*>(&StatsSensorRegistry_);
    }

    auto found = FindIfPtr(MetricsCrossSections_.begin(), MetricsCrossSections_.end(), [name](const auto& crossSection) {
        return crossSection.Name == name;
    });
    return found ? found->SensorsRegistry.Get() : nullptr;
}

} // namespace NUdpMetrics
