#pragma once

#include <infra/libs/udp_metrics/client/client.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/protobuf/json/json2proto.h>

#include <util/folder/path.h>
#include <util/stream/file.h>

namespace NUdpMetrics {

template <typename TRequest>
class TUdpSender {
public:
    static void Run(int argc, const char* argv[]) {
        TOptions options = ParseOptions(argc, argv);

        TNetworkAddress address = MakeAddress(options);
        TRequest request = MakeRequest(options);

        TClient<TRequest> client(std::move(address));
        client.IncreaseMetrics(request);
    }

private:
    struct TOptions {
        TString Address;
        TString JsonRequest;
        TFsPath FilePath;
    };

    static TOptions ParseOptions(int argc, const char* argv[]) {
        NLastGetopt::TOpts opts;

        opts.SetFreeArgsNum(0);

        TOptions result;

        opts.AddLongOption("address", "udp_metrics server address in <hostname>:<port> format")
            .Optional()
            .RequiredArgument()
            .StoreResult(&result.Address);

        opts.AddLongOption("data", "Request in JSON format")
            .Optional()
            .RequiredArgument()
            .StoreResult(&result.JsonRequest);

        opts.AddLongOption("path", "Path to file with request in JSON format")
            .Optional()
            .RequiredArgument()
            .StoreResult(&result.FilePath);

        NLastGetopt::TOptsParseResult{&opts, argc, argv};

        if (result.JsonRequest.empty() == !result.FilePath.Exists()) {
            ythrow yexception() << "Only one option must be specified: \"--data\" or \"--path\"";
        }

        return result;
    }

    static TNetworkAddress MakeAddress(const TOptions& options) {
        TStringBuf host;
        TStringBuf port;

        if (!TStringBuf(options.Address).TryRSplit(':', host, port)) {
            ythrow yexception() << "Incorrect address format";
        }

        return TNetworkAddress(TString{host}, FromString<ui16>(port));
    }

    static TRequest MakeRequest(const TOptions& options) {
        THolder<IInputStream> inputStream;
        if (!options.JsonRequest.empty()) {
            inputStream = MakeHolder<TMemoryInput>(options.JsonRequest);
        } else {
            inputStream = MakeHolder<TUnbufferedFileInput>(options.FilePath);
        }

        return  NProtobufJson::Json2Proto<TRequest>(*inputStream);
    }
};

} // namespace NUdpMetrics
