#include "service.h"

#include <util/string/cast.h>
#include <util/system/hostname.h>

namespace NInfra {

TUdpService::TUdpService(const TUdpServiceConfig& config,
                         NUdp::TCallback callback)
    : Config_(config)
    , Callback_(callback)
{
    NUdp::TServerOptions options;
    options
        .SetPort(config.GetPort())
        .SetWorkers(config.GetWorkers())
        .SetExecutors(config.GetExecutors())
        .SetErrorThreads(config.GetErrorThreads());

    if (config.HasQueueSizeLimit()) {
        options.SetQueueSizeLimit(config.GetQueueSizeLimit());
    }

    if (config.HasErrorQueueSizeLimit()) {
        options.SetErrorQueueSizeLimit(config.GetErrorQueueSizeLimit());
    }

    if (config.HasPollTimeout()) {
        options.SetPollTimeout(FromString<TDuration>(config.GetPollTimeout()));
    }

    Server_.Reset(new NUdp::TUdpServer(options, Callback_));
}

void TUdpService::Start(TLogFramePtr logFrame) {
    Server_->Start();
    logFrame->LogEvent(NLogEvent::TStartService("udp", ToString(GetHostName()) + ":" + ToString(Config_.GetPort())));
}

void TUdpService::Wait(TLogFramePtr logFrame) {
    Server_->Wait();
    logFrame->LogEvent(NLogEvent::TStopService("udp"));
}

void TUdpService::ShutDown(TLogFramePtr logFrame) {
    Server_->Stop();
    logFrame->LogEvent(NLogEvent::TStopService("udp"));
}

void TUdpService::SetErrorCallback(NUdp::TErrorCallback errorCallback) {
    Server_->SetErrorCallback(std::move(errorCallback));
}

void TUdpService::SetQueueOverflowCallback(NUdp::TQueueOverflowCallback queueOverflowCallback) {
    Server_->SetQueueOverflowCallback(std::move(queueOverflowCallback));
}

} // namespace NInfra

