#pragma once

#include <util/generic/maybe.h>
#include <util/generic/ptr.h>
#include <util/generic/string.h>
#include <util/network/address.h>
#include <util/ysaveload.h>

namespace NUnbound {

////////////////////////////////////////////////////////////////////////////////

class TProjectOutboundStats;

////////////////////////////////////////////////////////////////////////////////

class TProjectId {
public:
    TProjectId(ui32 id);

    ui32 Id() const noexcept;

    TString ToString() const;

    bool operator==(const TProjectId& rhs) const noexcept = default;

    static TMaybe<ui32> ProjectIdFromClientSubnetAddress(
        const NAddr::IRemoteAddr& clientSubnetAddress
    ) {
        if (!clientSubnetAddress.Addr() || clientSubnetAddress.Addr()->sa_family != AF_INET6) {
            return Nothing();
        }

        const in6_addr* ss = &((const sockaddr_in6*)clientSubnetAddress.Addr())->sin6_addr;
        ui8* bytes = (ui8*)ss;
        return (bytes[8] << 24) |
                (bytes[9] << 16) |
                (bytes[10] << 8) |
                (bytes[11]);
    }

    static TMaybe<TProjectId> FromClientSubnetAddress(
        const NAddr::IRemoteAddr& clientSubnetAddress
    ) {
        if (TMaybe<ui32> id = ProjectIdFromClientSubnetAddress(clientSubnetAddress);
            !id.Defined())
        {
            return Nothing();
        } else {
            return TProjectId(*id);
        }
    }

    Y_SAVELOAD_DEFINE(Id_);

private:
    ui32 Id_;
};

////////////////////////////////////////////////////////////////////////////////

struct TProjectOutboundLimiterConfig {
    size_t MaxCacheSize;
};

////////////////////////////////////////////////////////////////////////////////

class TProjectOutboundLimiter {
public:
    enum EStatus {
        OK = 1,
        LIMITED = 2,
    };

public:
    TProjectOutboundLimiter(TProjectOutboundLimiterConfig config);
    ~TProjectOutboundLimiter();

    EStatus OnRequest(
        NAddr::IRemoteAddrPtr clientSubnetAddress,
        NAddr::IRemoteAddrRef nameserverAddress,
        TString nameserverName,
        time_t now);

    void AddStats(
        TProjectOutboundStats& stats);

private:
    class TImpl;
    THolder<TImpl> Impl_;
};

////////////////////////////////////////////////////////////////////////////////

class TProjectOutboundStats {
public:
    using TIterateCallbackFunc = std::function<bool(
            const TProjectId&,
            NAddr::IRemoteAddrRef,
            const TString&,
            ui64
        )>;

    TProjectOutboundStats();
    ~TProjectOutboundStats();

    void Add(
        TProjectId projectId,
        NAddr::IRemoteAddrRef nameserverAddress,
        TString nameserverName,
        ui64 requestsNumber,
        TInstant lastTimestamp);

    void Add(
        const TProjectOutboundStats& other);

    bool Iterate(TIterateCallbackFunc&& func) const;

    TInstant LowestTimestamp() const noexcept;

    size_t EntriesNumber() const noexcept;

    void Save(IOutputStream* s) const;
    void Load(IInputStream* s);

private:
    class TImpl;
    THolder<TImpl> Impl_;
};

////////////////////////////////////////////////////////////////////////////////

} // namespace NUnbound
