#include "service.h"

#include <infra/libs/unbound/yp_dns/logger/events/events_decl.ev.pb.h>
#include <infra/libs/unbound/yp_dns/logger/make_events.h>

#include <infra/libs/yp_dns/record_set/record_set.h>
#include <infra/libs/yp_dns/replication/merge.h>
#include <infra/libs/yp_dns/zone/response_policy.h>
#include <infra/libs/yp_dns/zone_listing/yp_replica.h>
#include <infra/libs/yp_dns/zonefile/writer.h>

#include <yp/cpp/yp/token.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/threading/async_task_batch/async_task_batch.h>
#include <library/cpp/threading/future/async.h>

#include <util/random/entropy.h>
#include <util/random/shuffle.h>
#include <util/system/hostname.h>
#include <util/system/mlock.h>

size_t THash<DNSName>::operator()(const DNSName& name) const {
    return name.hash();
}

namespace NYP::DNS {

using NYP::NClient::NApi::NProto::TDnsRecordSetSpec;

namespace NSensors {

constexpr TStringBuf NAMESPACE = "yp_dns_service";

// Labels
constexpr TStringBuf ZONE = "zone";
constexpr TStringBuf RECORD_TYPE = "record_type";
constexpr TStringBuf STATUS = "status";

// Label values
constexpr TStringBuf UNKNOWN_ZONE = "<unknown>";
constexpr TStringBuf STATUS_UNDEFINED = "UNDEFINED";
constexpr TStringBuf STATUS_OK = "OK";
constexpr TStringBuf STATUS_UNKNOWN_ZONE = "UNKNOWN_ZONE";
constexpr TStringBuf STATUS_NOT_READY = "NOT_READY";
constexpr TStringBuf STATUS_ERROR = "ERROR";

// Sensors
constexpr TStringBuf REQUESTS = "requests";
constexpr TStringBuf EMPTY = "empty";

constexpr TStringBuf LOOKUP = "lookup";
constexpr TStringBuf LIST_ZONE_RECORD_SETS = "list_zone_record_sets";
constexpr TStringBuf LIST_ZONE_DATA = "list_zone_data";
constexpr TStringBuf LIST_DYNAMIC_ZONES = "list_dynamic_zones";

constexpr TStringBuf READY = "ready";

constexpr TStringBuf START_FAILED = "start.failed";

constexpr TStringBuf PING_RESPONSE_TIME = "ping.response_time";
constexpr TStringBuf SHUTDOWN_RESPONSE_TIME = "shutdown.response_time";
constexpr TStringBuf SENSORS_RESPONSE_TIME = "sensors.response_time";
constexpr TStringBuf SENSORS_JSON_RESPONSE_TIME = "sensors_json.response_time";
constexpr TStringBuf REOPEN_LOG_RESPONSE_TIME = "reopen_log.response_time";
constexpr TStringBuf LOOKUP_RESPONSE_TIME = "lookup.response_time";
constexpr TStringBuf LIST_ZONE_RECORD_SETS_RESPONSE_TIME = "list_zone_record_sets.response_time";
constexpr TStringBuf LIST_ZONE_DATA_RESPONSE_TIME = "list_zone_data.response_time";
constexpr TStringBuf LIST_DYNAMIC_ZONES_RESPONSE_TIME = "list_dynamic_zones.response_time";
constexpr std::initializer_list<std::tuple<TStringBuf, double, double>> HISTOGRAMS_INIT_PARAMETERS_YP_DNS = {
    {PING_RESPONSE_TIME, 1.5, 10.0},
    {SHUTDOWN_RESPONSE_TIME, 1.5, 10.0},
    {SENSORS_RESPONSE_TIME, 1.5, 10.0},
    {SENSORS_JSON_RESPONSE_TIME, 1.5, 10.0},
    {REOPEN_LOG_RESPONSE_TIME, 1.5, 10.0},
    {LOOKUP_RESPONSE_TIME, 1.5, 10.0},
    {LIST_ZONE_RECORD_SETS_RESPONSE_TIME, 1.5, 10.0},
    {LIST_ZONE_DATA_RESPONSE_TIME, 1.5, 10.0},
    {LIST_DYNAMIC_ZONES_RESPONSE_TIME, 1.5, 10.0},
};

} // namespace NSensors

namespace {

constexpr ui64 DNS_RECORD_SETS_STORAGE_FORMAT_VERSION = 8;
constexpr ui64 DNS_ZONES_STORAGE_FORMAT_VERSION = 2;

template <typename TReq>
TStringBuf GetYpClusterName(NInfra::TRequestPtr<TReq> request) {
    TStringBuf path = request->Path();
    path.SkipPrefix("/");
    return path.SplitOff('/').NextTok('/');
}

bool Equals(const google::protobuf::Message& lhs, const google::protobuf::Message& rhs) {
    return google::protobuf::util::MessageDifferencer::Equivalent(lhs, rhs);
}

template <typename TContainer, typename TRandGen>
void ShuffleRepeatedProto(TContainer& values, TRandGen&& gen) {
    for (size_t i = 1; i < static_cast<size_t>(values.size()); ++i) {
        values[i].Swap(&values[gen.Uniform(i + 1)]);
    }
}

NApi::TRecord MakeResultRecord(const TDnsRecordSetSpec::TResourceRecord& record) {
    NApi::TRecord result;
    result.set_ttl(record.ttl());
    result.set_class_(record.class_());
    result.set_type(TDnsRecordSetSpec::TResourceRecord::EType_Name(record.type()));
    result.set_data(record.data());
    return result;
}

NApi::TRecordSet MakeResultRecordSet(const NYpDns::TRecordSet& recordSet) {
    NApi::TRecordSet result;
    result.set_id(recordSet.Meta().id());
    result.mutable_records()->Reserve(recordSet.Spec().records().size());
    for (const auto& record : recordSet.Spec().records()) {
        *result.add_records() = MakeResultRecord(record);
    }
    return result;
}

} // anonymous namespace

struct TModifyReplicaElementCallback : public NYPReplica::IModifyReplicaElementCallback<NYPReplica::TDnsRecordSetReplicaObject> {
    TModifyReplicaElementCallback(TDnsService* service)
        : Service_(service)
    {
    }

    void Do(const TMaybe<NYPReplica::TDnsRecordSetReplicaObject>& /* oldValue */, NYPReplica::TDnsRecordSetReplicaObject& newValue, NYPReplica::EObjectType newValueType) override {
        if (newValueType == NYPReplica::EObjectType::DELETE) {
            return;
        }

        const DNSName domain(newValue.GetKey());
        TAtomicSharedPtr<const NYpDns::TZoneConfig> zone = Service_->DetermineZone(domain, nullptr);

        if (!zone) {
            return;
        }

        NClient::TDnsRecordSet recordSet = newValue.GetObject();
        auto& records = *recordSet.MutableSpec()->mutable_records();

        bool modify = false;

        const NYpDns::TFilterStoredRecordsPolicy& filterPolicy = zone->GetFilterStoredRecordsPolicy();
        switch (filterPolicy.GetOrder()) {
            case NYpDns::TFilterStoredRecordsPolicy::RANDOM: {
                const TString hashStr = TString::Join(HostName(), '#', newValue.GetKey());

                ShuffleRepeatedProto(records, TReallyFastRng32(MurmurHash<ui32>(hashStr.data(), hashStr.size())));
                modify = true;
                break;
            }
            case NYpDns::TFilterStoredRecordsPolicy::ORIGINAL:
            default:
                break;
        }

        if (i32 maxRecordsNumber = filterPolicy.GetMaxRecordsNumber(); maxRecordsNumber != -1 && maxRecordsNumber < records.size()) {
            records.Truncate(filterPolicy.GetMaxRecordsNumber());
            modify = true;
        }

        if (modify) {
            newValue = NYPReplica::TDnsRecordSetReplicaObject(recordSet);
        }
    }

private:
    TDnsService* Service_;
};

TDnsService::TDnsService(const TConfig& config)
    : ConfigHolder_(config, config.GetWatchPatchConfig(), config.GetConfigUpdatesLoggerConfig())
    , Config_(ConfigHolder_.Accessor())
    , Logger_(CONFIG_SNAPSHOT_VALUE(Config_, GetLoggerConfig()))
    , BackupLogger_(CONFIG_SNAPSHOT_VALUE(Config_, GetBackupLoggerConfig()))
    , SensorGroup_(NSensors::NAMESPACE)
    , HttpService_(CONFIG_SNAPSHOT_VALUE(Config_, GetHttpServiceConfig()), CreateRouter(*this, *Config_))
    , RandomGenerator_(Seed())
    , ReplicasManagementPool_(new TThreadPool())
{
    const TConfigPtr initialConfig = Config_.Get();
    if (initialConfig->GetMemoryLock().GetLockSelfMemory()) {
        LockSelfMemory();
    }

    ConfigHolder_.SetSwitchConfigsCallback([this](const TConfig& oldConfig, const TConfig& newConfig) {
        SwitchConfigsCallback(oldConfig, newConfig);
    });

    InitReplicas();
    InitZones(); // must be after InitReplicas
    InitSensors();
    InitListZonePool();

    NInfra::TIntGaugeSensor(SensorGroup_, NSensors::READY).Set(0);
    NInfra::TIntGaugeSensor(SensorGroup_, NSensors::START_FAILED).Set(0);
}


NThreading::TFuture<void> TDnsService::Start() {
    NInfra::TLogFramePtr startLogFrame = Logger_.SpawnFrame();
    NThreading::TFuture<void> startFuture;
    try {
        startLogFrame->LogEvent(NUnbound::NEventlog::TServiceStart());

        ConfigHolder_.StartWatchPatch();
        StartReplicas(startLogFrame);
        HttpService_.Start(startLogFrame);

        startFuture = InitReplicasFuture_;
    } catch (...) {
        startFuture = NThreading::MakeErrorFuture<void>(std::current_exception());
    }

    return startFuture.Subscribe([this, startLogFrame](const NThreading::TFuture<void>& f) {
        try {
            f.TryRethrow();
            startLogFrame->LogEvent(NUnbound::NEventlog::TServiceStartSuccess());
        } catch (...) {
            startLogFrame->LogEvent(TLOG_ERR, NUnbound::NEventlog::TServiceStartFailure(CurrentExceptionMessage()));
            NInfra::TIntGaugeSensor(SensorGroup_, NSensors::START_FAILED).Set(1);
        }
    });
}

void TDnsService::Stop() {
    auto frame = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        NUnbound::NEventlog::TServiceRequestStart(NUnbound::NEventlog::EServiceRequestType::SHUTDOWN),
        NUnbound::NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::SHUTDOWN_RESPONSE_TIME)
    );

    NInfra::TIntGaugeSensor(SensorGroup_, NSensors::READY).Set(0);

    StopReplicas();
    HttpService_.Stop();
}

bool TDnsService::IsReady() const {
    if (ReplicasInited_.load()) {
        return true;
    }
    const bool ready = InitReplicasFuture_.HasValue();
    if (ready) {
        ReplicasInited_.store(ready);
    }
    return ready;
}

void TDnsService::InitReplicas() {
    const TConfigPtr config = Config_.Get();
    const TString pathToYpTokenFile = config->GetYpTokenFilePath();
    const TString ypToken = pathToYpTokenFile ? NClient::ReadTokenFromFile(pathToYpTokenFile) : NClient::FindToken();

    if (config->GetDynamicZonesConfig().GetEnabled()) {
        DnsZonesReplicaLogger_ = MakeHolder<NInfra::TLogger>(
            config->GetDynamicZonesConfig().GetReplicaLoggerConfig());

        DnsZonesReplica_ = MakeHolder<NYpDns::NDynamicZones::TDnsZonesReplica>(
            Config_.Accessor<NYP::NYPReplica::TYPReplicaConfig>("DynamicZonesConfig/YPReplicaConfig"),
            Config_.Accessor<NYP::NYPReplica::TYPClusterConfig>("DynamicZonesConfig/YPClusterConfig"),
            NYP::NYPReplica::TTableRulesHolder<NYpDns::NDynamicZones::TDnsZoneReplicaObject>(),
            ypToken,
            *DnsZonesReplicaLogger_,
            DNS_ZONES_STORAGE_FORMAT_VERSION);
    }

    for (const auto& cluster : config->GetYPClusterConfigs()) {
        auto& replica = Replicas_.emplace(cluster.GetName(), MakeHolder<TYPReplica>(
            Config_.Accessor<NYPReplica::TYPReplicaConfig>("YPReplicaConfig"),
            Config_.Accessor<NYPReplica::TYPClusterConfig>({"YPClusterConfigs", cluster.GetName()}),
            NYP::NYPReplica::TTableRulesHolder<NYP::NYPReplica::TDnsRecordSetReplicaObject>(),
            ypToken,
            BackupLogger_,
            DNS_RECORD_SETS_STORAGE_FORMAT_VERSION
        )).first->second;

        replica->SetModifyReplicaElementCallback<NYP::NYPReplica::TDnsRecordSetReplicaObject>(MakeHolder<TModifyReplicaElementCallback>(this));
    }

    ui32 replicasManagementPoolSize = 0;
    const ui32 defaultPoolSize =
        config->GetYPClusterConfigs().size() +
        config->GetDynamicZonesConfig().GetEnabled();
    if (!config->HasReplicasManagementThreadPoolSize()) {
        replicasManagementPoolSize = defaultPoolSize;
    } else {
        replicasManagementPoolSize = Min(
            config->GetReplicasManagementThreadPoolSize(),
            2 * defaultPoolSize  // x2 – for start and stop
        );
    }
    ReplicasManagementPool_->Start(Max(1u, replicasManagementPoolSize));
}

void TDnsService::StartReplicas(NInfra::TLogFramePtr logFrame) {
    TVector<NThreading::TFuture<void>> replicasStartFutures;
    replicasStartFutures.reserve(Replicas_.size() + !!DnsZonesReplica_);


    if (DnsZonesReplica_) {
        replicasStartFutures.push_back(NThreading::Async(
            [this] {
                DnsZonesReplica_->Start();
            },
            *ReplicasManagementPool_
        ));
    }

    for (auto& [cluster, replica] : Replicas_) {
        replicasStartFutures.push_back(NThreading::Async(
            [cluster = cluster, replicaPtr = replica.Get(), logFrame] {
                try {
                    logFrame->LogEvent(NUnbound::NEventlog::TReplicaStart(cluster));
                    replicaPtr->Start();
                    logFrame->LogEvent(NUnbound::NEventlog::TReplicaStartSuccess(cluster));
                } catch (...) {
                    logFrame->LogEvent(TLOG_ERR,
                        NUnbound::NEventlog::TReplicaStartFailure(cluster, CurrentExceptionMessage()));
                    throw;
                }
            },
            *ReplicasManagementPool_
        ));
    }

    InitReplicasFuture_ = NThreading::WaitAll(replicasStartFutures).Subscribe(
        [this, logFrame](const NThreading::TFuture<void>& f) {
            try {
                f.TryRethrow();
                NInfra::TIntGaugeSensor(SensorGroup_, NSensors::READY).Set(1);
            } catch (...) {
                logFrame->LogEvent(TLOG_ERR,
                    NUnbound::NEventlog::TReplicasStartFailure(CurrentExceptionMessage()));
            }
        }
    );
}

void TDnsService::StopReplicas() {
    // We can't wait for thread pool if it's full,
    // so we use TAsyncTaskBatch to call Stop()s at least in this thread.
    TAsyncTaskBatch tasks(ReplicasManagementPool_.Get());

    if (DnsZonesReplica_) {
        tasks.Add([this] {
            DnsZonesReplica_->Stop();
        });
    }

    for (auto& [cluster, replica] : Replicas_) {
        tasks.Add([replicaPtr = replica.Get()] {
            replicaPtr->Stop();
        });
    }

    tasks.WaitAllAndProcessNotStarted();
}

void TDnsService::InitSensors() {
    // init histograms
    for (const auto& [histogramName, baseOfHistogram, scaleOfHistogram] : NSensors::HISTOGRAMS_INIT_PARAMETERS_YP_DNS) {
        if (!HistogramSensors_.contains(histogramName)) {
            HistogramSensors_.emplace(histogramName, MakeIntrusive<NInfra::THistogramRateSensor>(
                SensorGroup_, histogramName, NMonitoring::ExponentialHistogram(NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT, baseOfHistogram, scaleOfHistogram)));
        }
    }

    // init sensors for listing dynamic zones
    {
        constexpr std::array<TStringBuf, 4> listDynamicZonesStatuses = {
            NSensors::STATUS_UNDEFINED,
            NSensors::STATUS_OK,
            NSensors::STATUS_ERROR,
            NSensors::STATUS_NOT_READY,
        };
        NInfra::TSensorGroup sensorGroup{SensorGroup_, NSensors::LIST_DYNAMIC_ZONES};
        NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS};
        for (const TStringBuf statusName : listDynamicZonesStatuses) {
            NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS, {{NSensors::STATUS, statusName}}};
        }
    }
}

void TDnsService::InitListZonePool() {
    const TConfigPtr config = Config_.Get();

    size_t threadsCount;
    if (config->GetListZonesConfig().HasMaxThreads()) {
        threadsCount = config->GetListZonesConfig().GetMaxThreads();
    } else {
        threadsCount = 1;
        for (const NYpDns::TZoneConfig& zone : config->GetZones()) {
            if (zone.GetSelectRecordSetMode() == NYpDns::ESelectRecordSetMode::MERGE) {
                threadsCount = Max(threadsCount, static_cast<size_t>(zone.GetYPClusters().size()));
            }
        }
    }
    ListZonePool_ = CreateThreadPool(threadsCount);
}

void TDnsService::InitZones() {
    const TConfigPtr config = Config_.Get();

    Zones_.reserve(config->GetZones().size());
    ReplicasByZone_.reserve(config->GetZones().size());
    for (const NYpDns::TZoneConfig& zone : config->GetZones()) {
        Zones_.emplace(DNSName(zone.GetName()), Config_.Accessor<NYpDns::TZoneConfig>({"Zones", zone.GetName()}));

        auto& replicas = ReplicasByZone_[zone.GetName()];
        replicas.reserve(zone.GetYPClusters().size());
        for (const TString& cluster : zone.GetYPClusters()) {
            if (const auto* replicaHolderPtr = Replicas_.FindPtr(cluster); replicaHolderPtr && replicaHolderPtr->Get()) {
                replicas.emplace_back(cluster, replicaHolderPtr->Get());
            }
        }
    }
}

void TDnsService::SwitchConfigsCallback(const TConfig& oldConfig, const TConfig& newConfig) {
    for (int i = 0; i < oldConfig.GetYPClusterConfigs().size(); ++i) {
        Y_ENSURE(oldConfig.GetYPClusterConfigs(i).GetName() == newConfig.GetYPClusterConfigs(i).GetName());

        const auto& oldClusterConfig = oldConfig.GetYPClusterConfigs(i);
        const auto& newClusterConfig = newConfig.GetYPClusterConfigs(i);
        if (!Equals(oldClusterConfig.GetRollbackToBackup(), newClusterConfig.GetRollbackToBackup())) {
            if (newClusterConfig.GetRollbackToBackup().GetRollback()) {
                NApi::TReqRollbackToBackup request;
                *request.MutableRollbackConfig() = newClusterConfig.GetRollbackToBackup();
                NApi::TRspRollbackToBackup response;
                RollbackToBackup(newClusterConfig.GetName(), request, response);
            }
        }

        if (oldClusterConfig.GetEnableUpdates() != newClusterConfig.GetEnableUpdates()) {
            if (newClusterConfig.GetEnableUpdates()) {
                NApi::TRspStartUpdates response;
                StartUpdates(newClusterConfig.GetName(), NApi::TReqStartUpdates{}, response);
            } else {
                NApi::TRspStopUpdates response;
                StopUpdates(newClusterConfig.GetName(), NApi::TReqStopUpdates{}, response);
            }
        }
    }
}

void TDnsService::Ping(NInfra::TRequestPtr<NApi::TReqPing>, NInfra::TReplyPtr<NApi::TRspPing> reply) {
    auto frame = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        NUnbound::NEventlog::TServiceRequestStart(NUnbound::NEventlog::EServiceRequestType::PING),
        NUnbound::NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::PING_RESPONSE_TIME)
    );

    NApi::TRspPing result;
    result.set_data("ok");
    reply->Set(result);
}

void TDnsService::ReopenLog(NInfra::TRequestPtr<NApi::TReqReopenLog>, NInfra::TReplyPtr<NApi::TRspReopenLog> reply) {
    auto frame = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        NUnbound::NEventlog::TServiceRequestStart(NUnbound::NEventlog::EServiceRequestType::REOPEN_LOG),
        NUnbound::NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::REOPEN_LOG_RESPONSE_TIME)
    );

    NApi::TRspReopenLog result;
    result.set_data("ok");
    reply->Set(result);

    Logger_.ReopenLog();
    BackupLogger_.ReopenLog();
    ConfigHolder_.ReopenLog();
}

void TDnsService::PrintSensors(TStringOutput& output, NInfra::ESensorsSerializationType serializationType) const {
    if (DnsZonesReplica_) {
        DnsZonesReplica_->UpdateSensors();
    }

    for (const auto& [name, replica] : Replicas_) {
        Y_ENSURE(replica);
        replica->UpdateSensors();
    }

    NInfra::SensorRegistryPrint(output, serializationType);
}

void TDnsService::Sensors(NInfra::TRequestPtr<NApi::TReqSensors>, NInfra::TReplyPtr<NApi::TRspSensors> reply) {
    auto frame = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        NUnbound::NEventlog::TServiceRequestStart(NUnbound::NEventlog::EServiceRequestType::SENSORS),
        NUnbound::NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::SENSORS_RESPONSE_TIME)
    );

    NApi::TRspSensors result;
    TStringOutput output(*result.MutableData());
    PrintSensors(output, NInfra::ESensorsSerializationType::SPACK_V1);
    reply->SetAttribute("Content-Type", "application/x-solomon-spack");

    reply->Set(result);
}

void TDnsService::SensorsJson(NInfra::TRequestPtr<NApi::TReqSensors>, NInfra::TReplyPtr<NApi::TRspSensors> reply) {
    auto frame = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        NUnbound::NEventlog::TServiceRequestStart(NUnbound::NEventlog::EServiceRequestType::SENSORS_JSON),
        NUnbound::NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::SENSORS_JSON_RESPONSE_TIME)
    );

    NApi::TRspSensors result;
    TStringOutput output(*result.MutableData());
    PrintSensors(output, NInfra::ESensorsSerializationType::JSON);

    reply->Set(result);
}

void TDnsService::Config(NInfra::TRequestPtr<NApi::TReqConfig>, NInfra::TReplyPtr<NApi::TRspConfig> reply) {
    NApi::TRspConfig result;
    TStringOutput stream(*result.MutableData());
    NJson::TJsonWriter writer(&stream, /* formatOutput */ true);
    NProtobufJson::Proto2Json(*Config_, writer);

    reply->Set(result);
}

void TDnsService::ListBackups(NInfra::TRequestPtr<NApi::TReqListBackups> request, NInfra::TReplyPtr<NApi::TRspListBackups> reply) {
    const TStringBuf clusterName = GetYpClusterName(request);

    NApi::TRspListBackups result;
    if (const auto* replicaPtr = Replicas_.FindPtr(clusterName); replicaPtr && replicaPtr->Get()) {
        const TVector<NYPReplica::TBackupInfo> infos = replicaPtr->Get()->ListBackups();
        for (const NYPReplica::TBackupInfo& info : infos) {
            NApi::TBackupInfo& backupInfo = *result.AddBackupInfos();
            backupInfo.SetId(info.Id);
            backupInfo.SetTimestamp(info.Timestamp);
            backupInfo.SetSize(info.Size);
            backupInfo.SetNumberFiles(info.NumberFiles);
            *backupInfo.MutableMeta() = info.Meta;
        }
    }
    reply->Set(result);
}

void TDnsService::RollbackToBackup(const TStringBuf clusterName, const NApi::TReqRollbackToBackup& request, NApi::TRspRollbackToBackup& response) {
    auto* replicaPtr = Replicas_.FindPtr(clusterName);

    if (!replicaPtr || !replicaPtr->Get()) {
        response.SetStatus(TStringBuilder() << "replica " << clusterName << " not found");
        return;
    }

    try {
        NYPReplica::TRollbackToBackupConfig rollbackConfig = request.GetRollbackConfig();
        rollbackConfig.SetRollback(true);

        if (replicaPtr->Get()->RollbackToBackup(rollbackConfig)) {
            response.SetStatus("ok");
        } else {
            response.SetStatus("failed");
        }
    } catch (...) {
        response.SetStatus(TStringBuilder() << "failed: " << CurrentExceptionMessage());
    }
}

void TDnsService::RollbackToBackup(NInfra::TRequestPtr<NApi::TReqRollbackToBackup> request, NInfra::TReplyPtr<NApi::TRspRollbackToBackup> reply) {
    NApi::TRspRollbackToBackup response;
    RollbackToBackup(GetYpClusterName(request), request->Get(), response);
    reply->Set(response);
}

void TDnsService::StartUpdates(const TStringBuf clusterName, const NApi::TReqStartUpdates&, NApi::TRspStartUpdates& response) {
    auto* replicaPtr = Replicas_.FindPtr(clusterName);

    if (!replicaPtr || !replicaPtr->Get()) {
        response.SetData(TStringBuilder() << "replica " << clusterName << " not found");
        return;
    }

    replicaPtr->Get()->EnableUpdates();
    response.SetData("ok");
}

void TDnsService::StartUpdates(NInfra::TRequestPtr<NApi::TReqStartUpdates> request, NInfra::TReplyPtr<NApi::TRspStartUpdates> reply) {
    NApi::TRspStartUpdates response;
    StartUpdates(GetYpClusterName(request), request->Get(), response);
    reply->Set(response);
}

void TDnsService::StopUpdates(const TStringBuf clusterName, const NApi::TReqStopUpdates&, NApi::TRspStopUpdates& response) {
    auto* replicaPtr = Replicas_.FindPtr(clusterName);

    if (!replicaPtr || !replicaPtr->Get()) {
        response.SetData(TStringBuilder() << "replica " << clusterName << " not found");
        return;
    }

    replicaPtr->Get()->DisableUpdates();
    response.SetData("ok");
}

void TDnsService::StopUpdates(NInfra::TRequestPtr<NApi::TReqStopUpdates> request, NInfra::TReplyPtr<NApi::TRspStopUpdates> reply) {
    NApi::TRspStopUpdates response;
    StopUpdates(GetYpClusterName(request), request->Get(), response);
    reply->Set(response);
}

void TDnsService::LockSelfMemory() {
    try {
        LockAllMemory(LockCurrentMemory);
    } catch (const yexception& e) {
        NInfra::NLogEvent::TServiceMemoryLockError ev("Failed to lock memory: " + CurrentExceptionMessage());
        Logger_.SpawnFrame()->LogEvent(ELogPriority::TLOG_ERR, ev);
    }
}

void TDnsService::ListZoneRecordSets(NInfra::TRequestPtr<NApi::TReqListZoneRecordSets> request, NInfra::TReplyPtr<NApi::TRspListZoneRecordSets> reply) {
    auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame<ELogPriority::TLOG_INFO>(
        Logger_,
        NUnbound::NEventlog::TServiceRequestStart(NUnbound::NEventlog::EServiceRequestType::LIST_ZONE_RECORD_SETS),
        NUnbound::NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::LIST_ZONE_RECORD_SETS_RESPONSE_TIME)
    );

    const NApi::TReqListZoneRecordSets& req = request->Get();

    logFrame->LogEvent(ELogPriority::TLOG_INFO, NUnbound::NEventlog::TListZoneRecordSetsRequestData(req.zone(), req.limit(), req.continuation_token()));
    NInfra::TSensorGroup sensorGroup{SensorGroup_, NSensors::LIST_ZONE_RECORD_SETS};
    sensorGroup.AddLabel(NSensors::ZONE, to_lower(req.zone()));
    NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

    NApi::TRspListZoneRecordSets rsp;

    TAtomicSharedPtr<const NYpDns::TZoneConfig> zoneConfig = DetermineZone(DNSName(req.zone()), logFrame);
    if (!zoneConfig) {
        logFrame->LogEvent(ELogPriority::TLOG_WARNING, NUnbound::NEventlog::TUnknownZone());

        rsp.set_status(NApi::TRspListZoneRecordSets::UNKNOWN_ZONE);

        NUnbound::NEventlog::TListZoneRecordSetsResult listZoneResultEvent;
        listZoneResultEvent.SetStatus(NApi::TRspListZoneRecordSets::EListZoneRecordSetsStatus_Name(rsp.status()));
        logFrame->LogEvent(ELogPriority::TLOG_INFO, std::move(listZoneResultEvent));

        NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS, {{NSensors::STATUS, NSensors::STATUS_UNKNOWN_ZONE}}}.Inc();

        reply->Set(rsp);
        return;
    }

    if (!IsReady()) {
        rsp.set_status(NApi::TRspListZoneRecordSets::NOT_READY);

        NUnbound::NEventlog::TListZoneRecordSetsResult listZoneResultEvent;
        listZoneResultEvent.SetStatus(NApi::TRspListZoneRecordSets::EListZoneRecordSetsStatus_Name(rsp.status()));
        logFrame->LogEvent(ELogPriority::TLOG_INFO, std::move(listZoneResultEvent));

        NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS, {{NSensors::STATUS, NSensors::STATUS_NOT_READY}}}.Inc();

        reply->Set(rsp);
        return;
    }

    const NYpDns::TZone zone(*zoneConfig);

    NYpDns::TListMulticlusterZoneOptions listOptions;
    listOptions.Limit = req.limit();
    listOptions.BuildSOARecordFromConfigIfNotFound = true;
    listOptions.BuildNSRecordsFromConfigIfNotFound = true;

    if (req.continuation_token()) {
        listOptions.SeekType = NYpDns::ESeekType::Next;
        try {
            listOptions.SeekKey = Base64Decode(req.continuation_token());
        } catch (...) {
            rsp.set_status(NApi::TRspListZoneRecordSets::INVALID_CONTINUATION_TOKEN);
            reply->Set(rsp);
            return;
        }
    } else {
        listOptions.SeekType = NYpDns::ESeekType::ToFirst;
    }

    const TYPReplicasList& zoneReplicas = GetYpReplicasForZone(zone.Config());
    NYpDns::TListRecordSetsResult listResult = NYpDns::ListMulticlusterZone(
        zone,
        zoneReplicas,
        listOptions,
        *ListZonePool_,
        logFrame,
        sensorGroup
    );

    rsp.set_status(NApi::TRspListZoneRecordSets::OK);
    for (const auto& [cluster, ypTimestamp] : listResult.YpTimestamps) {
        (*rsp.mutable_yp_timestamps())[cluster] = ypTimestamp;
    }
    rsp.mutable_record_sets()->Reserve(listResult.RecordSets.size());
    for (const NYpDns::TRecordSet& recordSetObject : listResult.RecordSets) {
        *rsp.add_record_sets() = MakeResultRecordSet(recordSetObject);
    }
    if (!listResult.RecordSets.empty()) {
        rsp.set_continuation_token(Base64Encode(listResult.RecordSets.back().Meta().id()));
    }

    NUnbound::NEventlog::TListZoneRecordSetsResult listZoneResultEvent;
    listZoneResultEvent.SetStatus(NApi::TRspListZoneRecordSets::EListZoneRecordSetsStatus_Name(rsp.status()));
    listZoneResultEvent.SetRecordSetsNumber(rsp.record_sets().size());
    listZoneResultEvent.SetContinuationToken(rsp.continuation_token());
    *listZoneResultEvent.MutableYpTimestamps() = rsp.yp_timestamps();
    logFrame->LogEvent(ELogPriority::TLOG_INFO, std::move(listZoneResultEvent));

    NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS, {{NSensors::STATUS, NSensors::STATUS_OK}}}.Inc();

    reply->Set(rsp);
}

void TDnsService::ListZoneData(NInfra::TRequestPtr<NApi::TReqListZoneData> request, NInfra::TReplyPtr<NApi::TRspListZoneData> reply) {
    reply->Set(ListZoneData(request->Get()));
}

void TDnsService::ListZoneDataRaw(NInfra::TRequestPtr<NApi::TReqListZoneData> request, NInfra::TReplyPtr<NApi::TRspListZoneData> reply) {
    NApi::TRspListZoneData rsp = ListZoneData(request->Get());
    if (rsp.status() != NApi::TRspListZoneData::OK) {
        throw yexception() << "Error: " << NApi::TRspListZoneData::EListZoneDataStatus_Name(rsp.status());
    }
    reply->Set(rsp);
}

NApi::TRspListZoneData TDnsService::ListZoneData(const NApi::TReqListZoneData& request) {
    auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        NUnbound::NEventlog::TServiceRequestStart(NUnbound::NEventlog::EServiceRequestType::LIST_ZONE_DATA),
        NUnbound::NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::LIST_ZONE_DATA_RESPONSE_TIME)
    );

    logFrame->LogEvent(ELogPriority::TLOG_INFO, NUnbound::NEventlog::TListZoneDataRequestData(request.zone()));
    NInfra::TSensorGroup sensorGroup{SensorGroup_, NSensors::LIST_ZONE_DATA};
    sensorGroup.AddLabel(NSensors::ZONE, to_lower(request.zone()));
    NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

    NApi::TRspListZoneData response;

    TAtomicSharedPtr<const NYpDns::TZoneConfig> zoneConfig = DetermineZone(DNSName(request.zone()), logFrame);
    if (!zoneConfig) {
        logFrame->LogEvent(ELogPriority::TLOG_WARNING, NUnbound::NEventlog::TUnknownZone());

        response.set_status(NApi::TRspListZoneData::UNKNOWN_ZONE);

        NUnbound::NEventlog::TListZoneDataResult listZoneResultEvent;
        listZoneResultEvent.SetStatus(NApi::TRspListZoneData::EListZoneDataStatus_Name(response.status()));
        logFrame->LogEvent(ELogPriority::TLOG_INFO, std::move(listZoneResultEvent));

        NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS, {{NSensors::STATUS, NSensors::STATUS_UNKNOWN_ZONE}}}.Inc();

        return response;
    }

    if (!IsReady()) {
        response.set_status(NApi::TRspListZoneData::NOT_READY);

        NUnbound::NEventlog::TListZoneDataResult listZoneResultEvent;
        listZoneResultEvent.SetStatus(NApi::TRspListZoneData::EListZoneDataStatus_Name(response.status()));
        logFrame->LogEvent(ELogPriority::TLOG_INFO, std::move(listZoneResultEvent));

        NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS, {{NSensors::STATUS, NSensors::STATUS_NOT_READY}}}.Inc();

        return response;
    }

    const NYpDns::TZone zone(*zoneConfig);

    NYpDns::TListMulticlusterZoneOptions listOptions;
    listOptions.BuildSOARecordFromConfigIfNotFound = true;
    listOptions.BuildNSRecordsFromConfigIfNotFound = true;

    const TYPReplicasList& zoneReplicas = GetYpReplicasForZone(zone.Config());
    NYpDns::TListRecordSetsResult listResult = NYpDns::ListMulticlusterZone(
        zone,
        zoneReplicas,
        std::move(listOptions),
        *ListZonePool_,
        logFrame,
        sensorGroup
    );

    response.set_status(NApi::TRspListZoneData::OK);

    TStringOutput output(*response.mutable_data());
    NYpDns::TZoneWriterOptions zoneWriterOptions;
    zoneWriterOptions.SetDefaultTtl = false;
    zoneWriterOptions.ZoneConfig = zone.Config();
    NYpDns::TZoneWriter zoneWriter(output, zoneWriterOptions);
    for (const NYpDns::TRecordSet& recordSet : listResult.RecordSets) {
        zoneWriter << recordSet;
    }
    zoneWriter.Finish();

    NUnbound::NEventlog::TListZoneDataResult listZoneResultEvent;
    listZoneResultEvent.SetStatus(NApi::TRspListZoneData::EListZoneDataStatus_Name(response.status()));
    listZoneResultEvent.SetRecordSetsNumber(listResult.RecordSets.size());
    for (const auto& [cluster, ypTimestamp] : listResult.YpTimestamps) {
        (*listZoneResultEvent.MutableYpTimestamps())[cluster] = ypTimestamp;
    }
    logFrame->LogEvent(ELogPriority::TLOG_INFO, std::move(listZoneResultEvent));

    NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS, {{NSensors::STATUS, NSensors::STATUS_OK}}}.Inc();

    return response;
}

void TDnsService::ListDynamicZones(NInfra::TRequestPtr<NApi::TReqListZones> request, NInfra::TReplyPtr<NApi::TRspListZones> reply) {
    Y_UNUSED(request);

    auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame(
        Logger_,
        NUnbound::NEventlog::TServiceRequestStart(NUnbound::NEventlog::EServiceRequestType::LIST_DYNAMIC_ZONES),
        NUnbound::NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::LIST_DYNAMIC_ZONES_RESPONSE_TIME)
    );

    NInfra::TSensorGroup sensorGroup{SensorGroup_, NSensors::LIST_DYNAMIC_ZONES};
    NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

    NApi::TRspListZones rsp;
    if (!IsReady()) {
        rsp.set_status(NApi::TRspListZones::NOT_READY);
    } else {
        try {
            TVector<NYpDns::TZone> zones = ListDynamicZones();
            rsp.set_status(NApi::TRspListZones::OK);
            rsp.mutable_zones()->Reserve(zones.size());
            for (const NYpDns::TZone& zone : zones) {
                *rsp.add_zones()->mutable_config() = zone.Config();
            }
        } catch (...) {
            rsp.set_status(NApi::TRspListZones::ERROR);
            rsp.set_error_message(CurrentExceptionMessage());
        }
    }

    ListDynamicZonesLogResult(logFrame, sensorGroup, rsp);

    reply->Set(rsp);
}

void TDnsService::ListDynamicZonesLogResult(NInfra::TLogFramePtr logFrame, const NInfra::TSensorGroup& sensorGroup, const NApi::TRspListZones& rsp) const {
    for (const auto& zone : rsp.zones()) {
        logFrame->LogEvent(ELogPriority::TLOG_INFO, MakeListZonesResultDataEvent(zone.config()));
    }
    logFrame->LogEvent(ELogPriority::TLOG_INFO, MakeListDynamicZonesResultEvent(rsp));

    TStringBuf statusName = NSensors::STATUS_UNDEFINED;
    switch (rsp.status()) {
        case NApi::TRspListZones::OK:
            statusName = NSensors::STATUS_OK;
            break;
        case NApi::TRspListZones::ERROR:
            statusName = NSensors::STATUS_ERROR;
            break;
        case NApi::TRspListZones::NOT_READY:
            statusName = NSensors::STATUS_NOT_READY;
            break;
        default:
            break;
    }
    NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS, {{NSensors::STATUS, statusName}}}.Inc();
}

NInfra::TLogger* TDnsService::GetLogger() {
    return &Logger_;
}

TVector<NYpDns::TZone> TDnsService::ListDynamicZones() const {
    if (!DnsZonesReplica_) {
        ythrow yexception() << "Dynamic zones are not initialized";
    }

    NYpDns::NDynamicZones::TDnsZonesReplicaListOptions listOptions;
    listOptions.Filter = [](const TStringBuf, const TVector<NYpDns::NDynamicZones::TDnsZonesReplicaStorageElement>& elements) {
        Y_ENSURE(elements.size() == 1);
        return elements.front().ReplicaObject.GetObject().Labels()["yp_dns"]["enable"].GetBooleanRobust();
    };
    auto elements = DnsZonesReplica_->ListElements(listOptions);

    TVector<NYpDns::TZone> result;
    result.reserve(elements.size());

    for (const auto& [key, values] : elements) {
        Y_ENSURE(values.size() == 1);
        result.emplace_back(values.front().ReplicaObject.GetObject());
    }

    return result;
}

TDnsService::TYPReplicasList TDnsService::GetYpReplicasForZone(const NYpDns::TZoneConfig& zone) const {
    // Check whether list is ready (zone is static)
    if (const TYPReplicasList* replicas = ReplicasByZone_.FindPtr(zone.GetName())) {
        return *replicas;
    }

    // Otherwize generate list (zone is dynamic)
    TDnsService::TYPReplicasList result;
    result.reserve(zone.GetYPClusters().size());
    for (const TString& cluster : zone.GetYPClusters()) {
        if (const THolder<TYPReplica>* replicaPtr = Replicas_.FindPtr(cluster)) {
            Y_ASSERT(replicaPtr && replicaPtr->Get());
            result.emplace_back(cluster, replicaPtr->Get());
        }
    }
    return result;
}

TAtomicSharedPtr<const NYpDns::TZoneConfig> TDnsService::DetermineZone(DNSName domain, NInfra::TLogFramePtr logFrame) const {
    const TString domainString = TString{domain.makeLowerCase().toStringNoDot()};
    TStringBuf domainStringView(domainString);

    if (logFrame) {
        logFrame->LogEvent(ELogPriority::TLOG_DEBUG, NUnbound::NEventlog::TDetermineZone(domainString));
    }

    do {
        if (TAtomicSharedPtr<const NYpDns::TZoneConfig> zone = FindZone(domain, domainStringView, logFrame)) {
            if (logFrame) {
                logFrame->LogEvent(ELogPriority::TLOG_DEBUG,
                    NUnbound::NEventlog::TDetermineZoneResult(NUnbound::NEventlog::TDetermineZoneResult::FOUND, zone->GetName()));
            }
            return zone;
        }
        domainStringView = domainStringView.After('.');
    } while (domain.chopOff());

    if (logFrame) {
        logFrame->LogEvent(ELogPriority::TLOG_DEBUG, DetermineZoneNotFoundResultEvent());
    }

    return nullptr;
}

TAtomicSharedPtr<const NYpDns::TZoneConfig> TDnsService::FindDynamicZone(const TStringBuf domain) const {
    if (!DnsZonesReplica_) {
        return nullptr;
    }

    auto result = DnsZonesReplica_->GetByKey<NYpDns::NDynamicZones::TDnsZoneReplicaObject>(domain);

    if (!result.Defined()) {
        return nullptr;
    }

    // Make sure that we can set Y_ENSURE here
    Y_ASSERT(result->Objects.size() == 1);
    const NYP::NClient::TDnsZone& zoneObject = result->Objects.front();

    if (!zoneObject.Labels()["yp_dns"]["enable"].GetBooleanRobust()) {
        return nullptr;
    }

    NYpDns::TZone zone(zoneObject);
    return MakeAtomicShared<const NYpDns::TZoneConfig>(zone.Config());
}

TAtomicSharedPtr<const NYpDns::TZoneConfig> TDnsService::FindZone(
    const DNSName& domain,
    const TStringBuf domainView,
    NInfra::TLogFramePtr logFrame
) const {
    if (auto* zoneConfigAccessorPtr = Zones_.FindPtr(domain)) {
        if (logFrame) {
            logFrame->LogEvent(ELogPriority::TLOG_DEBUG,
                NUnbound::NEventlog::TFindZoneResult(NUnbound::NEventlog::TFindZoneResult::STATIC, zoneConfigAccessorPtr->Get()->GetName()));
        }
        return zoneConfigAccessorPtr->Get();
    } else if (auto dynamicZoneConfig = FindDynamicZone(domainView)) {
        if (logFrame) {
            logFrame->LogEvent(ELogPriority::TLOG_DEBUG,
                NUnbound::NEventlog::TFindZoneResult(NUnbound::NEventlog::TFindZoneResult::DYNAMIC, dynamicZoneConfig->GetName()));
        }
        return dynamicZoneConfig;
    }
    return nullptr;
}

TMaybe<ui32> TDnsService::GetSerial(const NYpDns::TZoneConfig& zone, const TYPReplicasList& replicas) const {
    switch (zone.GetSerialGenerateMode()) {
        case NYpDns::ESerialNumberGenerateMode::ORIGINAL:
            return Nothing();
        case NYpDns::ESerialNumberGenerateMode::YP_TIMESTAMP_BASED:
        default: {
            ui64 timestamp = 0;
            for (const auto& [cluster, replica] : replicas) {
                timestamp = Max(timestamp, replica->GetYpTimestamp().GetOrElse(0));
            }
            return NYpDns::MakeSerialFromYpTimestamp(timestamp);
        }
    }
}

TMap<NYpDns::ERecordType, TVector<NYpDns::TRecord>> TDnsService::Lookup(const TString& domain) {
    auto [logFrame, bio] = NInfra::CreateBiographedLoggerFrame<ELogPriority::TLOG_DEBUG>(
        Logger_,
        NUnbound::NEventlog::TServiceRequestStart(NUnbound::NEventlog::EServiceRequestType::LOOKUP),
        NUnbound::NEventlog::TServiceRequestStop(),
        HistogramSensors_.at(NSensors::LOOKUP_RESPONSE_TIME)
    );

    logFrame->LogEvent(ELogPriority::TLOG_DEBUG, NUnbound::NEventlog::TLookupRequestData(domain));

    TMap<NYpDns::ERecordType, TVector<NYpDns::TRecord>> result;

    const DNSName name(domain);

    NInfra::TSensorGroup sensorGroup{SensorGroup_, NSensors::LOOKUP};

    TAtomicSharedPtr<const NYpDns::TZoneConfig> zone = DetermineZone(name, logFrame);
    if (zone == nullptr) {
        logFrame->LogEvent(ELogPriority::TLOG_DEBUG, NUnbound::NEventlog::TUnknownZone());

        sensorGroup.AddLabel(NSensors::ZONE, NSensors::UNKNOWN_ZONE);
        NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();
        return result;
    }

    logFrame->LogEvent(ELogPriority::TLOG_DEBUG, MakeZoneInfoEvent(*zone));

    sensorGroup.AddLabel(NSensors::ZONE, zone->GetName());
    NInfra::TRateSensor{sensorGroup, NSensors::REQUESTS}.Inc();

    TVector<TString> keysToLookup = {name.makeLowerCase().toStringNoDot().data()};

    if (zone->GetSupportPTR()) {
        keysToLookup.emplace_back(name.makeLowerCase().toString());
    }

    if (zone->GetSupportSRV()) {
        keysToLookup.emplace_back((DNSName("_id_") + name).makeLowerCase().toStringNoDot());
    }

    const TYPReplicasList& zoneReplicas = GetYpReplicasForZone(*zone);
    TVector<NClient::TDnsRecordSet> selectionResult;
    for (const TString& key : keysToLookup) {
        TMaybe<NClient::TDnsRecordSet> keySelectionResult;

        switch (zone->GetSelectRecordSetMode()) {
            case NYpDns::ESelectRecordSetMode::LARGEST_TIMESTAMP: {
                ui64 maxTimestamp = 0;
                for (const auto& [cluster, replica] : zoneReplicas) {
                    auto replicaSelectionResult = replica->GetByKey<NYPReplica::TDnsRecordSetReplicaObject>(key);
                    if (replicaSelectionResult.Defined() && !replicaSelectionResult->Objects.empty() && maxTimestamp < replicaSelectionResult->YpTimestamp) {
                        maxTimestamp = replicaSelectionResult->YpTimestamp;
                        keySelectionResult = std::move(replicaSelectionResult->Objects.front());
                    }
                }
                break;
            }
            case NYpDns::ESelectRecordSetMode::MERGE: {
                THashMap<TString, TMaybe<NYpDns::TRecordSet>> recordSetReplicas(zoneReplicas.size());
                for (const auto& [cluster, replica] : zoneReplicas) {
                    auto replicaSelectionResult = replica->GetByKey<NYPReplica::TDnsRecordSetReplicaObject>(key);
                    if (replicaSelectionResult.Defined() && !replicaSelectionResult->Objects.empty()) {
                        recordSetReplicas.emplace(cluster, replicaSelectionResult->Objects.front());
                    } else {
                        recordSetReplicas.emplace(cluster, Nothing());
                    }
                }
                TMaybe<NYpDns::TRecordSet> mergedRecordSet = NYpDns::MergeInOne(
                    recordSetReplicas,
                    NYpDns::TMergeOptions()
                        .SetFormChangelist(false)
                        .SetMergeAcls(false)
                );
                if (mergedRecordSet.Defined()) {
                    keySelectionResult = mergedRecordSet->MakeYpObject();
                }
                break;
            }
        }

        if (keySelectionResult.Defined()) {
            selectionResult.push_back(std::move(*keySelectionResult));
        }
    }

    const TMaybe<ui32> serial = GetSerial(*zone, zoneReplicas);
    for (const NClient::TDnsRecordSet& dnsRecordSet : selectionResult) {
        for (const auto& record : dnsRecordSet.Spec().records()) {
            NYpDns::TRecord resourceRecord = NYpDns::CreateRecordFromProto(name, record, zone->GetDefaultTtl());
            switch (resourceRecord.Type) {
                case NYpDns::ERecordType::SOA: {
                    if (serial.Defined()) {
                        NYpDns::TSOAData soaData(resourceRecord.Data);
                        soaData.Serial = *serial;
                        resourceRecord.Data = soaData.ToString();
                    }
                }
                default:
                    break;
            }
            NYpDns::ERecordType type = resourceRecord.Type;
            result[type].push_back(std::move(resourceRecord));
        }
    }

    if (name == DNSName(zone->GetName())) {
        if (auto [it, inserted] = result.try_emplace(NYpDns::ERecordType::NS); inserted || it->second.empty()) {
            it->second.reserve(zone->GetNameservers().size());
            for (const TString& nameserver : zone->GetNameservers()) {
                it->second.push_back(NYpDns::CreateNSRecord(name, nameserver, zone->GetNSRecordTtl()));
            }
        }
        if (auto [it, inserted] = result.try_emplace(NYpDns::ERecordType::SOA); inserted || it->second.empty()) {
            result[NYpDns::ERecordType::SOA].push_back(NYpDns::CreateSOARecord(*zone, serial.GetOrElse(zone->GetSOASerial())));
        }
    }

    for (auto& [type, records] : result) {
        SortUnique(records);
        if (const NYpDns::TTypeResponsePolicy* responsePolicy = NYpDns::FindTypeResponsePolicy(*zone, type)) {
            if (responsePolicy->GetOrder() == NYpDns::TTypeResponsePolicy::RANDOM) {
                Shuffle(records.begin(), records.end(), RandomGenerator_);
            }
            if (i32 recordsNumber = responsePolicy->GetMaxRecordsNumber(); recordsNumber != -1 && recordsNumber < static_cast<i32>(records.size())) {
                records.resize(recordsNumber);
            }
        }
    }

    NUnbound::NEventlog::TLookupResult lookupResultEvent;
    for (NYpDns::ERecordType recordType : GetEnumAllValues<NYpDns::ERecordType>()) {
        if (const auto* it = result.FindPtr(recordType); !it || it->empty()) {
            NInfra::TRateSensor{sensorGroup, NSensors::EMPTY, {{NSensors::RECORD_TYPE, ToString(recordType)}}}.Inc();
        } else {
            (*lookupResultEvent.MutableRecordsNumber())[ToString(recordType)] = it->size();
        }
    }
    logFrame->LogEvent(ELogPriority::TLOG_DEBUG, std::move(lookupResultEvent));

    return result;
}

} // namespace NYP::DNS
