#pragma once

#include <infra/libs/unbound/yp_dns/config/config.pb.h>
#include <infra/libs/unbound/yp_dns/router_api/router_api.h>
#include <infra/libs/unbound/yp_dns/service_iface/service_iface.h>

#include <infra/libs/yp_dns/dynamic_zones/dns_zones_replica.h>
#include <infra/libs/yp_dns/record_set/record.h>
#include <infra/libs/yp_dns/zone/zone.h>

#include <infra/libs/http_service/service.h>
#include <infra/libs/yp_replica/replica_objects.h>
#include <infra/libs/yp_replica/yp_replica.h>

#include <infra/contrib/pdns/power_dns/dnsname.hh>

#include <yp/yp_proto/yp/client/api/proto/data_model.pb.h>

#include <library/cpp/threading/future/future.h>

#include <util/generic/hash.h>
#include <util/generic/string.h>
#include <util/random/fast.h>

template <>
struct THash<DNSName> {
    size_t operator()(const DNSName& name) const;
};

namespace NYP::DNS {

class TDnsService: public IService {
public:
    using TYPReplica = NYPReplica::TYPReplica<NYPReplica::TDnsRecordSetReplicaObject>;
    using TYPReplicasList = TVector<std::pair<TString, const TYPReplica*>>;
    using TConfigPtr = TAtomicSharedPtr<const TConfig>;

    TDnsService(const TConfig& config);

    NThreading::TFuture<void> Start() override;

    void Stop();

    bool IsReady() const;

    void Ping(NInfra::TRequestPtr<NApi::TReqPing>, NInfra::TReplyPtr<NApi::TRspPing> reply) override;

    void ReopenLog(NInfra::TRequestPtr<NApi::TReqReopenLog>, NInfra::TReplyPtr<NApi::TRspReopenLog> reply) override;

    void Sensors(NInfra::TRequestPtr<NApi::TReqSensors>, NInfra::TReplyPtr<NApi::TRspSensors> reply) override;

    void SensorsJson(NInfra::TRequestPtr<NApi::TReqSensors>, NInfra::TReplyPtr<NApi::TRspSensors> reply) override;

    void Config(NInfra::TRequestPtr<NApi::TReqConfig>, NInfra::TReplyPtr<NApi::TRspConfig> reply) override;

    void ListBackups(NInfra::TRequestPtr<NApi::TReqListBackups>, NInfra::TReplyPtr<NApi::TRspListBackups>) override;

    void RollbackToBackup(const TStringBuf clusterName, const NApi::TReqRollbackToBackup&, NApi::TRspRollbackToBackup&);
    void RollbackToBackup(NInfra::TRequestPtr<NApi::TReqRollbackToBackup>, NInfra::TReplyPtr<NApi::TRspRollbackToBackup>) override;

    void StartUpdates(const TStringBuf clusterName, const NApi::TReqStartUpdates&, NApi::TRspStartUpdates&);
    void StartUpdates(NInfra::TRequestPtr<NApi::TReqStartUpdates>, NInfra::TReplyPtr<NApi::TRspStartUpdates>) override;

    void StopUpdates(const TStringBuf clusterName, const NApi::TReqStopUpdates&, NApi::TRspStopUpdates&);
    void StopUpdates(NInfra::TRequestPtr<NApi::TReqStopUpdates>, NInfra::TReplyPtr<NApi::TRspStopUpdates>) override;

    void ListZoneRecordSets(NInfra::TRequestPtr<NApi::TReqListZoneRecordSets> request, NInfra::TReplyPtr<NApi::TRspListZoneRecordSets> reply) override;

    void ListZoneData(NInfra::TRequestPtr<NApi::TReqListZoneData> request, NInfra::TReplyPtr<NApi::TRspListZoneData> reply) override;
    void ListZoneDataRaw(NInfra::TRequestPtr<NApi::TReqListZoneData> request, NInfra::TReplyPtr<NApi::TRspListZoneData> reply) override;

    NApi::TRspListZoneData ListZoneData(const NApi::TReqListZoneData& request);

    void ListDynamicZones(NInfra::TRequestPtr<NApi::TReqListZones>, NInfra::TReplyPtr<NApi::TRspListZones>) override;

    TVector<NYpDns::TZone> ListDynamicZones() const;

    TYPReplicasList GetYpReplicasForZone(const NYpDns::TZoneConfig& zone) const;

    TAtomicSharedPtr<const NYpDns::TZoneConfig> DetermineZone(DNSName domain, NInfra::TLogFramePtr logFrame) const;

    TAtomicSharedPtr<const NYpDns::TZoneConfig> FindDynamicZone(const TStringBuf domain) const;

    TAtomicSharedPtr<const NYpDns::TZoneConfig> FindZone(const DNSName& domain, const TStringBuf domainView, NInfra::TLogFramePtr logFrame) const;

    const NYpDns::TTypeResponsePolicy* FindTypeResponsePolicy(const NYpDns::TZoneConfig& zoneConfig, NYpDns::ERecordType recordType) const;

    TMap<NYpDns::ERecordType, TVector<NYpDns::TRecord>> Lookup(const TString& domain);

private:
    void LockSelfMemory();

    void InitReplicas();
    void StartReplicas(NInfra::TLogFramePtr logFrame);
    void StopReplicas();

    void InitSensors();

    void InitListZonePool();

    void InitZones();

    void PrintSensors(TStringOutput& output, NInfra::ESensorsSerializationType serializationType) const;

    void SwitchConfigsCallback(const TConfig& oldConfig, const TConfig& newConfig);

    TMaybe<ui32> GetSerial(const NYpDns::TZoneConfig& zone, const TYPReplicasList& replicas) const;

private:
    // Logging helpers
    void ListDynamicZonesLogResult(NInfra::TLogFramePtr logFrame, const NInfra::TSensorGroup& sensorGroup, const NApi::TRspListZones& rsp) const;

    NInfra::TLogger* GetLogger() override;

private:
    NUpdatableProtoConfig::TConfigHolder<TConfig> ConfigHolder_;
    NUpdatableProtoConfig::TAccessor<TConfig> Config_;
    NInfra::TLogger Logger_;
    NInfra::TLogger BackupLogger_;
    THolder<NInfra::TLogger> DnsZonesReplicaLogger_;

    const NInfra::TSensorGroup SensorGroup_;
    THashMap<TString, TIntrusivePtr<NInfra::THistogramRateSensor>> HistogramSensors_;

    NInfra::THttpService HttpService_;

    THolder<NYpDns::NDynamicZones::TDnsZonesReplica> DnsZonesReplica_;

    THashMap<TString, THolder<TYPReplica>> Replicas_;
    THashMap<TString, TYPReplicasList> ReplicasByZone_;

    THashMap<DNSName, NUpdatableProtoConfig::TAccessor<NYpDns::TZoneConfig>> Zones_;

    mutable TReallyFastRng32 RandomGenerator_;

    mutable std::atomic<bool> ReplicasInited_ = {false};
    THolder<IThreadPool> ReplicasManagementPool_;
    NThreading::TFuture<void> InitReplicasFuture_;

    THolder<IThreadPool> ListZonePool_;
};

} // namespace NYP::DNS
