#include "unified_agent_log_backend.h"

#include <logbroker/unified_agent/client/cpp/logger/backend.h>

#include <library/cpp/logger/file.h>
#include <library/cpp/logger/log.h>
#include <library/cpp/logger/stream.h>

namespace NInfra::NUnifiedAgentLogBackend {

class TUnifiedAgentLogBackend::TImpl {
public:
    inline TImpl(const TUnifiedAgentLogBackendConfig& config, TLogBackend* slave = nullptr)
        : Slave_(slave)
    {
        if (config.HasLogFilePath() && !config.GetLogFilePath().empty()) {
            ClientLog_ = TLog(MakeHolder<TFileLogBackend>(config.GetLogFilePath()));
        } else {
            ClientLog_ = TLog(MakeHolder<TStreamLogBackend>(&Cerr));
        }
        auto clientParameters = NUnifiedAgent::TClientParameters(config.GetUri()).SetLog(ClientLog_);

        if (config.HasSharedSecretKey()) {
            clientParameters = clientParameters.SetSharedSecretKey(config.GetSharedSecretKey());
        }

        if (config.HasMaxInflightBytes()) {
            clientParameters = clientParameters.SetMaxInflightBytes(config.GetMaxInflightBytes());
        }        

        NUnifiedAgent::TSessionParameters sessionParameters;
        THashMap<TString, TString> metaMap(config.GetMeta().begin(), config.GetMeta().end());
        sessionParameters.SetMeta(metaMap);
        UnifiedAgentBackend_ = NUnifiedAgent::MakeLogBackend(std::move(clientParameters), std::move(sessionParameters));
    }

    inline void WriteData(const TLogRecord& rec) {
        if (Slave_) {
            Slave_->WriteData(rec);
        }
        UnifiedAgentBackend_->WriteData(rec);
    }

    inline void ReopenLog() {
        if (Slave_) {
            Slave_->ReopenLog();
        }
        UnifiedAgentBackend_->ReopenLog();
    }

    inline void ReopenLogNoFlush() {
        if (Slave_) {
            Slave_->ReopenLogNoFlush();
        }
        UnifiedAgentBackend_->ReopenLogNoFlush();
    }

    size_t SlaveQueueSize() const {
        if (Slave_) {
            return Slave_->QueueSize();
        }
        ythrow yexception() << "slave is missing";
    }

private:
    TLog ClientLog_;
    TLogBackend* Slave_;
    THolder<TLogBackend> UnifiedAgentBackend_;
};

TUnifiedAgentLogBackend::TUnifiedAgentLogBackend(const TUnifiedAgentLogBackendConfig& config, TLogBackend* slave)
    : Impl_(new TImpl(config, slave))
{
}

TUnifiedAgentLogBackend::~TUnifiedAgentLogBackend() {
}

void TUnifiedAgentLogBackend::WriteData(const TLogRecord& rec) {
    Impl_->WriteData(rec);
}

void TUnifiedAgentLogBackend::ReopenLog() {
    Impl_->ReopenLog();
}

void TUnifiedAgentLogBackend::ReopenLogNoFlush() {
    Impl_->ReopenLogNoFlush();
}

size_t TUnifiedAgentLogBackend::SlaveQueueSize() const {
    return Impl_->SlaveQueueSize();
}

}
