#pragma once

#include "holder.h"

#include <util/folder/path.h>
#include <util/generic/ptr.h>
#include <util/string/join.h>
#include <util/system/rwlock.h>

namespace NUpdatableProtoConfig {

namespace {

template <typename TSubConfig, typename TConfig>
TAtomicSharedPtr<const TSubConfig> CopySubConfig(const TConfig& config, const TStringBuf path) {
    const google::protobuf::Message* message = &config;
    auto descriptor = message->GetDescriptor();
    auto reflection = message->GetReflection();

    TStringBuf left = path;
    for (TStringBuf part; left.NextTok('/', part);) {
        auto field = descriptor->FindFieldByName(TString(part));
        Y_ENSURE(field, "member '" << part << "' not found");
        if (field->is_repeated()) {
            Y_ENSURE(left.NextTok('/', part));

            TString keyFieldName = field->options().GetExtension(Key);

            const auto& repeatedField = reflection->template GetRepeatedPtrField<google::protobuf::Message>(*message, field);
            bool found = false;
            for (int i = 0; i < repeatedField.size(); ++i) {
                const google::protobuf::Message* repMessage = &repeatedField[i];
                auto repDescriptor = repMessage->GetDescriptor();
                auto repReflection = repMessage->GetReflection();
                auto keyField = repDescriptor->FindFieldByName(keyFieldName);
                Y_ENSURE(keyField, "member '" << part << "' not found");
                Y_ENSURE(keyField->type() == google::protobuf::FieldDescriptor::TYPE_STRING);
                if (repReflection->GetString(*repMessage, keyField) == part) {
                    message = repMessage;
                    descriptor = repDescriptor;
                    reflection = repReflection;
                    found = true;
                    break;
                }
            }
            Y_ENSURE(found, "no field found");
        } else {
            Y_ENSURE(field->type() == google::protobuf::FieldDescriptor::TYPE_MESSAGE);
            message = &reflection->GetMessage(*message, field);
            descriptor = message->GetDescriptor();
            reflection = message->GetReflection();
        }
    }

    const TSubConfig* subconfig = dynamic_cast<const TSubConfig*>(message);
    Y_ENSURE(subconfig, "subconfig type does not match proto message by path '" << path << "'");
    return MakeAtomicShared<TSubConfig>(*subconfig);
}

} // anonymous namespace

namespace NPrivate {

struct TFuncExecutor {
    template <typename F>
    auto operator | (F&& function) const {
        return function();
    }
};

} // namespace NPrivate

#define WITH_CONFIG_SNAPSHOT(config, snapshot) \
    ::NUpdatableProtoConfig::NPrivate::TFuncExecutor{} | [&, snapshot = (config).Get()] ()

#define CONFIG_SNAPSHOT_VALUE(config, ...) \
    ( \
        WITH_CONFIG_SNAPSHOT(config, __configSnapshot) -> std::remove_reference_t<decltype((config).Get()->__VA_ARGS__)> { \
            return __configSnapshot->__VA_ARGS__; \
        } \
    )

template <typename TConfig>
class TAccessor: public IUpdateSubscriber {
public:
    using TUpdateConfigCallback = std::function<void(const TConfig&, const TConfig&, const TWatchContext&)>;

    TAccessor(IConfigHolder* holder, const TStringBuf path)
        : Holder_(holder)
        , Path_(path)
    {
        if (!UpdateCacheIfNeeded()) {
            ythrow yexception() << "failed to get subconfig by path '" << Path_ << "'";
        }
    }

    TAccessor(const TAccessor<TConfig>& other)
        : Holder_(other.Holder_)
        , Path_(other.Path_)
    {
        {
            TReadGuard guard(other.Mutex_);
            ParentConfig_ = other.ParentConfig_;
            CachedConfig_ = other.CachedConfig_;
        }

        if (!UpdateCacheIfNeeded()) {
            ythrow yexception() << "failed to get subconfig by path '" << Path_ << "'";
        }
    }

    TAccessor(TAccessor<TConfig>&& other)
        : TAccessor(other)
    {
        if (other.Subscribed_) {
            TUpdateConfigCallback updateCallback = other.UpdateConfigCallback_;
            SubscribeForUpdate(std::move(updateCallback));
            other.UnsubscribeForUpdate();
        }
    }

    ~TAccessor() {
        UnsubscribeForUpdate();
    }

    template <typename TSubConfig>
    TAccessor<TSubConfig> Accessor(const TVector<TString>& path) const {
        return Accessor<TSubConfig>(JoinSeq("/", path));
    }

    template <typename TSubConfig>
    TAccessor<TSubConfig> Accessor(const TStringBuf path) const {
        return TAccessor<TSubConfig>(Holder_, Path_ ? Join('/', Path_, path) : path);
    }

    TAtomicSharedPtr<const TConfig> Get() const {
        UpdateCacheIfNeeded();
        TReadGuard guard(Mutex_);
        return CachedConfig_;
    }

    TConfig operator*() const {
        return *Get();
    }

    void SubscribeForUpdate(TUpdateConfigCallback&& callback) {
        UpdateConfigCallback_ = std::move(callback);
        if (!Subscribed_) {
            Subscribed_ = true;
            Holder_->Subscribe(this);
        }
    }

    void UnsubscribeForUpdate() {
        if (Subscribed_) {
            Subscribed_ = false;
            Holder_->Unsubscribe(this);
            UpdateConfigCallback_ = {};
        }
    }

    void RequestReopenLogs() {
        Holder_->ReopenLog();
    }

    bool RequestUpdate(const TWatchContext& context) {
        return Holder_->Update(context);
    }

private:
    bool UpdateCacheIfNeeded() const {
        {
            TReadGuard guard(Mutex_);
            if (ParentConfig_ == Holder_->ConfigMessage()) {
                return true;
            }
        }

        TWriteGuard guard(Mutex_);
        if (ParentConfig_ == Holder_->ConfigMessage()) {
            return true;
        }

        ParentConfig_ = Holder_->ConfigMessage();

        TAtomicSharedPtr<const TConfig> newValue;
        try {
            newValue = CopySubConfig<TConfig>(*ParentConfig_, Path_);
        } catch (...) {
            return false;
        }

        CachedConfig_.Swap(newValue);
        return true;
    }

    void OnUpdate(const google::protobuf::Message& oldParentConfig, const google::protobuf::Message& newParentConfig, const TWatchContext& context = {}) const override {
        if (UpdateConfigCallback_) {
            TAtomicSharedPtr<const TConfig> oldConfig = CopySubConfig<TConfig>(oldParentConfig, Path_);
            TAtomicSharedPtr<const TConfig> newConfig = CopySubConfig<TConfig>(newParentConfig, Path_);
            UpdateConfigCallback_(*oldConfig, *newConfig, context);
        }
    }

private:
    IConfigHolder* const Holder_;
    const TString Path_;
    mutable TRWMutex Mutex_;
    mutable TAtomicSharedPtr<const google::protobuf::Message> ParentConfig_;
    mutable TAtomicSharedPtr<const TConfig> CachedConfig_;

    bool Subscribed_ = false;
    TUpdateConfigCallback UpdateConfigCallback_;
};

} // namespace NUpdatableProtoConfig
