#pragma once

#include "patch_file_watcher.h"
#include "patcher.h"
#include "sensors.h"

#include <infra/libs/logger/logger.h>
#include <infra/libs/sensors/sensor.h>
#include <infra/libs/sensors/sensor_group.h>
#include <infra/libs/updatable_proto_config/protos/config.pb.h>
#include <infra/libs/updatable_proto_config/protos/events_decl.ev.pb.h>
#include <infra/libs/updatable_proto_config/protos/extensions.pb.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <google/protobuf/message.h>
#include <google/protobuf/util/message_differencer.h>

#include <util/folder/path.h>
#include <util/generic/hash_set.h>
#include <util/generic/ptr.h>
#include <util/string/join.h>
#include <util/string/subst.h>
#include <util/system/rwlock.h>

namespace NUpdatableProtoConfig {

class IUpdateSubscriber {
public:
    virtual void OnUpdate(const google::protobuf::Message& /* oldConfig */, const google::protobuf::Message& /* newConfig */, const TWatchContext& /* context */) const { }
};

template <typename TConfig>
class TAccessor;

class IConfigHolder {
    template <typename TSubConfig>
    friend class TAccessor;
public:
    virtual ~IConfigHolder() = default;

    virtual void Start() { }
    virtual void Stop() { }
    virtual void ReopenLog() { }
    virtual bool Update(const TWatchContext& /* context */ = {}) {
        return false;
    }

private:
    virtual TAtomicSharedPtr<const google::protobuf::Message> ConfigMessage() const = 0;
    virtual void Subscribe(IUpdateSubscriber* /* subscriber */) { }
    virtual void Unsubscribe(IUpdateSubscriber* /* subscriber */) { }
};

template <typename TConfig>
class IConfigHolderBase: public IConfigHolder {
public:
    using TSwitchConfigCallback = std::function<void(const TConfig&, const TConfig&)>;

    virtual TAtomicSharedPtr<const TConfig> Config() const = 0;

    virtual void SetSwitchConfigsCallback(TSwitchConfigCallback /* callback */) { }

    template <typename TSubConfig>
    TAccessor<TSubConfig> Accessor(const TVector<TString>& path) {
        return Accessor<TSubConfig>(JoinSeq("/", path));
    }

    template <typename TSubConfig = TConfig>
    TAccessor<TSubConfig> Accessor(const TStringBuf path = {}) {
        return TAccessor<TSubConfig>(this, path);
    }
};

template <typename TConfig>
using TConfigHolderPtr = TAtomicSharedPtr<IConfigHolderBase<TConfig>>;

template <typename TConfig>
class TStaticConfigHolder : public IConfigHolderBase<TConfig> {
public:
    TStaticConfigHolder(TConfig config)
        : Config_(MakeAtomicShared<const TConfig>(std::move(config)))
    {
    }

    TAtomicSharedPtr<const TConfig> Config() const override {
        return Config_;
    }

private:
    TAtomicSharedPtr<const google::protobuf::Message> ConfigMessage() const override {
        return Config_;
    }

private:
    TAtomicSharedPtr<const TConfig> Config_;
};

template <typename TConfig>
class TConfigHolder : public IConfigHolderBase<TConfig> {
public:
    using TSwitchConfigCallback = typename IConfigHolderBase<TConfig>::TSwitchConfigCallback;

    TConfigHolder(TConfig config, const TWatchPatchConfig& watchConfig, const NInfra::TLoggerConfig& loggerConfig)
        : OriginalConfig_(MakeAtomicShared<TConfig>(std::move(config)))
        , SensorGroup_(NSensors::NAMESPACE)
        , Patcher_(*OriginalConfig_, SensorGroup_)
        , Logger_(loggerConfig)
        , Watcher_(watchConfig, Logger_, SensorGroup_, [this](const TFsPath& patchPath, const NJson::TJsonValue& patch, NInfra::TLogFramePtr logFrame, const TWatchContext& context = {}) -> bool {
            return PatchConfig(patchPath, patch, logFrame, context);
        })
    {
        Proto2JsonConfig_.SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName);
        Update();
    }

    void Start() override {
        StartWatchPatch();
    }

    void Stop() override {
        StopWatchPatch();
    }

    void StartWatchPatch() {
        Watcher_.Start();
    }

    void StopWatchPatch() {
        Watcher_.Stop();
    }

    bool Update(const TWatchContext& context = {}) override {
        return Watcher_.Watch(context);
    }

    void SetSwitchConfigsCallback(TSwitchConfigCallback callback) override {
        SwitchConfigsCallback_ = std::move(callback);
    }

    void ReopenLog() override {
        Logger_.SpawnFrame()->LogEvent(NInfra::NLogEvent::TReopenLog());
        Logger_.ReopenLog();
    }

    TAtomicSharedPtr<const TConfig> Config() const override {
        TReadGuard guard(Mutex_);
        if (UseOriginal_ || !PatchedConfig_) {
            return OriginalConfig_;
        }
        return PatchedConfig_;
    }

private:
    TAtomicSharedPtr<const google::protobuf::Message> ConfigMessage() const override {
        return Config();
    }

    void Subscribe(IUpdateSubscriber* subscriber) override {
        TGuard<TMutex> guard(SubscribersMutex_);
        UpdateSubscribers_.insert(subscriber);
    }

    void Unsubscribe(IUpdateSubscriber* subscriber) override {
        TGuard<TMutex> guard(SubscribersMutex_);
        UpdateSubscribers_.erase(subscriber);
    }

private:
    bool PatchConfig(const TFsPath& patchPath, const NJson::TJsonValue& patch, NInfra::TLogFramePtr logFrame, const TWatchContext& context = {}) {
        const TVector<std::pair<TStringBuf, TStringBuf>> labels = {
            {NSensors::PATH, patchPath.GetPath()},
        };

        TGuard<TMutex> patchGuard(PatchMutex_);

        TAtomicSharedPtr<const TConfig> prevActualConfig = Config();
        TAtomicSharedPtr<const TConfig> newPatchedConfig;
        bool newUseOriginalFlagValue = false;

        if (!patch.IsDefined()) {
            logFrame->LogEvent(TUndefinedPatch(patchPath.GetPath()));
            newUseOriginalFlagValue = false;
            newPatchedConfig = OriginalConfig_;
        } else {
            logFrame->LogEvent(TPatchWith(patchPath.GetPath(), NJson::WriteJson(patch["patch"], /* formatOutput */ false)));
            newPatchedConfig = Patcher_.Patch(patch["patch"], logFrame, labels);
            newUseOriginalFlagValue = patch["dont_apply"].GetBooleanRobust();
        }
        Y_ENSURE(newPatchedConfig);
        logFrame->LogEvent(TPatchConfigResult(newUseOriginalFlagValue, NProtobufJson::Proto2Json(*newPatchedConfig, Proto2JsonConfig_)));

        TString diff;
        google::protobuf::util::MessageDifferencer differencer;
        // TODO: Set EQUIVALENT mode
        differencer.ReportDifferencesToString(&diff);
        TAtomicSharedPtr<const TConfig> newActualConfig = newUseOriginalFlagValue ? OriginalConfig_ : newPatchedConfig;
        const bool hasChanges =
            !differencer.Compare(*prevActualConfig, *newActualConfig) ||
            !PatchedConfig_ ||
            !google::protobuf::util::MessageDifferencer::Equals(*PatchedConfig_, *newPatchedConfig) ||
            UseOriginal_ != newUseOriginalFlagValue;

        SubstGlobal(diff, '\n', ';');
        logFrame->LogEvent(TResultConfigsDiff(diff));

        NInfra::TIntGaugeSensor(SensorGroup_, NSensors::USE_ORIGINAL_CONFIG).Set(newUseOriginalFlagValue);
        NInfra::TIntGaugeSensor(SensorGroup_, NSensors::USE_PATCHED_CONFIG, labels).Set(!newUseOriginalFlagValue);
        NInfra::TRateSensor(SensorGroup_, NSensors::NONEMPTY_DIFF, labels).Add(hasChanges);

        if (hasChanges) {
            {
                TWriteGuard guard(Mutex_);
                PatchedConfig_ = newPatchedConfig;
                UseOriginal_ = newUseOriginalFlagValue;
            }

            try {
                if (SwitchConfigsCallback_) {
                    SwitchConfigsCallback_(*prevActualConfig, *Config());
                }
                with_lock (SubscribersMutex_) {
                    for (const IUpdateSubscriber* subscriber : UpdateSubscribers_) {
                        if (subscriber) {
                            subscriber->OnUpdate(*prevActualConfig, *Config(), context);
                        }
                    }
                }
                NInfra::TRateSensor(SensorGroup_, NSensors::SWITCH_CONFIGS_CALLBACK_SUCCESS, labels).Inc();
            } catch (...) {
                logFrame->LogEvent(ELogPriority::TLOG_ERR, TSwitchConfigsCallbackError(CurrentExceptionMessage()));
                NInfra::TRateSensor(SensorGroup_, NSensors::SWITCH_CONFIGS_CALLBACK_ERROR, labels).Inc();
            }
        }

        return hasChanges;
    }

private:
    TMutex PatchMutex_;
    TRWMutex Mutex_;
    bool UseOriginal_ = false;
    TAtomicSharedPtr<const TConfig> OriginalConfig_;
    TAtomicSharedPtr<const TConfig> PatchedConfig_;

    const NInfra::TSensorGroup SensorGroup_;

    TSwitchConfigCallback SwitchConfigsCallback_;

    TMutex SubscribersMutex_;
    THashSet<const IUpdateSubscriber*> UpdateSubscribers_;

    NProtobufJson::TProto2JsonConfig Proto2JsonConfig_;

    TConfigPatcher<TConfig> Patcher_;
    NInfra::TLogger Logger_;
    TPatchFileWatcher Watcher_;
};

template <template <typename> typename THolder = TConfigHolder, typename TConfig, typename... TArgs>
TConfigHolderPtr<TConfig> CreateConfigHolder(const TConfig& config, TArgs&&... args) {
    return MakeAtomicShared<THolder<TConfig>>(config, std::forward<TArgs>(args)...);
}

template<typename TConfig>
TConfigHolderPtr<TConfig> CreateConfigHolder(const TConfig& config, const TConfigHolderConfig& holderOpts) {
    if (holderOpts.GetEnabled()) {
        return CreateConfigHolder(
            config,
            holderOpts.GetWatchPatchConfig(),
            holderOpts.GetConfigUpdatesLoggerConfig()
        );
    }
    return CreateConfigHolder<TStaticConfigHolder>(config);
}

} // namespace NUpdatableProtoConfig
