#include "accessor.h"
#include "holder.h"

#include <infra/libs/updatable_proto_config/ut/protos/config.pb.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/test_common.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/scheme/ut_utils/scheme_ut_utils.h>
#include <library/cpp/testing/unittest/registar.h>

#include <google/protobuf/util/message_differencer.h>

#include <util/stream/file.h>
#include <util/system/tempfile.h>
#include <util/system/yield.h>

using namespace NInfra;
using namespace NUpdatableProtoConfig;

namespace {

constexpr TStringBuf LOG_FILE = "test_eventlog";

TLoggerConfig GetLoggerConfig(TStringBuf fileId) {
    return NTestCommon::CreateLoggerConfig(TStringBuf("DEBUG"), TString::Join(LOG_FILE, "_", fileId));
}

TTestConfig CreateDefaultConfig() {
    TTestSubConfig subConfig;
    for (int i = 0; i < 10; ++i) {
        TTestSubSubConfig subSubConfig;
        subSubConfig.SetKey1("KEY" + ToString(i));
        subSubConfig.SetValue1("VALUE" + ToString(i));
        *subConfig.add_subconfig() = subSubConfig;
    }

    TTestConfig config;
    for (int i = 0; i < 10; ++i) {
        *config.AddRepeatedStringParam() = TStringBuilder() << "value" << i;
    }

    for (int i = 0; i < 10; ++i) {
        subConfig.SetKey("key" + ToString(i));
        subConfig.SetValue("value" + ToString(i));
        *config.AddUpdatableRepeatedParam() = subConfig;
    }

    for (int i = 10; i < 20; ++i) {
        subConfig.SetKey("key" + ToString(i));
        subConfig.SetValue("value" + ToString(i));
        *config.AddRepeatedParam() = subConfig;
    }

    return config;
}

TWatchPatchConfig CreateDefaultWatchConfig() {
    TWatchPatchConfig watchConfig;
    watchConfig.SetPath("patch.json");
    watchConfig.SetValidPatchPath("valid_patch.json");
    watchConfig.SetFrequency("10ms");
    return watchConfig;
}

TDuration GetSleepTime(const TWatchPatchConfig& config) {
    return TDuration::Parse(config.GetFrequency()) * 100;
}

template <typename TConfig>
TConfigHolderPtr<TConfig> CreateConfigHolder(const TConfig& config, TStringBuf loggerId = "", const TWatchPatchConfig& watchConfig = {}, bool cleanPatches = true) {
    if (cleanPatches) {
        TFsPath(watchConfig.GetPath()).ForceDelete();
        TFsPath(watchConfig.GetValidPatchPath()).ForceDelete();
    }
    return CreateConfigHolder(config, watchConfig, GetLoggerConfig(loggerId));
}

template <typename TConfig>
TConfigHolderPtr<TConfig> CreateStaticConfigHolder(const TConfig& config) {
    return CreateConfigHolder<TStaticConfigHolder>(config);
}

template <typename TConfig>
bool Equals(const TConfig& lhs, const TConfig& rhs) {
    return google::protobuf::util::MessageDifferencer::Equivalent(lhs, rhs);
}

void SleepForUpdate(const TDuration& sleepTime) {
    ThreadYield();
    Sleep(sleepTime);
}

template <typename TConfig>
void ApplyPatch(const NJson::TJsonValue& patch, TFile& file, TConfigHolderPtr<TConfig>& holder) {
    TUnbufferedFileOutput output(file.GetName());
    NJson::WriteJson(&output, &patch);
    holder->Update();
}

void ApplyPatch(const NJson::TJsonValue& patch, TFile& file, const TDuration& sleepTime = {}) {
    TUnbufferedFileOutput output(file.GetName());
    NJson::WriteJson(&output, &patch);
    SleepForUpdate(sleepTime);
}

NJson::TJsonValue ReadJsonFromFile(const TString& file) {
    TUnbufferedFileInput input(file);
    return NJson::ReadJsonTree(&input);
}

void CheckPatchValid(const TWatchPatchConfig& config) {
    UNIT_ASSERT_JSON_EQ_JSON(
        ReadJsonFromFile(config.GetPath()),
        ReadJsonFromFile(config.GetValidPatchPath())
    );
}

} // anonymous namespace

Y_UNIT_TEST_SUITE(ConfigHolderTests) {
    Y_UNIT_TEST(TestConfigGetter) {
        TTestConfig config = CreateDefaultConfig();
        TConfigHolderPtr<TTestConfig> holder = CreateConfigHolder(config, TStringBuf(Name_));

        UNIT_ASSERT(holder->Config());
        UNIT_ASSERT(Equals(config, *holder->Config()));

        config.SetStringParam("another value");
        // TConfigHolder holds a copy of config
        UNIT_ASSERT_STRINGS_UNEQUAL(config.GetStringParam(), holder->Config()->GetStringParam());
    }

    Y_UNIT_TEST(TestAccessorGetter) {
        TTestConfig config = CreateDefaultConfig();
        TConfigHolderPtr<TTestConfig> holder = CreateConfigHolder(config, TStringBuf(Name_));

        {
            TAccessor<TTestConfig> accessor = holder->Accessor();
            UNIT_ASSERT(Equals(config, *accessor));
        }

        {
            TAccessor<TTestSubConfig> accessor = holder->Accessor<TTestSubConfig>("ComplexParam");
            UNIT_ASSERT(Equals(config.GetComplexParam(), *accessor));
        }

        {
            for (int j = 0; j < config.GetComplexParam().GetSubConfig().size(); ++j) {
                const TString subSubConfigKey = "KEY" + ToString(j);
                TAccessor<TTestSubSubConfig> subSubAccessor = holder->Accessor<TTestSubSubConfig>({"UpdatableRepeatedParam/ComplexParam/SubConfig", subSubConfigKey});

                UNIT_ASSERT(Equals(config.GetComplexParam().GetSubConfig(j), *subSubAccessor));
            }
        }

        {
            for (int i = 0; i < config.GetUpdatableRepeatedParam().size(); ++i) {
                const TString subConfigKey = "key" + ToString(i);
                TAccessor<TTestSubConfig> subAccessor = holder->Accessor<TTestSubConfig>({"UpdatableRepeatedParam", subConfigKey});
                UNIT_ASSERT(Equals(config.GetUpdatableRepeatedParam(i), *subAccessor));
                for (int j = 0; j < config.GetUpdatableRepeatedParam(i).GetSubConfig().size(); ++j) {
                    const TString subSubConfigKey = "KEY" + ToString(j);
                    TAccessor<TTestSubSubConfig> subSubAccessor = holder->Accessor<TTestSubSubConfig>({"UpdatableRepeatedParam", subConfigKey, "SubConfig", subSubConfigKey});

                    UNIT_ASSERT(Equals(config.GetUpdatableRepeatedParam(i).GetSubConfig(j), *subSubAccessor));
                }
            }
        }

        // Key is not set for RepeatedParam
        {
            for (int i = 0; i < config.GetUpdatableRepeatedParam().size(); ++i) {
                const TString subConfigKey = "key" + ToString(i);
                UNIT_ASSERT_EXCEPTION(holder->Accessor<TTestSubConfig>({"RepeatedParam", subConfigKey}), yexception);
            }
        }

        // invalid subconfig path
        UNIT_ASSERT_EXCEPTION(holder->Accessor<TTestSubConfig>("InvalidParamName"), yexception);
        UNIT_ASSERT_EXCEPTION(holder->Accessor("InvalidParamName"), yexception);
        UNIT_ASSERT_EXCEPTION(holder->Accessor<TTestConfig>("InvalidParamName"), yexception);

        // accessor supports only protomessages so far
        UNIT_ASSERT_EXCEPTION(holder->Accessor<TString>("StringParam"), yexception);

        // wrong subconfig type
        UNIT_ASSERT_EXCEPTION(holder->Accessor<TTestConfig>("ComplexParam"), yexception);
        UNIT_ASSERT_EXCEPTION(holder->Accessor("ComplexParam"), yexception);
        UNIT_ASSERT_EXCEPTION(holder->Accessor<TTestSubConfig>(), yexception);
    }

    Y_UNIT_TEST(TestWatchPatchStartStop) {
        const TTestConfig config = CreateDefaultConfig();

        const TWatchPatchConfig watchConfig = CreateDefaultWatchConfig();
        const TDuration sleepTime = GetSleepTime(watchConfig);

        TConfigHolderPtr<TTestConfig> holder = CreateConfigHolder(config, TStringBuf(Name_), watchConfig);
        TAccessor<TTestConfig> accessor = holder->Accessor();
        TTempFileHandle file(watchConfig.GetPath());

        UNIT_ASSERT(Equals(config, *holder->Config()));
        UNIT_ASSERT(Equals(config, *accessor));

        NJson::TJsonValue patch;
        patch["patch"]["StringParam"] = "value2";
        ApplyPatch(patch, file, sleepTime);

        // watch not started
        UNIT_ASSERT(Equals(config, *holder->Config()));
        UNIT_ASSERT(Equals(config, *accessor));

        holder->Start();
        SleepForUpdate(sleepTime);

        const TString updatedValue = patch["patch"]["StringParam"].GetString();
        UNIT_ASSERT_STRINGS_EQUAL(holder->Config()->GetStringParam(), updatedValue);
        UNIT_ASSERT_STRINGS_EQUAL(CONFIG_SNAPSHOT_VALUE(accessor, GetStringParam()), updatedValue);

        holder->Stop();

        patch["patch"]["StringParam"] = "value3";
        ApplyPatch(patch, file, sleepTime);

        // watch updates stopped
        UNIT_ASSERT_STRINGS_EQUAL(holder->Config()->GetStringParam(), updatedValue);
        UNIT_ASSERT_STRINGS_EQUAL(CONFIG_SNAPSHOT_VALUE(accessor, GetStringParam()), updatedValue);
    }

    Y_UNIT_TEST(TestDontApplyPatch) {
        const TTestConfig config = CreateDefaultConfig();

        const TWatchPatchConfig watchConfig = CreateDefaultWatchConfig();

        TConfigHolderPtr<TTestConfig> holder = CreateConfigHolder(config, TStringBuf(Name_), watchConfig);
        TAccessor<TTestConfig> accessor = holder->Accessor();

        TTempFileHandle file(watchConfig.GetPath());

        const ui32 initValue = config.GetUint32Param();

        auto testValue = [&](ui32 expected) {
            UNIT_ASSERT_EQUAL(config.GetUint32Param(), initValue);
            UNIT_ASSERT_EQUAL(holder->Config()->GetUint32Param(), expected);
            UNIT_ASSERT_EQUAL(CONFIG_SNAPSHOT_VALUE(accessor, GetUint32Param()), expected);

            TTestConfig newConfig = config;
            newConfig.SetUint32Param(expected);
            UNIT_ASSERT(Equals(newConfig, *holder->Config()));
            UNIT_ASSERT(Equals(newConfig, *accessor));
        };

        testValue(initValue);

        NJson::TJsonValue patch;

        ui32 newValue = 1337;
        patch["patch"]["Uint32Param"] = newValue;
        ApplyPatch(patch, file, holder);
        testValue(newValue);

        patch["dont_apply"] = true;
        ApplyPatch(patch, file, holder);
        testValue(initValue);

        newValue = 101;
        patch["patch"]["Uint32Param"] = newValue;
        ApplyPatch(patch, file, holder);
        testValue(initValue);

        patch["dont_apply"] = false;
        ApplyPatch(patch, file, holder);
        testValue(newValue);
    }

    Y_UNIT_TEST(TestInvalidPatchFile) {
        const TTestConfig config = CreateDefaultConfig();

        const TWatchPatchConfig watchConfig = CreateDefaultWatchConfig();

        TConfigHolderPtr<TTestConfig> holder = CreateConfigHolder(config, TStringBuf(Name_), watchConfig);
        holder->Start();

        TAccessor<TTestConfig> accessor = holder->Accessor();

        TTempFileHandle file(watchConfig.GetPath());

        const ui32 initValue = config.GetUint32Param();

        auto testValue = [&](ui32 expected) {
            UNIT_ASSERT_VALUES_EQUAL(config.GetUint32Param(), initValue);
            UNIT_ASSERT_VALUES_EQUAL(holder->Config()->GetUint32Param(), expected);
            UNIT_ASSERT_VALUES_EQUAL(CONFIG_SNAPSHOT_VALUE(accessor, GetUint32Param()), expected);

            TTestConfig newConfig = config;
            newConfig.SetUint32Param(expected);
            UNIT_ASSERT(Equals(newConfig, *holder->Config()));
            UNIT_ASSERT(Equals(newConfig, *accessor));
        };

        {
            TUnbufferedFileOutput output(file.GetName());
            output << "asd";
            holder->Update();

            testValue(initValue);
        }

        NJson::TJsonValue patch;
        ui32 newValue = 101;
        UNIT_ASSERT_UNEQUAL(initValue, newValue);
        patch["patch"]["Uint32Param"] = newValue;
        ApplyPatch(patch, file, holder);
        testValue(newValue);

        // patch fails, remain last patched config
        patch["patch"]["Uint32Param"] = "invalid type value";
        ApplyPatch(patch, file, holder);
        testValue(newValue);

        // dont_apply does not affect while patch is still invalid
        patch["dont_apply"] = true;
        ApplyPatch(patch, file, holder);
        testValue(newValue);

        patch["patch"] = NJson::JSON_MAP;
        ApplyPatch(patch, file, holder);
        testValue(initValue);

        // invalid patch, don't apply
        patch["dont_apply"] = false;
        patch["patch"]["Uint32Param"] = "invalid type value";
        ApplyPatch(patch, file, holder);
        testValue(initValue);

        patch["patch"]["Uint32Param"] = newValue;
        ApplyPatch(patch, file, holder);
        testValue(newValue);

        // drop patch file, patch rolled back
        NFs::Remove(file.GetName());
        UNIT_ASSERT(!NFs::Exists(file.GetName()));
        holder->Update();
        testValue(initValue);
    }

    Y_UNIT_TEST(TestUpdateSubConfigs) {
        const TTestConfig config = CreateDefaultConfig();

        const TWatchPatchConfig watchConfig = CreateDefaultWatchConfig();

        TConfigHolderPtr<TTestConfig> holder = CreateConfigHolder(config, TStringBuf(Name_), watchConfig);
        TAccessor<TTestConfig> accessor = holder->Accessor();

        TTempFileHandle file(watchConfig.GetPath());

        for (int paramId = 0; paramId < config.GetUpdatableRepeatedParam().size(); ++paramId) {
            const TString paramKey = "key" + ToString(paramId);
            TAccessor<TTestSubConfig> subAccessor = accessor.Accessor<TTestSubConfig>({"UpdatableRepeatedParam", paramKey});

            const TString initValue = config.GetUpdatableRepeatedParam(paramId).GetValue();
            auto testValue = [&](const TString& expected) {
                UNIT_ASSERT_STRINGS_EQUAL(config.GetUpdatableRepeatedParam(paramId).GetValue(), initValue);
                UNIT_ASSERT_STRINGS_EQUAL(holder->Config()->GetUpdatableRepeatedParam(paramId).GetValue(), expected);
                UNIT_ASSERT_STRINGS_EQUAL(CONFIG_SNAPSHOT_VALUE(accessor, GetUpdatableRepeatedParam(paramId).GetValue()), expected);
                UNIT_ASSERT_STRINGS_EQUAL(CONFIG_SNAPSHOT_VALUE(subAccessor, GetValue()), expected);

                TTestConfig newConfig = config;
                newConfig.MutableUpdatableRepeatedParam(paramId)->SetValue(expected);
                UNIT_ASSERT(Equals(newConfig, *holder->Config()));
                UNIT_ASSERT(Equals(newConfig, *accessor));
            };

            holder->Update();
            testValue("value" + ToString(paramId));

            NJson::TJsonValue patch;
            for (int i = 0; i < 10; ++i) {
                TString newValue = "value_updated" + ToString(i);
                patch["patch"]["UpdatableRepeatedParam"][paramKey]["Value"] = newValue;
                ApplyPatch(patch, file, holder);

                testValue(newValue);
            }

            patch["patch"] = NJson::JSON_MAP;
            ApplyPatch(patch, file, holder);

            testValue(initValue);
        }
    }

    Y_UNIT_TEST(TestUpdateCallback) {
        const TTestConfig config = CreateDefaultConfig();

        const TWatchPatchConfig watchConfig = CreateDefaultWatchConfig();

        TConfigHolderPtr<TTestConfig> holder = CreateConfigHolder(config, TStringBuf(Name_), watchConfig);
        TTestConfig lastOldConfig;
        TTestConfig lastNewConfig;
        ui32 updates = 0;
        holder->SetSwitchConfigsCallback([&](const TTestConfig& oldConfig, const TTestConfig& newConfig) {
            if (!Equals(oldConfig, newConfig)) {
                lastOldConfig = oldConfig;
                lastNewConfig = newConfig;
                ++updates;
            }
        });

        holder->Start();

        TTempFileHandle file(watchConfig.GetPath());

        TTestConfig currentConfig = *holder->Config();

        NJson::TJsonValue patch;

        UNIT_ASSERT_EQUAL(updates, 0);

        patch["patch"]["Uint32Param"] = 101;
        ApplyPatch(patch, file, holder);
        UNIT_ASSERT(Equals(lastOldConfig, currentConfig));
        currentConfig = *holder->Config();
        UNIT_ASSERT(Equals(lastNewConfig, currentConfig));
        UNIT_ASSERT_VALUES_EQUAL(updates, 1);

        patch["patch"]["UpdatableRepeatedParam"]["key0"]["SubConfig"]["KEY0"]["Value1"] = "VALUE2";
        ApplyPatch(patch, file, holder);
        UNIT_ASSERT(Equals(lastOldConfig, currentConfig));
        currentConfig = *holder->Config();
        UNIT_ASSERT(Equals(lastNewConfig, currentConfig));
        UNIT_ASSERT_VALUES_EQUAL(updates, 2);

        patch["dont_apply"] = true;
        ApplyPatch(patch, file, holder);
        UNIT_ASSERT(Equals(lastOldConfig, currentConfig));
        currentConfig = *holder->Config();
        UNIT_ASSERT(Equals(lastNewConfig, currentConfig));
        UNIT_ASSERT_VALUES_EQUAL(updates, 3);

        // no callback when patch is invalid
        patch["dont_apply"] = false;
        patch["patch"]["ComplexParam"] = "aoaoao";
        ApplyPatch(patch, file, holder);
        currentConfig = *holder->Config();
        UNIT_ASSERT_VALUES_EQUAL(updates, 3);

        patch["patch"] = NJson::JSON_MAP;
        ApplyPatch(patch, file, holder);
        UNIT_ASSERT_VALUES_EQUAL(updates, 3);
    }

    Y_UNIT_TEST(TestCopyValidPatch) {
        const TTestConfig config = CreateDefaultConfig();

        const TWatchPatchConfig watchConfig = CreateDefaultWatchConfig();

        TConfigHolderPtr<TTestConfig> holder = CreateConfigHolder(config, TStringBuf(Name_), watchConfig);

        const TFsPath validPatchPath{watchConfig.GetValidPatchPath()};
        UNIT_ASSERT(!validPatchPath.Exists());

        holder->Start();

        TTempFileHandle file(watchConfig.GetPath());

        NJson::TJsonValue patch;
        patch["patch"]["Uint32Param"] = 101;
        ApplyPatch(patch, file, holder);

        UNIT_ASSERT(validPatchPath.Exists());
        CheckPatchValid(watchConfig);
        UNIT_ASSERT_JSON_EQ_JSON(patch, ReadJsonFromFile(watchConfig.GetValidPatchPath()));

        {
            TUnbufferedFileOutput output(file.GetName());
            output << "asd";
            holder->Update();
        }

        // valid patch did not changed
        UNIT_ASSERT_JSON_EQ_JSON(patch, ReadJsonFromFile(watchConfig.GetValidPatchPath()));

        // valid patch should be deleted if patch file is deleted
        NFs::Remove(file.GetName());
        UNIT_ASSERT(!NFs::Exists(file.GetName()));
        holder->Update();
        UNIT_ASSERT(!validPatchPath.Exists());
    }

    Y_UNIT_TEST(TestStartFromValidPatch) {
        const TTestConfig config = CreateDefaultConfig();

        const TWatchPatchConfig watchConfig = CreateDefaultWatchConfig();

        const TFsPath validPatchPath{watchConfig.GetValidPatchPath()};

        ui32 newValue = 101;

        TTempFileHandle file(watchConfig.GetPath());

        NJson::TJsonValue patch;
        patch["patch"]["Uint32Param"] = newValue;
        {
            TConfigHolderPtr<TTestConfig> holder = CreateConfigHolder(config, TStringBuf(Name_), watchConfig);
            holder->Start();

            // create valid patch
            ApplyPatch(patch, file, holder);
            CheckPatchValid(watchConfig);

            // make patch invalid
            TUnbufferedFileOutput output(file.GetName());
            output << "asd";
            holder->Update();
        }

        TConfigHolderPtr<TTestConfig> holder = CreateConfigHolder(config, TStringBuf(Name_), watchConfig, /* cleanPatches */ false);

        UNIT_ASSERT(validPatchPath.Exists());

        // loaded from valid patch
        UNIT_ASSERT_JSON_EQ_JSON(patch, ReadJsonFromFile(watchConfig.GetValidPatchPath()));
        UNIT_ASSERT_VALUES_EQUAL(holder->Config()->GetUint32Param(), newValue);

        holder->Start();

        // still using valid patch
        UNIT_ASSERT_JSON_EQ_JSON(patch, ReadJsonFromFile(watchConfig.GetValidPatchPath()));
        UNIT_ASSERT_VALUES_EQUAL(holder->Config()->GetUint32Param(), newValue);
    }

    Y_UNIT_TEST(TestStaticConfigHolder) {
        TTestConfig config = CreateDefaultConfig();
        TConfigHolderPtr<TTestConfig> holder = CreateStaticConfigHolder(config);

        UNIT_ASSERT(holder->Config());
        UNIT_ASSERT(Equals(config, *holder->Config()));

        config.SetStringParam("another value");
        // TStaticConfigHolder holds a copy of config
        UNIT_ASSERT_STRINGS_UNEQUAL(config.GetStringParam(), holder->Config()->GetStringParam());
    }
}
