#include "patch_file_watcher.h"

#include "sensors.h"

#include <infra/libs/sensors/sensor.h>
#include <infra/libs/updatable_proto_config/protos/events_decl.ev.pb.h>

#include <library/cpp/json/json_reader.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NUpdatableProtoConfig {

TPatchFileWatcher::TPatchFileWatcher(const TWatchPatchConfig& config, NInfra::TLogger& logger, const NInfra::TSensorGroup& sensorGroup, TCallback callback)
    : WatchThread_(MakeHolder<NInfra::TBackgroundThread>(
        [this]{ return Watch(); }, TDuration::Parse(config.GetFrequency())
    ))
    , Logger_(logger)
    , SensorGroup_(sensorGroup, NSensors::PATCH_WATCHER_NAMESPACE)
    , Path_(config.GetPath())
    , CopiedPatchPath_(config.GetPath() + ".copy")
    , ValidPatchPath_(config.GetValidPatchPath())
    , Callback_(std::move(callback))
{
}

TPatchFileWatcher::~TPatchFileWatcher() {
    Stop();
}

void TPatchFileWatcher::Start() {
    Logger_.SpawnFrame()->LogEvent(TStartWatchPatchFile(Path_));
    WatchThread_->Start();
}

void TPatchFileWatcher::Stop() {
    Logger_.SpawnFrame()->LogEvent(TStopWatchPatchFile(Path_));
    WatchThread_->Stop();
}

TWatchResult TPatchFileWatcher::WatchFile(const TFsPath& path, bool copyContent, NInfra::TLogFramePtr logFrame, const TWatchContext& context) const {
    const TVector<std::pair<TStringBuf, TStringBuf>> labels = {
        {NSensors::PATH, path.GetPath()},
    };

    NJson::TJsonValue patch = NJson::TJsonValue::UNDEFINED;
    if (NFs::Exists(Path_)) {
        NInfra::TIntGaugeSensor(SensorGroup_, NSensors::FILE_EXISTS, labels).Set(1);
        try {
            TString content = TUnbufferedFileInput(path).ReadAll();
            logFrame->LogEvent(ELogPriority::TLOG_DEBUG, TPatchFileContent(path, content));
            Y_ENSURE(NJson::ReadJsonTree(content, &patch, /* throwOnError */ true));
            NInfra::TRateSensor(SensorGroup_, NSensors::PARSE_SUCCESS, labels).Inc();

            if (copyContent) {
                TUnbufferedFileOutput output(CopiedPatchPath_);
                output.Write(content);
            }
        } catch (...) {
            logFrame->LogEvent(ELogPriority::TLOG_ERR, TReadPatchFileError(path, CurrentExceptionMessage()));
            NInfra::TRateSensor(SensorGroup_, NSensors::PARSE_ERROR, labels).Inc();
            return {false, false};
        }
    } else {
        CopiedPatchPath_.DeleteIfExists();
        NInfra::TIntGaugeSensor(SensorGroup_, NSensors::FILE_EXISTS, labels).Set(0);
    }

    bool diffFound = false;
    try {
        diffFound = Callback_(path, patch, logFrame, context);
        NInfra::TRateSensor(SensorGroup_, NSensors::CALLBACK_SUCCESS, labels).Inc();
    } catch (...) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TWatchPatchFileCallbackError(path, CurrentExceptionMessage()));
        NInfra::TRateSensor(SensorGroup_, NSensors::CALLBACK_ERROR, labels).Inc();
        return {false, diffFound};
    }

    return {true, diffFound};
}

bool TPatchFileWatcher::Watch(const TWatchContext& context) const {
    NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();

    TGuard<TMutex> watchGuard(Mutex_);

    TWatchResult result;
    if (result = WatchFile(Path_, /* copyContent */ true, logFrame, context); !result.WatchSucceeded || !CopyFile(CopiedPatchPath_, ValidPatchPath_, logFrame)) {
        result = WatchFile(ValidPatchPath_, /* copyContent */ false, logFrame, context);
        return result.DiffFound;
    }

    return result.DiffFound;
}

bool TPatchFileWatcher::CopyFile(const TFsPath& from, const TFsPath& to, NInfra::TLogFramePtr logFrame) const {
    const TFsPath tmpFile(to.GetPath() + ".tmp");
    const TVector<std::pair<TStringBuf, TStringBuf>> labels = {
        {NSensors::FROM_FILE, from.GetPath()},
        {NSensors::TMP_FILE, tmpFile.GetPath()},
        {NSensors::TO_FILE, to.GetPath()},
    };

    try {
        if (from.Exists()) {
            from.CopyTo(tmpFile, /* force */ true);
            tmpFile.RenameTo(to);
        } else {
            to.DeleteIfExists();
        }
        NInfra::TRateSensor(SensorGroup_, NSensors::COPY_FILE_SUCCESS, labels).Inc();
    } catch (...) {
        logFrame->LogEvent(ELogPriority::TLOG_ERR, TCopyFileError(from.GetPath(), tmpFile.GetPath(), to.GetPath(), CurrentExceptionMessage()));
        NInfra::TRateSensor(SensorGroup_, NSensors::COPY_FILE_ERROR, labels).Inc();
        return false;
    }

    return true;
}

} // namespace NUpdatableProtoConfig
