#include "patcher.h"

#include <infra/libs/sensors/sensor.h>
#include <infra/libs/updatable_proto_config/protos/events_decl.ev.pb.h>
#include <infra/libs/updatable_proto_config/protos/extensions.pb.h>

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <library/cpp/json/json_writer.h>

namespace NUpdatableProtoConfig {

google::protobuf::Message* MutableValueByPath(google::protobuf::Message* config, TStringBuf path) {
    google::protobuf::Message* message = config;
    auto descriptor = message->GetDescriptor();
    auto reflection = message->GetReflection();

    for (TStringBuf part; path.NextTok('/', part);) {
        auto field = descriptor->FindFieldByName(TString(part));
        Y_ENSURE(field, "member '" << part << "' not found in '" << descriptor->full_name() << "'");

        if (field->is_repeated()) {
            Y_ENSURE(path.NextTok('/', part));

            TString keyFieldName = field->options().GetExtension(Key);
            Y_ENSURE(!keyFieldName.empty(), "no key specified for repeated field '" << field->full_name() << "'");

            auto* repeatedField = reflection->MutableRepeatedPtrField<google::protobuf::Message>(message, field);
            bool found = false;
            for (int i = 0; i < repeatedField->size(); ++i) {
                google::protobuf::Message* repMessage = repeatedField->Mutable(i);
                auto repDescriptor = repMessage->GetDescriptor();
                auto repReflection = repMessage->GetReflection();

                auto keyField = repDescriptor->FindFieldByName(keyFieldName);
                Y_ENSURE(keyField, "member '" << part << "' not found in '" << repDescriptor->full_name() << "'");
                Y_ENSURE(keyField->type() == google::protobuf::FieldDescriptor::TYPE_STRING);
                if (repReflection->GetString(*repMessage, keyField) == part) {
                    message = repMessage;
                    descriptor = repDescriptor;
                    reflection = repReflection;
                    found = true;
                    break;
                }
            }
            Y_ENSURE(found, "no field found");
        } else {
            Y_ENSURE(field->type() == google::protobuf::FieldDescriptor::TYPE_MESSAGE);
            message = reflection->MutableMessage(message, field);
            descriptor = message->GetDescriptor();
            reflection = message->GetReflection();
        }
    }

    Y_ENSURE(message);
    return message;
}

void MergeJson2Proto(const NJson::TJsonValue& value, google::protobuf::Message* message) {
    NProtobufJson::TJson2ProtoConfig cfg;
    cfg.SetReplaceRepeatedFields(true);
    NProtobufJson::MergeJson2Proto(value, *message, cfg);
}

TScanPatchCallback::TScanPatchCallback(google::protobuf::Message* config, NInfra::TLogFramePtr logFrame, const NInfra::TSensorGroup& sensorGroup)
    : Config_(config)
    , LogFrame_(logFrame)
    , SensorGroup_(sensorGroup)
{
    Proto2JsonConfig_.SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName);
}

void TScanPatchCallback::operator()(const TStringBuf path, const NJson::TJsonValue& value) {
    LogFrame_->LogEvent(TMerge(TString{path}, NJson::WriteJson(value, /* formatOutput */ false)));
    try {
        auto* message = MutableValueByPath(Config_, path);
        LogFrame_->LogEvent(TValueBeforeMerge(NProtobufJson::Proto2Json(*message, Proto2JsonConfig_)));
        MergeJson2Proto(value, message);
        LogFrame_->LogEvent(TValueAfterMerge(NProtobufJson::Proto2Json(*message, Proto2JsonConfig_)));
        NInfra::TRateSensor(SensorGroup_, NSensors::MERGE_SUCCESS, {{NSensors::MERGE_PATH, path}}).Inc();
    } catch (...) {
        LogFrame_->LogEvent(TMergeError(CurrentExceptionMessage()));
        NInfra::TRateSensor(SensorGroup_, NSensors::MERGE_ERROR, {{NSensors::MERGE_PATH, path}}).Inc();
        throw;
    }
}

void ScanPatch(const TString& path, const NJson::TJsonValue& value, IScanCallback& callback) {
    if (value.IsMap()) {
        NJson::TJsonValue patchValue;
        for (const auto& [k, v] : value.GetMap()) {
            if (v.IsMap()) {
                ScanPatch(path ? TString::Join(path, "/", k) : k, v, callback);
            } else {
                patchValue[k] = v;
            }
        }
        if (patchValue.IsDefined()) {
            callback(path, patchValue);
        }
    }
}

} // namespace NUpdatableProtoConfig
