#pragma once

#include "sensors.h"

#include <infra/libs/logger/log_frame.h>
#include <infra/libs/sensors/sensor_group.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/protobuf/json/config.h>

#include <google/protobuf/message.h>

namespace NUpdatableProtoConfig {

void MergeValueByPath(google::protobuf::Message* config, TStringBuf path, const NJson::TJsonValue& value);

class IScanCallback {
public:
    virtual ~IScanCallback() = default;
    virtual void operator()(const TStringBuf path, const NJson::TJsonValue& value) = 0;
};

class TScanPatchCallback: public IScanCallback {
public:
    TScanPatchCallback(google::protobuf::Message* config, NInfra::TLogFramePtr logFrame, const NInfra::TSensorGroup& sensorGroup);

    void operator()(const TStringBuf path, const NJson::TJsonValue& value) override;

private:
    google::protobuf::Message* Config_;
    NProtobufJson::TProto2JsonConfig Proto2JsonConfig_;
    NInfra::TLogFramePtr LogFrame_;
    const NInfra::TSensorGroup SensorGroup_;
};

void ScanPatch(const TString& path, const NJson::TJsonValue& value, IScanCallback& callback);

template <typename TConfig>
class TConfigPatcher {
public:
    TConfigPatcher(TConfig config, const NInfra::TSensorGroup& sensorGroup)
        : Config_(config)
        , SensorGroup_(sensorGroup, NSensors::PATCHER_NAMESPACE)
    {
    }

    TAtomicSharedPtr<TConfig> Patch(const NJson::TJsonValue& patch, NInfra::TLogFramePtr logFrame, const TVector<std::pair<TStringBuf, TStringBuf>>& sensorLabels) const {
        NInfra::TSensorGroup sensorGroup = SensorGroup_;
        sensorGroup.AddLabels(sensorLabels);

        TAtomicSharedPtr<TConfig> patchedConfig = MakeAtomicShared<TConfig>(Config_);
        TScanPatchCallback scanPatchCallback(patchedConfig.Get(), logFrame, sensorGroup);
        ScanPatch("", patch, scanPatchCallback);

        return patchedConfig;
    }

private:
    TConfig Config_;
    const NInfra::TSensorGroup SensorGroup_;
};

} // namespace NUpdatableProtoConfig
