#pragma once

#include <build/scripts/c_templates/svnversion.h>

#include <infra/libs/version_handle/config/api.pb.h>
#include <infra/libs/version_handle/config/events_decl.ev.pb.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/service_iface/request.h>
#include <infra/libs/service_iface/str_iface.h>
#include <infra/libs/sensors/sensor.h>

#include <library/cpp/build_info/build_info_static.h>

#include <util/generic/maybe.h>
#include <util/string/cast.h>
#include <util/string/vector.h>
#include <util/system/compiler.h>

namespace NInfra::NVersionHandle {

    static TString SerializeRequestsAttributes(const TAttributes& attributes) {
        TString result;
        for (const auto& [key, value] : attributes) {
            result += TString::Join(key, "=", value, ";");
        }
        return result;
    }

    class TBinaryApi {
    public:

        TBinaryApi() 
            : VersionSensorGroup_("version")
        {
            TIntGaugeSensor(VersionSensorGroup_, "svn_revision").Set(::GetProgramSvnRevision());
        }

        virtual ~TBinaryApi() = default;

        void SVNRevision(TRequestPtr<NBinaryApi::TReqSVNRevision> request, TReplyPtr<NBinaryApi::TRspSVNRevision> reply) {
            if (auto logger = GetLogger()) {
                logger->SpawnFrame()->LogEvent(NEventlog::TSVNRevision(SerializeRequestsAttributes(request->Attributes())));
            }

            Y_UNUSED(request);
            int svnRevision = ::GetProgramSvnRevision();
            TString data;
            if (svnRevision <= 0) {
                data = ::GetProgramCommitId();
            } else {
                data = ToString(svnRevision);
            }
            NBinaryApi::TRspSVNRevision result;
            result.SetData(data);
            reply->Set(result);
        }

        void Branch(TRequestPtr<NBinaryApi::TReqBranch> request, TReplyPtr<NBinaryApi::TRspBranch> reply) {
            if (auto logger = GetLogger()) {
                logger->SpawnFrame()->LogEvent(NVersionHandle::NEventlog::TBranch(SerializeRequestsAttributes(request->Attributes())));
            }

            Y_UNUSED(request);
            TString data = ::GetBranch();
            NBinaryApi::TRspBranch result;
            result.SetData(data);
            reply->Set(result);
        }

        void CommitId(TRequestPtr<NBinaryApi::TReqCommit> request, TReplyPtr<NBinaryApi::TRspCommit> reply) {
            if (auto logger = GetLogger()) {
                logger->SpawnFrame()->LogEvent(NVersionHandle::NEventlog::TCommit(SerializeRequestsAttributes(request->Attributes())));
            }

            Y_UNUSED(request);
            TString data = ::GetProgramCommitId();
            NBinaryApi::TRspCommit result;
            result.SetData(data);
            reply->Set(result);
        }

        void VersionJson(TRequestPtr<NBinaryApi::TReqAllProgramInfo> request, TReplyPtr<NBinaryApi::TRspAllProgramInfo> reply) {
            if (auto logger = GetLogger()) {
                logger->SpawnFrame()->LogEvent(NVersionHandle::NEventlog::TAllProgramInfo(SerializeRequestsAttributes(request->Attributes())));
            }

            Y_UNUSED(request);
            NBinaryApi::TRspAllProgramInfo result;
            result.SetBuildUser(::GetProgramBuildUser());
            result.SetBuildHost(::GetProgramBuildHost());
            result.SetBuildDate(::GetProgramBuildDate());
            result.SetBuildTimestamp(ToString(::GetProgramBuildTimestamp()));
            result.SetSVNRevision(ToString(::GetProgramSvnRevision()));
            result.SetBranch(::GetBranch());
            result.SetCommit(::GetProgramCommitId());

            auto compilerInfos = SplitString(::GetCompilerVersion(), "\n");
            for (const auto& info : compilerInfos) {
                for (auto it = info.begin(); it != info.end(); ++it) {
                    if (*it != ' ') {
                        result.AddCompilerVersion(TString(it, info.end()));
                        break;
                    }
                }
            }

            auto compilerFlags = SplitString(::GetCompilerFlags(), " ");
            for (const auto& flag : compilerFlags) {
                result.AddCompilerFlags(flag);
            }

            reply->Set(result);
        }

        // add handlers here
    private:
        virtual TLogger* GetLogger() {
            return nullptr; // we dont have logger object, but our heirs have
        }

        TSensorGroup VersionSensorGroup_;

    };

    template <typename TRouter>
    TRouter& AddSVNRevisionHandler(TRouter& router) {
        router
            .template Add<
                TEmptyRequest<NBinaryApi::TReqSVNRevision>,
                TRawDataReply<NBinaryApi::TRspSVNRevision>>("/version/svn/revision", "Show svn revision", &TBinaryApi::SVNRevision);
        return router;
    }

    template <typename TRouter>
    TRouter& AddBranchNameHandler(TRouter& router) {
        router
            .template Add<
                TEmptyRequest<NBinaryApi::TReqBranch>,
                TRawDataReply<NBinaryApi::TRspBranch>>("/version/branch", "Show branch name", &TBinaryApi::Branch);
        return router;
    }

    template <typename TRouter>
    TRouter& AddCommitIdHandler(TRouter& router) {
        router
            .template Add<
                TEmptyRequest<NBinaryApi::TReqCommit>,
                TRawDataReply<NBinaryApi::TRspCommit>>("/version/commit", "Show commit id", &TBinaryApi::CommitId);
        return router;
    }

    template <typename TRouter>
    TRouter& AddAllProgramInfoHandler(TRouter& router) {
        router
            .template Add<
                TEmptyRequest<NBinaryApi::TReqAllProgramInfo>,
                TProto2JsonReply<NBinaryApi::TRspAllProgramInfo>>("/version/all", "Return json about version", &TBinaryApi::VersionJson);
        return router;
    }

    template <typename TRouter>
    TRouter& AddBinaryHandlers(TRouter& router) {
        AddSVNRevisionHandler(router);
        AddBranchNameHandler(router);
        AddCommitIdHandler(router);
        AddAllProgramInfoHandler(router);
        return router;
    }

} // namespace NVersionHandle
