#include "changelist.h"

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

namespace NYpDns {

bool operator<(const TBaseVersions& lhs, const TBaseVersions& rhs) {
    const TVector<TString> lhsClusters = lhs.Clusters();
    const TVector<TString> rhsClusters = rhs.Clusters();
    TVector<TString> intersection;
    SetIntersection(lhsClusters.begin(), lhsClusters.end(),
                    rhsClusters.begin(), rhsClusters.end(),
                    std::back_inserter(intersection));
    return lhs.VersionsList(intersection) < rhs.VersionsList(intersection);
}

TChange::TChange() = default;
TChange::TChange(const TChange&) = default;
TChange::TChange(TChange&&) = default;

TChange::TChange(NProto::TChange change)
    : NProto::TChange(std::move(change))
{
}

TChange::TChange(const NJson::TJsonValue& json) {
    FromJson(json);
}

TChange& TChange::operator=(const TChange&) = default;
TChange& TChange::operator=(TChange&&) = default;

NJson::TJsonValue TChange::ToJson() const {
    NJson::TJsonValue result;
    NProtobufJson::Proto2Json(*this, result);
    return result;
}

void TChange::FromJson(const NJson::TJsonValue& json) {
    NProtobufJson::Json2Proto(json, *this);
}

bool TChange::operator==(const TChange& rhs) const {
    return google::protobuf::util::MessageDifferencer::Equals(*this, rhs);
}

////////////////////////////////////////////////////////////////////////////////////////

TChangelist::TChangelist() = default;
TChangelist::TChangelist(const TChangelist&) = default;
TChangelist::TChangelist(TChangelist&&) = default;

TChangelist::TChangelist(const NJson::TJsonValue& json) {
    FromJson(json);
}

TChangelist& TChangelist::operator=(const TChangelist&) = default;
TChangelist& TChangelist::operator=(TChangelist&&) = default;

NJson::TJsonValue TChangelist::ToJson() const {
    NJson::TJsonValue result;
    NProtobufJson::Proto2Json(*this, result);
    return result;
}

void TChangelist::FromJson(const NJson::TJsonValue& json) {
    NProtobufJson::Json2Proto(json, *this);
}

void TChangelist::AddChangeEntry(TChange change) {
    *add_changes() = std::move(change);
}

const TBaseVersions* TChangelist::BaseVersions() const {
    return static_cast<const TBaseVersions*>(&base_versions());
}

TBaseVersions* TChangelist::MutableBaseVersions() {
    return static_cast<TBaseVersions*>(mutable_base_versions());
}

bool TChangelist::MatchHash(const TString& hashValue) const {
    return record_set_hash().empty() || record_set_hash() == hashValue;
}

bool TChangelist::MatchHashExact(const TString& hashValue) const {
    return record_set_hash() == hashValue;
}

bool TChangelist::operator==(const TChangelist& rhs) const {
    return google::protobuf::util::MessageDifferencer::Equals(*this, rhs);
}

} // namespace NYpDns
