#include "labels.h"
#include "name.h"
#include "name_ref.h"
#include "view.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/yexception.h>

using namespace NYpDns;

template <typename TName>
void ValidateNameLength() {
    UNIT_ASSERT_NO_EXCEPTION(TName(""));
    UNIT_ASSERT_NO_EXCEPTION(TName("."));

    const TString validDnsName =
        TString("a") * 63 + "." +
        TString("b") * 63 + "." +
        TString("c") * 63 + "." +
        TString("d") * 61;
    UNIT_ASSERT_EQUAL(validDnsName.size(), 253);
    UNIT_ASSERT_NO_EXCEPTION(TName(validDnsName));
    UNIT_ASSERT_NO_EXCEPTION(TName(validDnsName + "."));

    TString tooLongDnsName = validDnsName + "d";
    UNIT_ASSERT_EQUAL(tooLongDnsName.size(), 254);
    UNIT_ASSERT_EXCEPTION_CONTAINS(TName(tooLongDnsName), yexception, "too long DNS name");
}

template <typename TName>
void ValidateLabelsLength() {
    UNIT_ASSERT_EXCEPTION_CONTAINS(TName("..a.yandex.net"), yexception, "empty label");
    UNIT_ASSERT_EXCEPTION_CONTAINS(TName(".a.yandex.net"), yexception, "empty label");
    UNIT_ASSERT_EXCEPTION_CONTAINS(TName("a..yandex.net"), yexception, "empty label");
    UNIT_ASSERT_EXCEPTION_CONTAINS(TName("a.yandex.net.."), yexception, "empty label");
    UNIT_ASSERT_EXCEPTION_CONTAINS(TName(".."), yexception, "empty label");

    UNIT_ASSERT_NO_EXCEPTION(TName(TString("a") * 63 + ".yandex.net"));
    UNIT_ASSERT_EXCEPTION_CONTAINS(TName(TString("a") * 64 + ".yandex.net"), yexception, "too long label in DNS name");
}

#define UNIT_ASSERT_NAMES(comparision_type, lhsName, rhsName) \
    do { \
        TLhsBase lhs(lhsName); \
        TRhsBase rhs(rhsName); \
        UNIT_ASSERT_##comparision_type(TLhsName(lhs), TRhsName(rhs)); \
    } while (false)

template <
    typename TLhsName, typename TLhsBase = TStringBuf,
    typename TRhsName = TLhsName, typename TRhsBase = TLhsBase
>
void ValidateEquality() {
    // Equal
    UNIT_ASSERT_NAMES(EQUAL, "", ".");
    UNIT_ASSERT_NAMES(EQUAL, "A.yanDeX.Net", "a.YanDex.nEt");

    // Unequal
    UNIT_ASSERT_NAMES(UNEQUAL, "test", ".");
    UNIT_ASSERT_NAMES(UNEQUAL, "A.yanDeX.Net", "b.YanDex.nEt");
}

template <
    typename TLhsName, typename TLhsBase = TStringBuf,
    typename TRhsName = TLhsName, typename TRhsBase = TLhsBase
>
void ValidateComparison() {
    // Less
    UNIT_ASSERT_NAMES(LT, "", "a");
    UNIT_ASSERT_NAMES(LT, "A.yanDeX.Net", "b.YanDex.nEt");

    // Less or equal
    UNIT_ASSERT_NAMES(LE, ".", "");
    UNIT_ASSERT_NAMES(LE, "", "a");
    UNIT_ASSERT_NAMES(LE, "A.yanDeX.Net", "a.YanDex.nEt");
    UNIT_ASSERT_NAMES(LE, "A.yanDeX.Net", "b.YanDex.nEt");

    // Greater
    UNIT_ASSERT_NAMES(GT, "a", "");
    UNIT_ASSERT_NAMES(GT, "b.YanDex.nEt", "A.yanDeX.Net");

    // Greater or equal
    UNIT_ASSERT_NAMES(GE, ".", "");
    UNIT_ASSERT_NAMES(GE, "a", "");
    UNIT_ASSERT_NAMES(GE, "b.YanDex.nEt", "A.yanDeX.Net");
    UNIT_ASSERT_NAMES(GE, "a.YanDex.nEt", "A.yanDeX.Net");
}

Y_UNIT_TEST_SUITE(Labels) {
    Y_UNIT_TEST(Size) {
        for (const TStringBuf root : {"", "."}) {
            const TLabels<TLabel> labels = TDnsName(root).Labels();
            UNIT_ASSERT_VALUES_EQUAL(labels.size(), 0);
        }

        for (int labelsNum = 1; labelsNum < 20; ++labelsNum) {
            const TLabels<TLabel> labels = TDnsName(TString("a.") * labelsNum).Labels();
            UNIT_ASSERT_VALUES_EQUAL(labels.size(), labelsNum);
        }
    }

    Y_UNIT_TEST(StringComparison) {
        const TLabels<TLabel> labels = TDnsName("A.yanDeX.Net").Labels();

        // labels are case-insensitive
        UNIT_ASSERT_VALUES_EQUAL(labels[0], "a");
        UNIT_ASSERT_VALUES_EQUAL(labels[1], "YANdex");
        UNIT_ASSERT_VALUES_EQUAL(labels[2], "nET");

        // but we can cast to TStringBuf which has case-sensitive chars comparison
        UNIT_ASSERT_VALUES_UNEQUAL(TStringBuf{labels[0]}, "a");
        UNIT_ASSERT_VALUES_UNEQUAL(TStringBuf{labels[1]}, "YANdex");
        UNIT_ASSERT_VALUES_UNEQUAL(TStringBuf{labels[2]}, "nET");
        UNIT_ASSERT_VALUES_EQUAL(TStringBuf{labels[0]}, "A");
        UNIT_ASSERT_VALUES_EQUAL(TStringBuf{labels[1]}, "yanDeX");
        UNIT_ASSERT_VALUES_EQUAL(TStringBuf{labels[2]}, "Net");
    }

    Y_UNIT_TEST(CompareEqual) {
        UNIT_ASSERT_EQUAL(
            TDnsName("").Labels(),
            TDnsName(".").Labels()
        );
        UNIT_ASSERT_EQUAL(
            TDnsName("A.yanDeX.Net").Labels(),
            TDnsName("a.YanDex.nEt").Labels()
        );
    }

    Y_UNIT_TEST(ToName) {
        for (const TStringBuf root : {"", "."}) {
            const TLabels<TLabel> labels = TDnsName(root).Labels();
            UNIT_ASSERT_VALUES_EQUAL(labels.ToNameNoDot(), ".");
            UNIT_ASSERT_VALUES_EQUAL(labels.ToNameWithDot(), ".");
        }

        const TLabels<TLabel> labels = TDnsName("A.yanDeX.Net").Labels();
        UNIT_ASSERT_VALUES_EQUAL(labels.ToNameNoDot(), "A.yanDeX.Net");
        UNIT_ASSERT_VALUES_EQUAL(labels.ToNameWithDot(), "A.yanDeX.Net.");
    }

    Y_UNIT_TEST(Normalize) {
        for (const TStringBuf root : {"", "."}) {
            TLabels<TLabel> labels = TDnsName(root).Labels();
            labels.Normalize();
            UNIT_ASSERT_VALUES_EQUAL(labels.ToNameNoDot(), ".");
            UNIT_ASSERT_VALUES_EQUAL(labels.ToNameWithDot(), ".");
        }

        TLabels<TLabel> labels = TDnsName("A.yanDeX.Net").Labels();
        labels.Normalize();
        UNIT_ASSERT_VALUES_EQUAL(labels.ToNameNoDot(), "a.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(labels.ToNameWithDot(), "a.yandex.net.");
    }

    Y_UNIT_TEST(Reverse) {
        for (const TStringBuf root : {"", "."}) {
            TLabels<TLabel> labels = TDnsName(root).Labels();
            labels.Reverse();
            UNIT_ASSERT_VALUES_EQUAL(labels.ToNameNoDot(), ".");
            UNIT_ASSERT_VALUES_EQUAL(labels.ToNameWithDot(), ".");
        }

        TLabels<TLabel> labels = TDnsName("A.yanDeX.Net").Labels();
        labels.Reverse();
        UNIT_ASSERT_VALUES_EQUAL(labels.ToNameNoDot(), "Net.yanDeX.A");
        UNIT_ASSERT_VALUES_EQUAL(labels.ToNameWithDot(), "Net.yanDeX.A.");
    }
}


Y_UNIT_TEST_SUITE(DnsName) {
    Y_UNIT_TEST(ValidateNameLength) {
        ValidateNameLength<TDnsName>();
    }

    Y_UNIT_TEST(ValidateLabelsLength) {
        ValidateLabelsLength<TDnsName>();
    }

    Y_UNIT_TEST(Normalize) {
        TDnsName name = TDnsName("A.yanDeX.Net").Normalize();
        TLabels<TLabel> labels = name.Labels();
        UNIT_ASSERT_VALUES_EQUAL(TStringBuf{labels[0]}, "a");
        UNIT_ASSERT_VALUES_EQUAL(TStringBuf{labels[1]}, "yandex");
        UNIT_ASSERT_VALUES_EQUAL(TStringBuf{labels[2]}, "net");
    }

    Y_UNIT_TEST(Compare) {
        ValidateEquality<TDnsName>();
        ValidateComparison<TDnsName>();

        ValidateEquality<TDnsName, TStringBuf, TDnsNameView, TStringBuf>();
        ValidateComparison<TDnsName, TStringBuf, TDnsNameView, TStringBuf>();

        ValidateEquality<TDnsName, TStringBuf, TDnsNameRef, TDnsName>();
        ValidateComparison<TDnsName, TStringBuf, TDnsNameRef, TDnsName>();
    }

    Y_UNIT_TEST(Hashable) {
        // Trivial
        {
            TDnsName nameV1("fqdn.yandex.net");
            UNIT_ASSERT_VALUES_EQUAL(
                nameV1.Hash(),
                THash<TDnsName>()(nameV1)
            );

            TDnsName nameV2("FQDN.yanDEX.neT");
            UNIT_ASSERT_VALUES_EQUAL(
                THash<TDnsName>()(nameV1),
                THash<TDnsName>()(nameV2)
            );

            TDnsName rootV1(".");
            TDnsName rootV2("");
            UNIT_ASSERT_VALUES_EQUAL(
                THash<TDnsName>()(rootV1),
                THash<TDnsName>()(rootV2)
            );

            UNIT_ASSERT_VALUES_UNEQUAL(
                THash<TDnsName>()(rootV1),
                THash<TDnsName>()(nameV1)
            );
        }

        // HashMap works
        {
            THashMap<TDnsName, size_t> idx;
            idx["."] = 0;
            idx[""] = 0;
            idx["a.yANdex.Net"] = 1;

            UNIT_ASSERT_VALUES_EQUAL(idx.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(idx.at("."), 0);
            UNIT_ASSERT_VALUES_EQUAL(idx.at("A.Yandex.neT"), 1);

            idx.erase(".");
            idx["yanDex.NEt"] = 1;
            idx["A.yanDex.NET"] = 2;

            UNIT_ASSERT_VALUES_EQUAL(idx.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(idx.at("YandEx.NeT"), 1);
            UNIT_ASSERT_VALUES_EQUAL(idx.at("a.YandEx.NeT"), 2);
        }
    }

    Y_UNIT_TEST(ToString) {
        for (const TStringBuf root : {"", "."}) {
            UNIT_ASSERT_VALUES_EQUAL(TDnsName(root).ToStringNoDot(), ".");
            UNIT_ASSERT_VALUES_EQUAL(TDnsName(root).ToStringWithDot(), ".");
        }

        UNIT_ASSERT_VALUES_EQUAL(TDnsName("A.yanDeX.Net").ToStringNoDot(), "A.yanDeX.Net");
        UNIT_ASSERT_VALUES_EQUAL(TDnsName("A.yanDeX.Net").ToStringWithDot(), "A.yanDeX.Net.");

        UNIT_ASSERT_VALUES_EQUAL(TDnsName("A.yanDeX.Net").Normalize().ToStringNoDot(), "a.yandex.net");
        UNIT_ASSERT_VALUES_EQUAL(TDnsName("A.yanDeX.Net").Normalize().ToStringWithDot(), "a.yandex.net.");
    }

    Y_UNIT_TEST(ToReversedString) {
        for (const TStringBuf root : {"", "."}) {
            UNIT_ASSERT_VALUES_EQUAL(TDnsName(root).ToReversedStringNoDot(), ".");
        }

        UNIT_ASSERT_VALUES_EQUAL(TDnsName("A.yanDeX.Net").ToReversedStringNoDot(), "Net.yanDeX.A");
        UNIT_ASSERT_VALUES_EQUAL(TDnsName("A.yanDeX.Net").Normalize().ToReversedStringNoDot(), "net.yandex.a");
    }
}

Y_UNIT_TEST_SUITE(DnsNameView) {
    Y_UNIT_TEST(ConstructFromDnsName) {
        static_assert(!std::is_assignable<TDnsNameView, TDnsName&>::value,
            "Please, do not try to create TDnsNameView from TDnsName. Use TDnsNameRef instead");
        static_assert(!std::is_assignable<TDnsNameView, TDnsName&&>::value,
            "Please, do not try to create TDnsNameView from TDnsName. Use TDnsNameRef instead");
    }

    Y_UNIT_TEST(ValidateNameLength) {
        ValidateNameLength<TDnsNameView>();
    }

    Y_UNIT_TEST(ValidateLabelsLength) {
        ValidateLabelsLength<TDnsNameView>();
    }

    Y_UNIT_TEST(Compare) {
        ValidateEquality<TDnsNameView>();
        ValidateComparison<TDnsNameView>();

        ValidateEquality<TDnsNameView, TStringBuf, TDnsName, TStringBuf>();
        ValidateComparison<TDnsNameView, TStringBuf, TDnsName, TStringBuf>();

        ValidateEquality<TDnsNameView, TStringBuf, TDnsNameRef, TDnsName>();
        ValidateComparison<TDnsNameView, TStringBuf, TDnsNameRef, TDnsName>();
    }

    Y_UNIT_TEST(IsPartOf) {
        UNIT_ASSERT(!TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("a.a.yandex.net")));
        UNIT_ASSERT(TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("a.yandex.net")));
        UNIT_ASSERT(TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("yandex.net")));
        UNIT_ASSERT(TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("net")));
        UNIT_ASSERT(TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView(".")));
        UNIT_ASSERT(TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("")));

        UNIT_ASSERT(!TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("b.yandex.net")));
        UNIT_ASSERT(!TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("yandex.ru")));
        UNIT_ASSERT(!TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("ya.net")));

        UNIT_ASSERT(!TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("yyandex.net")));
        UNIT_ASSERT(!TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("dex.net")));
        UNIT_ASSERT(!TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("t")));
    }

    Y_UNIT_TEST(IsPartOfCaseInsensitive) {
        UNIT_ASSERT(!TDnsNameView("a.yandeX.net").IsPartOf(TDnsNameView("a.a.yaNdex.net")));
        UNIT_ASSERT(TDnsNameView("a.yandex.nET").IsPartOf(TDnsNameView("a.yaNDex.net")));
        UNIT_ASSERT(TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("yaNdex.Net")));
        UNIT_ASSERT(TDnsNameView("A.YANDEX.NET").IsPartOf(TDnsNameView("yandex.net")));
        UNIT_ASSERT(TDnsNameView("a.yANdex.nEt").IsPartOf(TDnsNameView("NEt")));
        UNIT_ASSERT(TDnsNameView("A.yaNdeX.nEt").IsPartOf(TDnsNameView(".")));
        UNIT_ASSERT(TDnsNameView("a.yaNDex.neT").IsPartOf(TDnsNameView("")));

        UNIT_ASSERT(!TDnsNameView("A.yaNDex.net").IsPartOf(TDnsNameView("B.yaNdex.net")));
        UNIT_ASSERT(!TDnsNameView("a.yAndEx.net").IsPartOf(TDnsNameView("yandeX.ru")));
        UNIT_ASSERT(!TDnsNameView("a.yandex.NET").IsPartOf(TDnsNameView("ya.NET")));

        UNIT_ASSERT(!TDnsNameView("a.yandex.net").IsPartOf(TDnsNameView("Yyandex.net")));
        UNIT_ASSERT(!TDnsNameView("a.yanDEX.NET").IsPartOf(TDnsNameView("dex.net")));
        UNIT_ASSERT(!TDnsNameView("A.YANdex.net").IsPartOf(TDnsNameView("t")));
    }
}

Y_UNIT_TEST_SUITE(DnsNameRef) {
    Y_UNIT_TEST(ConstructFromDnsName) {
        // from lvalue: OK
        {
            TDnsName name("a.yandex.test");
            TDnsNameRef ref(name);
        }

        // from rvalue: FAIL
        static_assert(!std::is_assignable<TDnsNameRef, TDnsName&&>::value);
    }

    Y_UNIT_TEST(Compare) {
        ValidateEquality<TDnsNameRef, TDnsName>();
        ValidateComparison<TDnsNameRef, TDnsName>();

        ValidateEquality<TDnsNameRef, TDnsName, TDnsName, TStringBuf>();
        ValidateComparison<TDnsNameRef, TDnsName, TDnsName, TStringBuf>();

        ValidateEquality<TDnsNameRef, TDnsName, TDnsNameView, TStringBuf>();
        ValidateComparison<TDnsNameRef, TDnsName, TDnsNameView, TStringBuf>();
    }

    Y_UNIT_TEST(CheckReferences) {
        TDnsName nameA("a.Yandex.Ru");
        TDnsNameRef nameARef(nameA);
        TDnsNameRef nameARefCopy = nameARef;

        TDnsName nameB("B.yandeX.tEst");

        UNIT_ASSERT(nameARef.IsRefTo(nameA));
        UNIT_ASSERT(nameARefCopy.IsRefTo(nameA));
        UNIT_ASSERT(!nameARef.IsRefTo(nameB));
        UNIT_ASSERT(!nameARefCopy.IsRefTo(nameB));

        nameA.Normalize();
        UNIT_ASSERT(nameARef.IsRefTo(nameA));
        UNIT_ASSERT(nameARefCopy.IsRefTo(nameA));
        UNIT_ASSERT(!nameARef.IsRefTo(nameB));
        UNIT_ASSERT(!nameARefCopy.IsRefTo(nameB));

        // references to nameA are invalid after this
        // do not use them
        nameA = nameB;
    }

    Y_UNIT_TEST(CompareWithReferencedHolder) {
        TDnsName name("a.Yandex.Ru");
        TDnsNameRef nameRef(name);

        UNIT_ASSERT_EQUAL(name, nameRef);
        UNIT_ASSERT_EQUAL(nameRef, name);
    }

    Y_UNIT_TEST(Root) {
        for (const auto root : {"", "."}) {
            const TDnsName name(root);
            const TDnsNameRef nameRef(name);
            UNIT_ASSERT(nameRef.IsRoot());
            UNIT_ASSERT_EQUAL(nameRef, TDnsNameRef::Root());
        }
    }

    Y_UNIT_TEST(ChopOff) {
        TDnsName name("Box.Pod.Yandex.Net");

        TDnsNameRef nameRef(name.Normalize());
        UNIT_ASSERT_EQUAL(nameRef, TDnsNameView("box.pod.yandex.net"));
        UNIT_ASSERT(nameRef.ChopOff());
        UNIT_ASSERT_EQUAL(nameRef, TDnsNameView("pod.yandex.net"));
        UNIT_ASSERT(nameRef.ChopOff());
        UNIT_ASSERT_EQUAL(nameRef, TDnsNameView("yandex.net"));
        UNIT_ASSERT(nameRef.ChopOff());
        UNIT_ASSERT_EQUAL(nameRef, TDnsNameView("net"));
        UNIT_ASSERT(nameRef.ChopOff());
        UNIT_ASSERT_EQUAL(nameRef, TDnsNameView("."));
        UNIT_ASSERT(!nameRef.ChopOff());
        UNIT_ASSERT_EQUAL(nameRef, TDnsNameView("."));
    }
}
