#pragma once

#include "consts.h"

#include <util/digest/sequence.h>
#include <util/generic/algorithm.h>
#include <util/generic/array_ref.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/generic/yexception.h>
#include <util/str_stl.h>
#include <util/string/join.h>
#include <util/string/split.h>

namespace NYpDns {

namespace NPrivate {

struct TCiCharTraits: public std::char_traits<char> {
    static bool eq(char c1, char c2) {
        return std::tolower(c1) == std::tolower(c2);
    }

    static bool lt(char c1, char c2) {
         return std::tolower(c1) < std::tolower(c2);
    }

    static int compare(const char* s1, const char* s2, std::size_t n) {
        return strnicmp(s1, s2, n);
    }

    static const char* find(const char* s, std::size_t n, char a) {
        const char la = std::tolower(a);
        for (std::size_t i = 0; i < n; ++i) {
            if (std::tolower(s[i]) == la) {
                return s;
            }
        }
        return nullptr;
    }
};

} // namepsace NPrivate

// TODO: Use library/cpp/case_insensitive_string
using TCiString = TBasicString<char, NPrivate::TCiCharTraits>;
using TCiStringBuf = TBasicStringBuf<char, NPrivate::TCiCharTraits>;

} // namespace NYpDns

template <>
struct THash<NYpDns::TCiStringBuf> {
    size_t operator()(NYpDns::TCiStringBuf s) const noexcept;
};

template <>
struct THash<NYpDns::TCiString> : THash<NYpDns::TCiStringBuf> {};

namespace NYpDns {

class TLabel: public TCiString {
public:
    using TBase = TCiString;

    using TBase::TBase;
};

inline void AppendToString(TString& dst, const NYpDns::TLabel& t) {
    dst.append(t);
}

class TLabelView: public TCiStringBuf {
public:
    using TBase = TCiStringBuf;

    using TBase::TBase;

    explicit TLabelView(const TStringBuf& s) noexcept
        : TBase(s.begin(), s.end())
    {
    }
};

template <typename TLabelType>
class TLabels: public TVector<TLabelType> {
public:
    using TBase = TVector<TLabelType>;
    using TSelf = TLabels;

    TLabels(TStringBuf name) {
        AddFromString(name);
    }

    inline TLabels(const TSelf& other)
        : TBase(other)
    {
    }

    inline TLabels(TSelf&& other) noexcept
        : TBase(std::forward<TSelf>(other))
    {
    }

    inline TSelf& operator=(const TSelf& other) {
        TBase::operator=(other);
        return *this;
    }

    inline TSelf& operator=(TSelf&& other) noexcept {
        TBase::operator=(std::forward<TSelf>(other));
        return *this;
    }

    TLabels& Normalize() {
        for (TLabelType& label : *this) {
            label.to_lower();
        }
        return *this;
    }

    TLabels& Reverse() {
        ::Reverse(this->begin(), this->end());
        return *this;
    }

    size_t Hash() const {
        return TRangeHash<>()(*this);
    }

    TString Join(char delim = '.') const {
        return JoinSeq(delim, *this);
    }

    TString ToNameNoDot() const {
        if (Y_UNLIKELY(this->empty())) {
            return ".";
        }
        return this->Join();
    }

    TString ToNameWithDot() const {
        TString result = ToNameNoDot();
        if (Y_UNLIKELY(!this->empty())) {
            result += ".";
        }
        return result;
    }

private:
    void AddFromString(TStringBuf name) {
        name.ChopSuffix(".");
        if (Y_UNLIKELY(name.size() > MAX_NAME_LENGTH)) {
            ythrow yexception() << "too long DNS name " << name << ": " << name.size() << " > " << MAX_NAME_LENGTH;
        }
        if (Y_LIKELY(!name.empty())) {
            for (const TStringBuf label : StringSplitter(name).Split('.')) {
                if (Y_UNLIKELY(label.empty())) {
                    ythrow yexception() << "empty label in DNS name " << name;
                }
                if (Y_UNLIKELY(label.size() > MAX_LABEL_LENGTH)) {
                    ythrow yexception() << "too long label in DNS name " << name << ": " << label.size() << " > " << MAX_LABEL_LENGTH;
                }
                this->emplace_back(label);
            }
        }
        this->shrink_to_fit();
    }
};

template <typename TLabelsType>
using TLabelsRef = TArrayRef<const TLabelsType>;

template <typename T>
concept LabelType = (
    std::same_as<std::remove_cvref_t<T>, TLabel> ||
    std::same_as<std::remove_cvref_t<T>, TLabelView>
);

template <typename Container>
concept LabelsContainer = requires(const Container& c) {
    requires LabelType<typename Container::value_type>;
    { c.size() };
    { c.cbegin() };
    { c.cend() };
};

template <LabelsContainer TLhsContainer, LabelsContainer TRhsContainer>
bool operator==(const TLhsContainer& lhs, const TRhsContainer& rhs) {
    return lhs.size() == rhs.size() && std::equal(lhs.cbegin(), lhs.cend(), rhs.cbegin());
}

template <LabelsContainer TLhsContainer, LabelsContainer TRhsContainer>
std::strong_ordering operator<=>(const TLhsContainer& lhs, const TRhsContainer& rhs) {
    auto lhsIt = lhs.cbegin();
    auto rhsIt = rhs.cbegin();
    for (; lhsIt != lhs.cend() && rhsIt != rhs.cend(); ++lhsIt, ++rhsIt) {
        if (auto cmp = lhsIt->compare(*rhsIt); cmp < 0) {
            return std::strong_ordering::less;
        } else if (cmp > 0) {
            return std::strong_ordering::greater;
        }
    }
    return lhs.size() <=> rhs.size();
}

} // namespace NYpDns

template <>
struct THash<NYpDns::TLabel> : THash<NYpDns::TCiString> {};

template <>
struct THash<NYpDns::TLabelView> : THash<NYpDns::TCiStringBuf> {};

template <typename TLabelType>
struct THash<NYpDns::TLabels<TLabelType>> {
    size_t operator()(const NYpDns::TLabels<TLabelType>& labels) const {
        return labels.Hash();
    }
};
