#include "client.h"

#include <infra/libs/yp_dns/dynamic_zones/zones_manager_service/client/client.h>

#include <infra/libs/yp_dns/dynamic_zones/services/protos/api/bridge.grpc.pb.h>

#include <infra/libs/yp_dns/dynamic_zones/helpers/yt/yt.h>

#include <infra/libs/leading_invader/leading_invader.h>

#include <infra/libs/background_thread/background_thread.h>
#include <infra/libs/outcome/result.h>
#include <infra/libs/service_iface/reply.h>
#include <infra/libs/service_iface/request.h>
#include <infra/libs/service_iface/str_iface.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/fwd.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <library/cpp/logger/global/global.h>

#include <contrib/libs/grpc/include/grpc++/channel.h>
#include <contrib/libs/grpc/include/grpc++/create_channel.h>

#include <util/datetime/base.h>
#include <util/generic/yexception.h>
#include <util/system/backtrace.h>
#include <util/system/hostname.h>
#include <util/system/rwlock.h>

namespace NYpDns::NDynamicZones {

////////////////////////////////////////////////////////////////////////////////

TRegistrationOptions::TRegistrationOptions(TRegistrationConfig config, TString instanceName, TString ytToken)
    : Enabled(config.GetEnabled())
    , YtProxy(*config.MutableYtProxy())
    , YtToken(std::move(ytToken))
    , InstancesPath(std::move(*config.MutableInstancesPath()))
    , InstanceName(!instanceName.empty() ? std::move(instanceName) : HostName())
{
}

////////////////////////////////////////////////////////////////////////////////

struct TRegisterError {
    TString Reason;
};

class TInstanceRegistrator {
public:
    TInstanceRegistrator(TRegistrationOptions options)
        : Options_(std::move(options))
        , YtClient_(NYT::CreateClient(
            Options_.YtProxy,
            NYT::TCreateClientOptions()
                .Token(Options_.YtToken)
        ))
        , LockPath_(NYT::JoinYPaths(Options_.InstancesPath, Options_.InstanceName))
    {
        LeadingInvaderConfig_.SetProxy(Options_.YtProxy);
        LeadingInvaderConfig_.SetToken(Options_.YtToken);
        LeadingInvaderConfig_.SetPath(LockPath_);
        LeadingInvaderConfig_.SetTimeout("10s");
        LeadingInvaderConfig_.SetRetryAcquireLockInterval("10ms");
    }

    void Register(NInfra::TLogFramePtr logFrame) {
        if (Options_.Enabled) {
            TWriteGuard guard(LeadingInvaderMutex_);
            NYtHelpers::CreateCypressDocumentIfMissing(YtClient_, LockPath_, logFrame);
            LeadingInvader_ = NInfra::NLeadingInvader::CreateLeadingInvader(LeadingInvaderConfig_);
        }
    }

    TExpected<void, TRegisterError> EnsureRegistered() {
        if (!Options_.Enabled) {
            return TExpected<void, TRegisterError>::DefaultSuccess();
        }

        TReadGuard guard(LeadingInvaderMutex_);
        if (!LeadingInvader_) {
            return TRegisterError{"Lock is not initialized"};
        }
        TExpected<void, NInfra::NLeadingInvader::TError> status = LeadingInvader_->EnsureLeading();
        if (!status) {
            return TRegisterError{status.Error().Reason};
        }

        return TExpected<void, TRegisterError>::DefaultSuccess();
    }

private:
    const TRegistrationOptions Options_;
    const NYT::IClientPtr YtClient_;
    const NYT::TYPath LockPath_;

    TRWMutex LeadingInvaderMutex_;
    NInfra::NLeadingInvader::TConfig LeadingInvaderConfig_;
    NInfra::NLeadingInvader::TLeadingInvaderHolder LeadingInvader_;
};

////////////////////////////////////////////////////////////////////////////////

template <typename TGrpcService>
class TGrpcServiceRequester {
public:
    TGrpcServiceRequester(
        const TStringBuf address,
        const TDuration defaultTimeout = TDuration::Seconds(10)
    )
        : Channel_(grpc::CreateChannel(TString{address}, grpc::InsecureChannelCredentials()))
        , Stub_(TGrpcService::NewStub(Channel_).release())
        , DefaultTimeout_(defaultTimeout)
    {
    }

    template <typename TRequest, typename TReply, typename TCallback>
    void Request(
        NInfra::TRequestPtr<TRequest> request,
        NInfra::TReplyPtr<TReply> reply,
        TCallback&& callback,
        TDuration timeout = {}
    ) const {
        grpc::ClientContext context;
        for (const auto& [key, value] : request->Attributes()) {
            context.AddMetadata(key, value);
        }

        if (!timeout) {
            timeout = DefaultTimeout_;
        }

        context.set_deadline(std::chrono::system_clock::now() + std::chrono::milliseconds(timeout.MilliSeconds()));

        TReply result;
        if (grpc::Status status = ((*Stub_).*callback)(&context, request->Get(), &result); !status.ok()) {
            ythrow yexception() << "Code: " << static_cast<ui64>(status.error_code()) << "\nMessage: " << status.error_message();
        }

        for (const auto& [key, value] : context.GetServerInitialMetadata()) {
            reply->SetAttribute(TString{key.data(), key.size()}, TString{value.data(), value.size()});
        }
        for (const auto& [key, value] : context.GetServerTrailingMetadata()) {
            reply->SetAttribute(TString{key.data(), key.size()}, TString{value.data(), value.size()});
        }

        reply->Set(result);
    }

    template <typename TRequest, typename TReply, typename TCallback>
    void Request(
        TRequest request,
        NInfra::TReplyPtr<TReply> reply,
        TCallback&& callback,
        const TDuration timeout = {}
    ) const {
        return Request(
            NInfra::RequestPtr<NInfra::TProtoRequest<TRequest>>("", std::move(request), NInfra::TAttributes()),
            reply,
            std::move(callback),
            timeout
        );
    }

private:
    const std::shared_ptr<grpc::Channel> Channel_;
    const THolder<typename TGrpcService::Stub> Stub_;
    const TDuration DefaultTimeout_;
};

////////////////////////////////////////////////////////////////////////////////

class TConfigurationReporter {
public:
    TConfigurationReporter(TReportConfigurationConfig config, IServiceClient& serviceClient)
        : Config_(config)
        , ServiceClient_(serviceClient)
        , ReporterThread_(MakeHolder<NInfra::TBackgroundThread>(
            [this] { Report(); },
            TDuration::Parse(Config_.GetReportPeriod())
        ))
    {
    }

    void Start() {
        if (Config_.GetEnabled()) {
            ReporterThread_->Start();
        }
    }

    void Stop() {
        ReporterThread_->Stop();
    }

    void Report() const {
        try {
            ServiceClient_.ReportConfiguration(TDuration::Parse(Config_.GetTimeout()));
        } catch (...) {
            // TODO: log error to log file
            ERROR_LOG << "Failed to report configuration: " << CurrentExceptionMessage() << "\n"
                      << TBackTrace::FromCurrentException().PrintToString() << Endl;
        }
    }

private:
    const TReportConfigurationConfig Config_;
    const IServiceClient& ServiceClient_;
    THolder<NInfra::IBackgroundThread> ReporterThread_;
};

////////////////////////////////////////////////////////////////////////////////

class TBridgeServiceClient::TImpl {
public:
    TImpl(
        TBridgeServiceClient& parent,
        TBridgeServiceClientConfig config,
        std::function<NBridgeService::NApi::TReqReportConfiguration()>&& configurationReportRequestGenerator
    )
        : Config_(std::move(config))
        , StateServiceRequester_(Config_.GetStateServiceClientConfig().GetAddress())
        , ConfigurationReportRequestGenerator_(std::move(configurationReportRequestGenerator))
        , InstanceRegistrator_(TRegistrationOptions(
            Config_.GetStateServiceClientConfig().GetRegistration(),
            Config_.GetStateServiceClientConfig().GetInstanceName(),
            /* ytToken */ ""
        ))
        , ConfigurationReporter_(Config_.GetStateServiceClientConfig().GetReportConfiguration(), parent)
    {
        Y_ENSURE(ConfigurationReportRequestGenerator_);
    }

    void StartReportConfiguration() {
        ConfigurationReporter_.Start();
    }

    void RegisterInstance(NInfra::TLogFramePtr logFrame) {
        InstanceRegistrator_.Register(logFrame);
    }

    TServiceType ServiceType() const {
        return "BRIDGE";
    }

    void ReportConfiguration(const TDuration& timeout) const {
        NBridgeService::NApi::TReqReportConfiguration request = ConfigurationReportRequestGenerator_();
        NBridgeService::NApi::TRspReportConfiguration response;

        NInfra::TAttributes responseAttributes;
        NInfra::TReplyPtr<NBridgeService::NApi::TRspReportConfiguration> reply = NInfra::ReplyPtr<NInfra::TProtoReply<NBridgeService::NApi::TRspReportConfiguration>>(
            response,
            responseAttributes
        );
        StateServiceRequester_.Request(request, reply, &NBridgeService::NApi::TStateService::Stub::ReportConfiguration, timeout);
    }

private:
    TBridgeServiceClientConfig Config_;
    TGrpcServiceRequester<NBridgeService::NApi::TStateService> StateServiceRequester_;
    std::function<NBridgeService::NApi::TReqReportConfiguration()> ConfigurationReportRequestGenerator_;

    TInstanceRegistrator InstanceRegistrator_;
    TConfigurationReporter ConfigurationReporter_;
};

TBridgeServiceClient::TBridgeServiceClient(
    TBridgeServiceClientConfig config,
    std::function<NBridgeService::NApi::TReqReportConfiguration()>&& configurationReportRequestGenerator
)
    : Impl_(MakeHolder<TImpl>(*this, std::move(config), std::move(configurationReportRequestGenerator)))
{
}

TBridgeServiceClient::~TBridgeServiceClient() = default;

void TBridgeServiceClient::StartReportConfiguration() {
    return Impl_->StartReportConfiguration();
}

void TBridgeServiceClient::RegisterInstance(NInfra::TLogFramePtr logFrame) {
    return Impl_->RegisterInstance(logFrame);
}

TServiceType TBridgeServiceClient::ServiceType() const {
    return Impl_->ServiceType();
}

void TBridgeServiceClient::ReportConfiguration(const TDuration& timeout) const {
    Impl_->ReportConfiguration(timeout);
}

////////////////////////////////////////////////////////////////////////////////

TServiceType TReplicatorServiceClient::ServiceType() const {
    return "REPLICATOR";
}

////////////////////////////////////////////////////////////////////////////////

TServiceType TYpDnsExportServiceClient::ServiceType() const {
    return "YP-DNS-EXPORT";
}

////////////////////////////////////////////////////////////////////////////////

class TClient::TImpl {
public:
    TImpl(TClientConfig config, IServiceClientPtr serviceClient)
        : Config_(std::move(config))
        , ZonesManagerClient_(TZonesManagerServiceClientOptions()
            .SetAddress(Config_.GetAddress())
            .SetTimeout(TDuration::Parse(Config_.GetTimeout()))
        )
        , ServiceClient_(serviceClient)
    {
    }

    TVector<TZone> ListZones() const {
        NApi::TRspListZones listZonesResponse = ZonesManagerClient_.ListZones(ServiceClient_->ServiceType());
        Y_ENSURE(listZonesResponse.status() == NApi::TRspListZones::OK);

        TVector<TZone> result;
        result.reserve(listZonesResponse.zones().size());
        for (NApi::TZone& zone : *listZonesResponse.mutable_zones()) {
            result.emplace_back(std::move(*zone.mutable_config()));
        }

        return result;
    }

private:
    const TClientConfig Config_;
    TZonesManagerServiceClient ZonesManagerClient_;
    IServiceClientPtr ServiceClient_;
};

////////////////////////////////////////////////////////////////////////////////

TClient::TClient(TClientConfig config, IServiceClientPtr serviceClient)
    : Impl_(MakeHolder<TImpl>(std::move(config), serviceClient))
{
}

TClient::~TClient() = default;

TVector<TZone> TClient::ListZones() const {
    return Impl_->ListZones();
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NDynamicZones
