#include "make_log_events.h"

#include <library/cpp/yson/node/node_io.h>

#include <util/generic/guid.h>

namespace NYpDns::NYtHelpers {

////////////////////////////////////////////////////////////////////////////////

NEventlog::TYtAttributeFilter MakeYtAttributeFilterEvent(
    const NYT::TAttributeFilter& attributeFilter
) {
    NEventlog::TYtAttributeFilter event;
    event.MutableAttributes()->Reserve(attributeFilter.Attributes_.size());
    for (const TString& attribute : attributeFilter.Attributes_) {
        event.AddAttributes(attribute);
    }
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TYtLock MakeYtLockEvent(
    NYT::ITransactionPtr transaction,
    const NYT::TYPath& path,
    NYT::ELockMode mode,
    const NYT::TLockOptions& options
) {
    NEventlog::TYtLock event;
    event.SetTransactionId(GetGuidAsString(transaction->GetId()));
    event.SetPath(path);
    event.SetMode(ToString(mode));
    event.SetWaitable(options.Waitable_);
    if (Y_UNLIKELY(options.AttributeKey_)) {
        event.SetAttributeKey(*options.AttributeKey_);
    }
    if (Y_UNLIKELY(options.ChildKey_)) {
        event.SetChildKey(*options.ChildKey_);
    }
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TYtUpdateWithLockOptions MakeUpdateWithLockOptionsEvent(
    const TUpdateWithLockOptions& options
) {
    NEventlog::TYtUpdateWithLockOptions event;
    event.SetCreateIfMissing(options.CreateIfMissing);
    event.SetWaitableLock(options.WaitableLock);
    return event;
}

NEventlog::TYtUpdateWithLock MakeYtUpdateWithLockEvent(
    const NYT::TYPath& lockPath,
    const NYT::TYPath& path,
    const NYT::TNode& data,
    const TUpdateWithLockOptions& options
) {
    NEventlog::TYtUpdateWithLock event;
    event.SetLockPath(lockPath);
    event.SetPath(path);
    event.SetData(NYT::NodeToYsonString(data));
    *event.MutableOptions() = MakeUpdateWithLockOptionsEvent(options);
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TYtUpdateOptions MakeYtUpdateOptionsEvent(
    const TUpdateOptions& options
) {
    NEventlog::TYtUpdateOptions event;
    event.SetCreateIfMissing(options.CreateIfMissing);
    return event;
}

NEventlog::TYtUpdateWithoutLock MakeYtUpdateWithoutLockEvent(
    const NYT::TYPath& path,
    const NYT::TNode& data,
    const TUpdateOptions& options
) {
    NEventlog::TYtUpdateWithoutLock event;
    event.SetPath(path);
    event.SetData(NYT::NodeToYsonString(data));
    *event.MutableOptions() = MakeYtUpdateOptionsEvent(options);
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TYtSet MakeYtSetEvent(
    const NYT::TYPath& path,
    const NYT::TNode& data
) {
    NEventlog::TYtSet event;
    event.SetPath(path);
    event.SetData(NYT::NodeToYsonString(data));
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TYtCreateOptions MakeYtCreateOptionsEvent(const NYT::TCreateOptions& options) {
    NEventlog::TYtCreateOptions event;
    event.SetRecursive(options.Recursive_);
    event.SetIgnoreExisting(options.IgnoreExisting_);
    event.SetForce(options.Force_);
    if (options.Attributes_) {
        event.SetAttributes(NYT::NodeToYsonString(*options.Attributes_));
    }
    return event;
}

NEventlog::TYtCreate MakeYtCreateEvent(
    const NYT::TYPath& path,
    const NYT::ENodeType& nodeType,
    const NYT::TCreateOptions& options
) {
    NEventlog::TYtCreate event;
    event.SetPath(path);
    event.SetNodeType(ToString(nodeType));
    *event.MutableOptions() = MakeYtCreateOptionsEvent(options);
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TYtGetOptions MakeYtGetOptionsEvent(
    const NYT::TGetOptions& options
) {
    NEventlog::TYtGetOptions event;
    if (options.AttributeFilter_) {
        *event.MutableAttributeFilter() = MakeYtAttributeFilterEvent(*options.AttributeFilter_);
    }
    if (Y_UNLIKELY(options.MaxSize_)) {
        event.SetMaxSize(*options.MaxSize_);
    }
    if (options.ReadFrom_) {
        event.SetReadFrom(ToString(*options.ReadFrom_));
    }
    return event;
}

NEventlog::TYtGet MakeYtGetEvent(
    const NYT::TYPath& path,
    const NYT::TGetOptions& options
) {
    NEventlog::TYtGet event;
    event.SetPath(path);
    *event.MutableOptions() = MakeYtGetOptionsEvent(options);
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TYtListOptions MakeYtListOptionsEvent(
    const NYT::TListOptions& options
) {
    NEventlog::TYtListOptions event;
    if (options.AttributeFilter_) {
        *event.MutableAttributeFilter() = MakeYtAttributeFilterEvent(*options.AttributeFilter_);
    }
    if (Y_UNLIKELY(options.MaxSize_)) {
        event.SetMaxSize(*options.MaxSize_);
    }
    return event;
}

NEventlog::TYtList MakeYtListEvent(
    const NYT::TYPath& path,
    const NYT::TListOptions& options
) {
    NEventlog::TYtList event;
    event.SetPath(path);
    *event.MutableOptions() = MakeYtListOptionsEvent(options);
    return event;
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NYtHelpers
