#include "yt.h"

#include "make_log_events.h"

#include <infra/libs/yp_dns/dynamic_zones/helpers/yt/events/events_decl.ev.pb.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <util/generic/guid.h>

namespace NYpDns::NYtHelpers {

////////////////////////////////////////////////////////////////////////////////

namespace {

NYT::ITransactionPtr StartTransaction(NYT::IClientBasePtr ytClient, NInfra::TLogFramePtr logFrame) {
    INFRA_LOG_INFO(NEventlog::TYtStartTransaction());

    try {
        NYT::ITransactionPtr transaction = ytClient->StartTransaction();
        INFRA_LOG_INFO(NEventlog::TYtStartTransactionSuccess(GetGuidAsString(transaction->GetId())));
        return transaction;
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtStartTransactionFailure(CurrentExceptionMessage()));
        throw;
    }
}

void CommitTransaction(NYT::ITransactionPtr transaction, NInfra::TLogFramePtr logFrame) {
    INFRA_LOG_INFO(NEventlog::TYtCommitTransaction(GetGuidAsString(transaction->GetId())));

    try {
        transaction->Commit();
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtCommitTransactionFailure(CurrentExceptionMessage()));
        throw;
    }
    INFRA_LOG_INFO(NEventlog::TYtCommitTransactionSuccess());
}

NYT::ILockPtr Lock(
    NYT::ITransactionPtr transaction,
    const NYT::TYPath& path,
    NYT::ELockMode mode,
    const NYT::TLockOptions& options,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(MakeYtLockEvent(transaction, path, mode, options));

    try {
        NYT::ILockPtr lock = transaction->Lock(path, mode, options);
        INFRA_LOG_INFO(NEventlog::TYtLockSuccess(GetGuidAsString(lock->GetId())));
        return lock;
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtLockFailure(CurrentExceptionMessage()));
        throw;
    }
}

void WaitForLockAcquired(NYT::ILockPtr lock, NInfra::TLogFramePtr logFrame) {
    INFRA_LOG_INFO(NEventlog::TYtWaitForLockAcquired(GetGuidAsString(lock->GetId())));

    try {
        lock->Wait();
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtWaitForLockAcquiredFailure(CurrentExceptionMessage()));
        throw;
    }
    INFRA_LOG_INFO(NEventlog::TYtWaitForLockAcquiredSuccess());
}

void Set(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    const NYT::TNode& data,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(MakeYtSetEvent(path, data));

    try {
        ytClient->Set(path, data);
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtSetFailure(CurrentExceptionMessage()));
        throw;
    }
    INFRA_LOG_INFO(NEventlog::TYtSetSuccess());
}

void Create(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NYT::ENodeType nodeType,
    const NYT::TCreateOptions& options,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(MakeYtCreateEvent(path, nodeType, options));

    try {
        ytClient->Create(path, nodeType, options);
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtCreateFailure(CurrentExceptionMessage()));
        throw;
    }
    INFRA_LOG_INFO(NEventlog::TYtCreateSuccess());
}

NYT::TNode Get(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    const NYT::TGetOptions& options,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(MakeYtGetEvent(path, options));

    try {
        NYT::TNode result = ytClient->Get(path, options);
        INFRA_LOG_INFO(NEventlog::TYtGetSuccess());
        return result;
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtGetFailure(CurrentExceptionMessage()));
        throw;
    }
}

NYT::TNode Get(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NInfra::TLogFramePtr logFrame
) {
    return Get(ytClient, path, NYT::TGetOptions(), logFrame);
}

NYT::TNode::TListType List(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    const NYT::TListOptions& listOptions,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(MakeYtListEvent(path, listOptions));

    try {
        NYT::TNode::TListType result = ytClient->List(path, listOptions);
        INFRA_LOG_INFO(NEventlog::TYtListSuccess(result.size()));
        return result;
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtListFailure(CurrentExceptionMessage()));
        throw;
    }
}

NYT::TNode::TListType List(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NInfra::TLogFramePtr logFrame
) {
    return List(ytClient, path, NYT::TListOptions(), logFrame);
}

bool Exists(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(NEventlog::TYtExistsEvent(path));
    try {
        bool result = ytClient->Exists(path);
        INFRA_LOG_INFO(NEventlog::TYtExistsSuccess(path, result));
        return result;
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtExistsFailure(path, CurrentExceptionMessage()));
        throw;
    }
}

} // anonymous namespace

////////////////////////////////////////////////////////////////////////////////

void CreateCypressNodeIfMissing(
    const NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NYT::ENodeType nodeType,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(NEventlog::TYtCreateCypressNodeIfMissing(path, ToString(nodeType)));
    try {
        if (!Exists(ytClient, path, logFrame)) {
            Create(
                ytClient,
                path,
                nodeType, 
                NYT::TCreateOptions()
                    .Recursive(true)
                    .IgnoreExisting(true)
                    .Force(false),
                logFrame
            );
        }
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtCreateCypressNodeIfMissingFailure(CurrentExceptionMessage()));
        throw;
    }
    INFRA_LOG_INFO(NEventlog::TYtCreateCypressNodeIfMissingSuccess());
}

void CreateCypressMapNodeIfMissing(
    const NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NInfra::TLogFramePtr logFrame
) {
    CreateCypressNodeIfMissing(ytClient, path, NYT::NT_MAP, logFrame);
}

void CreateCypressDocumentIfMissing(
    const NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NInfra::TLogFramePtr logFrame
) {
    CreateCypressNodeIfMissing(ytClient, path, NYT::NT_DOCUMENT, logFrame);
}

////////////////////////////////////////////////////////////////////////////////

void UpdateWithLock(
    const NYT::IClientBasePtr ytClient,
    const NYT::TYPath& lockPath,
    const NYT::TYPath& path,
    const NYT::TNode& data,
    const TUpdateWithLockOptions& options,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(MakeYtUpdateWithLockEvent(lockPath, path, data, options));
    try {
        if (options.CreateIfMissing) {
            CreateCypressDocumentIfMissing(ytClient, lockPath, logFrame);
        }

        NYT::ITransactionPtr transaction = StartTransaction(ytClient, logFrame);
        NYT::ILockPtr lock = Lock(
            transaction,
            lockPath,
            NYT::LM_EXCLUSIVE,
            NYT::TLockOptions()
                .Waitable(options.WaitableLock),
            logFrame
        );
        WaitForLockAcquired(lock, logFrame);

        UpdateWithoutLock(
            transaction,
            path,
            data,
            TUpdateOptions()
                .SetCreateIfMissing(options.CreateIfMissing),
            logFrame
        );
        CommitTransaction(transaction, logFrame);
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtUpdateWithLockFailure(CurrentExceptionMessage()));
        throw;
    }
    INFRA_LOG_INFO(NEventlog::TYtUpdateWithLockSuccess());
}

void UpdateWithLock(
    const NYT::IClientBasePtr ytClient,
    const NYT::TYPath& lockPath,
    const NYT::TYPath& path,
    const NYT::TNode& data,
    NInfra::TLogFramePtr logFrame
) {
    UpdateWithLock(ytClient, lockPath, path, data, TUpdateWithLockOptions(), logFrame);
}

void UpdateWithLock(
    const NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    const NYT::TNode& data,
    const TUpdateWithLockOptions& options,
    NInfra::TLogFramePtr logFrame
) {
    UpdateWithLock(ytClient, path, path, data, options, logFrame);
}

void UpdateWithLock(
    const NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    const NYT::TNode& data,
    NInfra::TLogFramePtr logFrame
) {
    UpdateWithLock(ytClient, path, data, TUpdateWithLockOptions(), logFrame);
}

void UpdateWithoutLock(
    const NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    const NYT::TNode& data,
    const TUpdateOptions& options,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(MakeYtUpdateWithoutLockEvent(path, data, options));
    try {
        if (options.CreateIfMissing) {
            CreateCypressDocumentIfMissing(ytClient, path, logFrame);
        }
        Set(ytClient, path, data, logFrame);
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtUpdateWithoutLockFailure(CurrentExceptionMessage()));
        throw;
    }
    INFRA_LOG_INFO(NEventlog::TYtUpdateWithoutLockSuccess());
}

void UpdateWithoutLock(
    const NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    const NYT::TNode& data,
    NInfra::TLogFramePtr logFrame
) {
    UpdateWithoutLock(ytClient, path, data, TUpdateOptions(), logFrame);
}

////////////////////////////////////////////////////////////////////////////////

NYT::TNode TryGetNodeData(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    const NYT::TGetOptions& options,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(NEventlog::TYtTryGetNodeData(path));

    try {
        NYT::TNode result = Get(ytClient, path, options, logFrame);
        INFRA_LOG_INFO(NEventlog::TYtTryGetNodeDataSuccess(true));
        return result;
    } catch (const NYT::TErrorResponse& error) {
        if (!error.IsResolveError()) {
            INFRA_LOG_INFO(NEventlog::TYtTryGetNodeDataFailure(error.what()));
            throw;
        }
        INFRA_LOG_INFO(NEventlog::TYtTryGetNodeDataSuccess(false));
        return NYT::TNode();
    }
}

NYT::TNode TryGetNodeData(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NInfra::TLogFramePtr logFrame
) {
    return TryGetNodeData(ytClient, path, NYT::TGetOptions{}, logFrame);
}

////////////////////////////////////////////////////////////////////////////////

THashSet<TString> ListNodesByFilter(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    const IFilter& filter,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(NEventlog::TYtListNodesByFilter(path));

    try {
        NYT::TAttributeFilter attributeFilter;
        for (const TString& attribute : filter.AttributesToRequest()) {
            attributeFilter.AddAttribute(std::move(attribute));
        }
        const NYT::TListOptions listOptions = NYT::TListOptions()
            .AttributeFilter(attributeFilter);

        const NYT::TNode::TListType nodes = List(ytClient, path, listOptions, logFrame);

        THashSet<TString> result;
        for (const NYT::TNode& node : nodes) {
            if (filter.Filter(node)) {
                result.insert(node.AsString());
            }
        }

        INFRA_LOG_INFO(NEventlog::TYtListNodesByFilterSuccess(result.size()));
        return result;
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtListNodesByFilterFailure(CurrentExceptionMessage()));
        throw;
    }
}

class TEmptyFilter : public IFilter {
public:
    TVector<TString> AttributesToRequest() const override {
        return {};
    }

    bool Filter(const NYT::TNode&) const override {
        return true;
    }
};

THashSet<TString> ListNodes(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NInfra::TLogFramePtr logFrame
) {
    return ListNodesByFilter(ytClient, path, TEmptyFilter(), logFrame);
}

class TLockedNodesListFilter : public IFilter {
public:
    TVector<TString> AttributesToRequest() const override {
        return {
            "locks",
        };
    }

    bool Filter(const NYT::TNode& node) const override {
        if (!node.GetAttributes().AsMap().contains("locks")) {
            return false;
        }
        for (const NYT::TNode& lock : node.GetAttributes().AsMap().at("locks").AsList()) {
            if (FromString<NYT::ELockMode>(lock["mode"].AsString()) != NYT::ELockMode::LM_EXCLUSIVE) {
                continue;
            }

            if (lock["state"] != "acquired") {
                continue;
            }

            return true;
        }

        return false;
    }
};

THashSet<TString> ListLockedNodes(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NInfra::TLogFramePtr logFrame
) {
    static TLockedNodesListFilter filter;
    return ListNodesByFilter(ytClient, path, filter, logFrame);
}

TVector<NYT::TNode> ListNodesData(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(NEventlog::TYtListNodesData(path));

    try {
        NYT::ITransactionPtr transaction = StartTransaction(ytClient, logFrame);
        NYT::ILockPtr snapshotLock = Lock(
            transaction,
            path, 
            NYT::LM_SNAPSHOT,
            NYT::TLockOptions()
                .Waitable(true),
            logFrame
        );
        WaitForLockAcquired(snapshotLock, logFrame);

        const NYT::TNode::TListType nodes = List(
            transaction,
            "#" + GetGuidAsString(snapshotLock->GetLockedNodeId()),
            logFrame
        );

        TVector<NYT::TNode> result;
        result.reserve(nodes.size());
        for (const NYT::TNode& node : nodes) {
            NYT::TNode resultNode;
            resultNode["name"] = node.AsString();

            const NYT::TYPath nodePath = NYT::JoinYPaths("#" + GetGuidAsString(snapshotLock->GetLockedNodeId()), node.AsString());
            resultNode["data"] = Get(transaction, nodePath, logFrame);

            result.emplace_back(std::move(resultNode));
        }

        INFRA_LOG_INFO(NEventlog::TYtListNodesDataSuccess(result.size()));
        return result;
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtListNodesDataFailure(CurrentExceptionMessage()));
        throw;
    }
}

////////////////////////////////////////////////////////////////////////////////

bool IsNodeLocked(
    NYT::IClientBasePtr ytClient,
    const NYT::TYPath& path,
    NInfra::TLogFramePtr logFrame
) {
    INFRA_LOG_INFO(NEventlog::TYtCheckIsNodeLocked(path));

    try {
        static TLockedNodesListFilter filter;

        NYT::TAttributeFilter attributeFilter;
        for (const TString& attribute : filter.AttributesToRequest()) {
            attributeFilter.AddAttribute(std::move(attribute));
        }

        const NYT::TGetOptions getOptions = NYT::TGetOptions()
            .AttributeFilter(attributeFilter);

        const NYT::TNode node = Get(ytClient, path, getOptions, logFrame);

        bool result = filter.Filter(node);
        INFRA_LOG_INFO(NEventlog::TYtCheckIsNodeLockedSuccess(result));
        return result;
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TYtCheckIsNodeLockedFailure(CurrentExceptionMessage()));
        throw;
    }
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NYtHelpers
