#include "make_log_events.h"

#include <infra/libs/yp_dns/zone/eventlog/make_events.h>
#include <infra/libs/yp_dns/zone/zone.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/yson/node/node_io.h>

namespace NYpDns::NDynamicZones {

////////////////////////////////////////////////////////////////////////////////

NEventlog::TZoneState MakeZoneStateEvent(const NYT::TNode& state) {
    NEventlog::TZoneState event;
    if (state["current_state"].IsString()) {
        event.SetCurrentState(state["current_state"].AsString());
    }
    if (state["target_state"].IsString()) {
        event.SetTargetState(state["target_state"].AsString());
    }
    return event;
}

NEventlog::TZoneState MakeZoneStateEvent(const NJson::TJsonValue& state) {
    NEventlog::TZoneState event;
    if (state["current_state"].IsString()) {
        event.SetCurrentState(state["current_state"].GetString());
    }
    if (state["target_state"].IsString()) {
        event.SetTargetState(state["target_state"].GetString());
    }
    return event;
}

NEventlog::TZoneData MakeZoneDataEvent(const TZone& zone, const NJson::TJsonValue& state) {
    NEventlog::TZoneData event;
    event.SetId(zone.GetId());
    event.SetName(zone.GetName());
    *event.MutableState() = MakeZoneStateEvent(state);
    *event.MutableConfig() = MakeZoneConfigEvent(zone.Config());
    return event;
}

NEventlog::TZoneData MakeZoneDataEvent(const NYT::TNode& zoneData) {
    NEventlog::TZoneData event;
    if (zoneData["data"]["id"].IsString()) {
        event.SetId(zoneData["data"]["id"].AsString());
    }
    if (zoneData["data"]["name"].IsString()) {
        event.SetId(zoneData["data"]["name"].AsString());
    }
    if (zoneData["data"]["state"].IsMap()) {
        *event.MutableState() = MakeZoneStateEvent(zoneData["data"]["state"]);
    }
    if (zoneData["data"]["config"].IsMap()) {
        TZone zone(zoneData["data"]["config"]);
        *event.MutableConfig() = MakeZoneConfigEvent(zone.Config());
    }
    return event;
}

NEventlog::TDnsZoneYpObject MakeDnsZoneYpObjectEvent(const TDnsZoneYpObject& ypObject) {
    const auto& [ypTimestamp, zone] = ypObject;
    NEventlog::TDnsZoneYpObject event;
    event.SetYpTimestamp(ypTimestamp);
    if (zone.Defined()) {
        const NProtobufJson::TProto2JsonConfig proto2JsonConfig =
            NProtobufJson::TProto2JsonConfig()
                .SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName);
        event.SetMeta(NProtobufJson::Proto2Json(zone->Meta(), proto2JsonConfig));
        event.SetSpec(NProtobufJson::Proto2Json(zone->Spec(), proto2JsonConfig));
        event.SetStatus(NProtobufJson::Proto2Json(zone->Status(), proto2JsonConfig));
        event.SetLabels(NJson::WriteJson(zone->Labels(), /* formatOutput */ false));
    }
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TZonesManagerCreateZone MakeZonesManagerCreateZoneEvent(const TZone& zone) {
    NEventlog::TZonesManagerCreateZone event;
    *event.MutableZone() = MakeZoneEvent(zone);
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TZonesManagerUpdateZone MakeZonesManagerUpdateZoneEvent(const TZone& zone) {
    NEventlog::TZonesManagerUpdateZone event;
    *event.MutableZone() = MakeZoneEvent(zone);
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TZonesManagerUpdateZoneData MakeZonesManagerUpdateZoneDataEvent(
    const TZone& zone,
    const NJson::TJsonValue& state
) {
    NEventlog::TZonesManagerUpdateZoneData event;
    *event.MutableData() = MakeZoneDataEvent(zone, state);
    return event;
}

NEventlog::TZonesManagerUpdateZoneDataValue MakeZonesManagerUpdateZoneDataValueEvent(const NYT::TNode& data) {
    NEventlog::TZonesManagerUpdateZoneDataValue event;
    event.SetData(NYT::NodeToYsonString(data));
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TZonesManagerUpdateZoneState MakeZonesManagerUpdateZoneStateEvent(
    const TZoneId& zoneId,
    const NJson::TJsonValue& state
) {
    NEventlog::TZonesManagerUpdateZoneState event;
    event.SetZoneId(zoneId);
    *event.MutableState() = MakeZoneStateEvent(state);
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TZonesManagerUpdateZoneCurrentState MakeZonesManagerUpdateZoneCurrentStateEvent(
    const TZoneId& zoneId,
    const NJson::TJsonValue& currentState
) {
    NEventlog::TZonesManagerUpdateZoneCurrentState event;
    event.SetZoneId(zoneId);
    if (currentState.IsString()) {
        event.SetCurrentState(currentState.GetString());
    } else {
        event.SetCurrentState(NJson::WriteJson(currentState, /* formatOutput */ false));
    }
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TZonesManagerListZonesResult MakeZonesManagerListZonesResultEvent(
    const TVector<TDnsZoneYpObject>& zones
) {
    NEventlog::TZonesManagerListZonesResult event;
    event.SetSize(zones.size());
    event.MutableZones()->Reserve(zones.size());
    for (const TDnsZoneYpObject& ypObject : zones) {
        *event.AddZones() = MakeDnsZoneYpObjectEvent(ypObject);
    }
    return event;
}

////////////////////////////////////////////////////////////////////////////////

NEventlog::TZonesManagerGetDnsZoneYpObjectSuccess MakeZonesManagerGetDnsZoneYpObjectSuccessEvent(
    TDnsZoneYpObject& zone
) {
    NEventlog::TZonesManagerGetDnsZoneYpObjectSuccess event;
    *event.MutableZone() = MakeDnsZoneYpObjectEvent(zone);
    return event;
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NDynamicZones
