#include "bridge.h"

#include <infra/libs/yp_dns/dynamic_zones/services/protos/events/bridge_events_decl.ev.pb.h>

#include <infra/libs/yp_dns/dynamic_zones/helpers/yt/yt.h>

#include <infra/libs/background_thread/background_thread.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/fwd.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <library/cpp/yson/node/node.h>
#include <library/cpp/yson/node/node_io.h>

#include <util/system/backtrace.h>

namespace NYpDns::NDynamicZones::NBridgeService {

////////////////////////////////////////////////////////////////////////////////

struct TActiveInstances {
    ui32 Version = 0;
    THashSet<TString> Ids;
};

struct TInstancesStates {
    ui32 Version = 0;
    TVector<NYT::TNode> States;
};

////////////////////////////////////////////////////////////////////////////////

namespace NEventlog {

TBridgeInstancesStates MakeInstancesStatesEvent(
    const TVector<NYT::TNode>& instancesStates
) {
    TBridgeInstancesStates event;
    event.MutableData()->Reserve(instancesStates.size());
    for (const NYT::TNode& instanceState : instancesStates) {
        event.AddData(NYT::NodeToYsonString(instanceState));
    }
    return event;
}

TBridgeStateServiceInstancesStates MakeBridgeStateServiceInstancesStatesEvent(
    const TInstancesStates& instancesStates
) {
    TBridgeStateServiceInstancesStates event;
    event.SetVersion(instancesStates.Version);
    *event.MutableStates() = MakeInstancesStatesEvent(instancesStates.States);
    return event;
}

TBridgeStateServiceActiveInstances MakeBridgeStateServiceActiveInstancesEvent(
    const TActiveInstances& instances
) {
    TBridgeStateServiceActiveInstances event;
    event.SetVersion(instances.Version);
    event.MutableInstances()->Reserve(instances.Ids.size());
    for (TString instance : instances.Ids) {
        event.AddInstances(std::move(instance));
    }
    return event;
}

} // namespace NEventlog

////////////////////////////////////////////////////////////////////////////////

class TStateService::TImpl {
    using TActiveInstancesPtr = TAtomicSharedPtr<const TActiveInstances>;
    using TInstancesStatesPtr = TAtomicSharedPtr<const TInstancesStates>;

public:
    TImpl(
        TStateServiceConfig config,
        TStateServiceCommonOptions commonOptions
    )
        : Config_(std::move(config))
        , CommonOptions_(std::move(commonOptions))
        , Logger_(Config_.GetLoggerConfig())
        , YtClient_(NYT::CreateClient(
            CommonOptions_.YtProxy,
            NYT::TCreateClientOptions()
                .Token(CommonOptions_.YtToken)
        ))
        , InstancesCypressPath_(NYT::JoinYPaths(CommonOptions_.CypressRootPath, "instances"))
        , InstancesStateCypressPath_(NYT::JoinYPaths(CommonOptions_.CypressRootPath, "state", "instances"))
        , CacheUpdater_([this] { TryUpdateCache(); }, TDuration::Seconds(1))
    {
    }

    void Start(NInfra::TLogFramePtr logFrame) {
        NYtHelpers::CreateCypressMapNodeIfMissing(YtClient_, InstancesCypressPath_, logFrame);
        NYtHelpers::CreateCypressMapNodeIfMissing(YtClient_, InstancesStateCypressPath_, logFrame);
        CacheUpdater_.Start();
    }

    void Wait() {
    }

    void ReopenLogs() {
        Logger_.ReopenLog();
    }

    void ReportConfiguration(
        NInfra::TRequestPtr<NApi::TReqReportConfiguration> request,
        NInfra::TReplyPtr<NApi::TRspReportConfiguration> reply
    ) {
        NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();

        const NApi::TReqReportConfiguration& req = request->Get();

        // TODO: Decide whether we need to check that instance is registered

        NYT::TNode data = NYT::TNode::CreateMap();
        for (const NApi::TZone& zone : req.zones()) {
            data[zone.id()] = NApi::TZone::EStatus_Name(zone.status());
        }

        const NYT::TYPath instanceStatePath = NYT::JoinYPaths(InstancesStateCypressPath_, req.instance_id());
        NYtHelpers::UpdateWithoutLock(YtClient_, instanceStatePath, data, logFrame);

        NApi::TRspReportConfiguration rsp;
        reply->Set(rsp);
    }

    bool AreAllInstancesHasZone(const TZoneId& zoneId, NInfra::TLogFramePtr logFrame) const {
        TActiveInstances activeInstances;
        if (TActiveInstancesPtr cachedActiveInstances = GetCachedActiveInstances()) {
            activeInstances = *cachedActiveInstances;
            INFRA_LOG_INFO(NEventlog::TBridgeStateServiceActiveInstancesVersion(activeInstances.Version));
        } else if (TActiveInstancesPtr currentActiveInstances = RequestActiveInstances(logFrame)) {
            activeInstances = *currentActiveInstances;
            INFRA_LOG_INFO(NEventlog::MakeBridgeStateServiceActiveInstancesEvent(activeInstances));
        } else {
            ythrow yexception() << "Failed to get list of active instances";
        }

        if (activeInstances.Ids.size() < Config_.GetMinInstancesNumber()) {
            ythrow yexception()
                << "Not enough Bridge instances are active: "
                << activeInstances.Ids.size()
                << ", need " << Config_.GetMinInstancesNumber();
        }

        TInstancesStatesPtr instancesStates = GetCachedInstancesStates();
        if (instancesStates) {
            INFRA_LOG_INFO(NEventlog::TBridgeStateServiceInstancesStatesVersion(instancesStates->Version));
        } else {
            instancesStates = RequestInstancesStates(logFrame);
            INFRA_LOG_INFO(NEventlog::MakeBridgeStateServiceInstancesStatesEvent(*instancesStates));
        }

        if (!instancesStates) {
            ythrow yexception() << "Failed to get instances states";
        }

        for (const NYT::TNode& instanceState : instancesStates->States) {
            if (!activeInstances.Ids.erase(instanceState["name"].AsString())) {
                continue;
            }

            const NYT::TNode* stateData = instanceState["data"].AsMap().FindPtr(zoneId);
            if (!stateData) {
                return false;
            }

            NApi::TZone::EStatus status;
            NApi::TZone::EStatus_Parse(stateData->AsString(), &status);

            if (status != NApi::TZone::OK) {
                return false;
            }
        }

        if (activeInstances.Ids.size() >= Config_.GetMaxInstancesWithoutState()) {
            ythrow yexception()
                << "Too many Bridge instances do not have state: "
                << activeInstances.Ids.size()
                << ", max allowed " << Config_.GetMaxInstancesWithoutState();
        }

        return true;
    }

private:
    void TryUpdateCache() {
        NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
        INFRA_LOG_INFO(NEventlog::TBridgeStateServiceUpdateCacheCycleStart());

        bool ok = true;
        try {
            UpdateInstancesStatesCache(logFrame);
        } catch (...) {
            ok = false;
            INFRA_LOG_ERROR(NEventlog::TBridgeStateServiceUpdateCacheCycleFailure(
                CurrentExceptionMessage(),
                TBackTrace::FromCurrentException().PrintToString()));
        }

        try {
            UpdateActiveInstancesCache(logFrame);
        } catch (...) {
            ok = false;
            INFRA_LOG_ERROR(NEventlog::TBridgeStateServiceUpdateCacheCycleFailure(
                CurrentExceptionMessage(),
                TBackTrace::FromCurrentException().PrintToString()));
        }

        if (ok) {
            INFRA_LOG_INFO(NEventlog::TBridgeStateServiceUpdateCacheCycleSuccess());
        }
    }

    TActiveInstancesPtr RequestActiveInstances(NInfra::TLogFramePtr logFrame) const {
        return MakeAtomicShared<TActiveInstances>(
            ActiveInstancesVersionCounter_++,
            NYtHelpers::ListLockedNodes(
                YtClient_,
                InstancesCypressPath_,
                logFrame
            )
        );
    }

    void UpdateActiveInstancesCache(NInfra::TLogFramePtr logFrame) {
        INFRA_LOG_INFO(NEventlog::TBridgeStateServiceUpdateActiveInstancesCacheStart());

        try {
            TActiveInstancesPtr activeInstances = RequestActiveInstances(logFrame);
            if (!activeInstances) {
                ythrow yexception() << "Failed to request active instances";
            }

            INFRA_LOG_INFO(NEventlog::MakeBridgeStateServiceActiveInstancesEvent(*activeInstances));

            SetCachedActiveInstances(activeInstances);
            INFRA_LOG_INFO(
                NEventlog::TBridgeStateServiceUpdateActiveInstancesCacheSuccess());
        } catch (...) {
            INFRA_LOG_ERROR(NEventlog::TBridgeStateServiceUpdateActiveInstancesCacheFailure(
                CurrentExceptionMessage()));
            throw;
        }
    }

    TActiveInstancesPtr GetCachedActiveInstances() const {
        TGuard<TMutex> guard(CachedActiveInstancesMutex_);
        return CachedActiveInstances_;
    }

    void SetCachedActiveInstances(TActiveInstancesPtr activeInstances) {
        TGuard<TMutex> guard(CachedActiveInstancesMutex_);
        CachedActiveInstances_ = activeInstances;
    }

    TInstancesStatesPtr RequestInstancesStates(NInfra::TLogFramePtr logFrame) const {
        return MakeAtomicShared<TInstancesStates>(
            InstancesStatesVersionCounter_++,
            NYtHelpers::ListNodesData(
                YtClient_,
                InstancesStateCypressPath_,
                logFrame
            )
        );
    }

    void UpdateInstancesStatesCache(NInfra::TLogFramePtr logFrame) {
        INFRA_LOG_INFO(NEventlog::TBridgeStateServiceUpdateInstancesStatesCacheStart());

        try {
            TInstancesStatesPtr instancesStates = RequestInstancesStates(logFrame);
            if (!instancesStates) {
                ythrow yexception() << "Failed to request instances states";
            }

            INFRA_LOG_INFO(NEventlog::MakeBridgeStateServiceInstancesStatesEvent(*instancesStates));

            SetCachedInstancesStates(instancesStates);
            INFRA_LOG_INFO(NEventlog::TBridgeStateServiceUpdateInstancesStatesCacheSuccess());
        } catch (...) {
            INFRA_LOG_ERROR(NEventlog::TBridgeStateServiceUpdateInstancesStatesCacheFailure(
                CurrentExceptionMessage()));
            throw;
        }
    }

    TInstancesStatesPtr GetCachedInstancesStates() const {
        TGuard<TMutex> guard(CachedInstancesStatesMutex_);
        return CachedInstancesStates_;
    }

    void SetCachedInstancesStates(TInstancesStatesPtr states) {
        TGuard<TMutex> guard(CachedInstancesStatesMutex_);
        CachedInstancesStates_ = states;
    }

private:
    const TStateServiceConfig Config_;
    const TStateServiceCommonOptions CommonOptions_;

    NInfra::TLogger Logger_;

    const NYT::IClientPtr YtClient_;
    const NYT::TYPath InstancesCypressPath_;
    const NYT::TYPath InstancesStateCypressPath_;

    mutable std::atomic<ui32> InstancesStatesVersionCounter_ = 0;
    TMutex CachedInstancesStatesMutex_;
    TInstancesStatesPtr CachedInstancesStates_;

    mutable std::atomic<ui32> ActiveInstancesVersionCounter_ = 0;
    TMutex CachedActiveInstancesMutex_;
    TActiveInstancesPtr CachedActiveInstances_;

    NInfra::TBackgroundThread CacheUpdater_;
};

////////////////////////////////////////////////////////////////////////////////

TStateService::TStateService(
    TStateServiceConfig config,
    TStateServiceCommonOptions commonOptions
)
    : Impl_(MakeHolder<TImpl>(std::move(config), std::move(commonOptions)))
{
}

TStateService::~TStateService() = default;

void TStateService::Start(NInfra::TLogFramePtr logFrame) {
    Impl_->Start(logFrame);
}

void TStateService::Wait() {
    Impl_->Wait();
}

void TStateService::ReopenLogs() {
    Impl_->ReopenLogs();
}

void TStateService::ReportConfiguration(
    NInfra::TRequestPtr<NApi::TReqReportConfiguration> request,
    NInfra::TReplyPtr<NApi::TRspReportConfiguration> reply
) {
    Impl_->ReportConfiguration(request, reply);
}

bool TStateService::AreAllInstancesHasZone(
    const TZoneId& zoneId,
    NInfra::TLogFramePtr logFrame
) const {
    return Impl_->AreAllInstancesHasZone(zoneId, logFrame);
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NDynamicZones::NBridgeService
