#pragma once

#include <infra/libs/logger/logger.h>
#include <infra/libs/service_iface/errors.h>
#include <infra/libs/service_iface/router.h>
#include <infra/libs/service_iface/str_iface.h>

#include <mapreduce/yt/interface/fwd.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <contrib/libs/grpc/include/grpc++/grpc++.h>

namespace NYpDns::NDynamicZones {

////////////////////////////////////////////////////////////////////////////////

struct TStateServiceCommonOptions {
    TString YtProxy;
    TString YtToken;
    NYT::TYPath CypressRootPath;

    TStateServiceCommonOptions& SetYtProxy(TString value) {
        YtProxy = std::move(value);
        return *this;
    }

    TStateServiceCommonOptions& SetYtToken(TString value) {
        YtToken = std::move(value);
        return *this;
    }

    TStateServiceCommonOptions& SetCypressRootPath(NYT::TYPath value) {
        CypressRootPath = std::move(value);
        return *this;
    }

    TStateServiceCommonOptions& AppendToCypressRootPath(NYT::TYPath value) {
        CypressRootPath = NYT::JoinYPaths(CypressRootPath, std::move(value));
        return *this;
    }
};

////////////////////////////////////////////////////////////////////////////////

class IService : public TNonCopyable {
public:
    virtual ~IService() = default;

    virtual void Start(NInfra::TLogFramePtr logFrame) = 0;
    virtual void Wait() = 0;
    virtual void ReopenLogs() = 0;
};

////////////////////////////////////////////////////////////////////////////////

class TGrpcCallHandler {
public:
    template <typename TRequest, typename TReply, typename TCallback>
    grpc::Status Call(grpc::ServerContext* ctx, const TRequest* request, TReply* reply, TCallback&& callback) {
        try {
            NInfra::TAttributes attributes;
            for (const auto& [key, value] : ctx->client_metadata()) {
                attributes.insert({TString(key.data(), key.size()), TString(value.data(), value.size())});
            }
            attributes.emplace(NInfra::NServiceAttributes::RemoteAddressAttributeName(), ctx->peer());

            NInfra::TRequestPtr<TRequest> requestPtr(new NInfra::TProtoRequest<TRequest>(TStringBuf(), *request, attributes));
            NInfra::TAttributes responseAttributes;
            NInfra::TReplyPtr<TReply> replyPtr(new NInfra::TProtoReply<TReply>(*reply, responseAttributes));
            callback(requestPtr, replyPtr);
            return grpc::Status::OK;
        } catch (const NInfra::TAccessDeniedError& ex) {
            return grpc::Status(grpc::StatusCode::UNAUTHENTICATED, std::string(CurrentExceptionMessage().data()));
        } catch (const NInfra::TServiceError& ex) {
            return grpc::Status(ex.GetGrpcStatusCode(), std::string(CurrentExceptionMessage().data()));
        } catch (...) {
            return grpc::Status(grpc::StatusCode::INTERNAL, std::string(CurrentExceptionMessage().data()));
        }
    }
};

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NDynamicZones
