#include "zones_manager.h"

#include "error.h"
#include "make_log_events.h"

#include <infra/libs/yp_dns/dynamic_zones/helpers/yt/yt.h>
#include <infra/libs/yp_dns/dynamic_zones/protos/events/events_decl.ev.pb.h>

#include <infra/libs/yp_dns/yp_helpers/misc/token.h>

#include <infra/libs/logger/logger.h>

#include <yp/cpp/yp/token.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/retry/retry.h>
#include <library/cpp/yson/node/node_io.h>

#include <util/generic/guid.h>
#include <util/system/backtrace.h>

namespace NYpDns::NDynamicZones {

////////////////////////////////////////////////////////////////////////////////

namespace {

constexpr ui64 STORAGE_FORMAT_VERSION = 2;

} // anonymous namespace

////////////////////////////////////////////////////////////////////////////////

TZoneObjectCreateOptions::TZoneObjectCreateOptions(const TZoneObjectCreateConfig& config)
    : Acl(
        config.GetZonesObjectsAcl().begin(),
        config.GetZonesObjectsAcl().end())
{
}

TZoneManagerOptions::TZoneManagerOptions(const TZonesManagerConfig& config)
    : YpAddress(config.GetYpAddress())
    , YpClientToken(NYpDns::FindYpClientToken())
    , YpReplicaToken(NYpDns::FindYpReplicaToken())
    , ReplicasConfigHolder(
        NUpdatableProtoConfig::CreateConfigHolder<NUpdatableProtoConfig::TStaticConfigHolder>(
            config.GetDnsZonesReplicasConfig()))
    , StaticZones(
        config.GetStaticZones().GetZones().begin(),
        config.GetStaticZones().GetZones().end())
    , ZoneObjectCreateOptions(config.GetZoneObjectCreateConfig())
    , ZonesStatesUpdateLoggerConfig(config.GetZonesStatesUpdateLoggerConfig())
{
}

////////////////////////////////////////////////////////////////////////////////

TZonesManager::TZonesManager(
    TZoneManagerOptions options,
    IZonesStateCoordinatorPtr stateCoordinator
)
    : Options_(std::move(options))
    , ZonesStatesUpdateLogger_(Options_.ZonesStatesUpdateLoggerConfig)
    , ReplicasLogger_(NInfra::TLogger(
        Options_.ReplicasConfigHolder->Config()->GetLoggerConfig()))
    , YpClient_(NYP::NClient::CreateClient(
        NYP::NClient::TClientOptions()
            .SetAddress(Options_.YpAddress)
            .SetEnableBalancing(true)
            .SetEnableSsl(true)
            .SetTimeout(TDuration::Seconds(30))
            .SetToken(Options_.YpClientToken)))
    , DnsZonesReplica_(MakeAtomicShared<TDnsZonesReplica>(
        Options_.ReplicasConfigHolder
            ->Accessor<NYP::NYPReplica::TYPReplicaConfig>("YpReplicaConfig"),
        Options_.ReplicasConfigHolder
            ->Accessor<NYP::NYPReplica::TYPClusterConfig>("YpClusterConfig"),
        NYP::NYPReplica::TTableRulesHolder<TDnsZoneReplicaObject>(),
        Options_.YpReplicaToken,
        ReplicasLogger_,
        STORAGE_FORMAT_VERSION))
    , StateCoordinator_(stateCoordinator)
    , ZonesStatesUpdater_(
        MakeHolder<NInfra::TBackgroundThread>([this] {
            NInfra::TLogFramePtr logFrame = ZonesStatesUpdateLogger_.SpawnFrame();
            try {
                UpdateZonesStates(logFrame);
            } catch (...) {
                INFRA_LOG_ERROR(NEventlog::TZonesManagerUpdateZonesStatesCycleFailure(
                    CurrentExceptionMessage(),
                    TBackTrace::FromCurrentException().PrintToString()));
            }
        }, TDuration::Seconds(1)))
{
}

void TZonesManager::Start(NInfra::TLogFramePtr logFrame) {
    INFRA_LOG_INFO(NEventlog::TZonesManagerStart());

    try {
        DnsZonesReplica_->Start();
        StartUpdateZonesStates();
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TZonesManagerStartFailure(
            CurrentExceptionMessage(),
            TBackTrace::FromCurrentException().PrintToString()));
        throw;
    }
    INFRA_LOG_INFO(NEventlog::TZonesManagerStartSuccess());
}

void TZonesManager::StartUpdateZonesStates() {
    ZonesStatesUpdater_->Start();
}

TDnsZoneYpObject TZonesManager::GetDnsZoneYpObject(
    const TZoneId& zoneId,
    NInfra::TLogFramePtr logFrame
) const try {
    INFRA_LOG_INFO(NEventlog::TZonesManagerGetDnsZoneYpObject(zoneId));

    auto replicaSnapshot = DnsZonesReplica_->GetReplicaSnapshot();

    const TDuration replicaAge = *DnsZonesReplica_->Age(replicaSnapshot);
    const TDuration maxAcceptableReplicaAge = TDuration::Seconds(10);
    INFRA_LOG_INFO(NEventlog::TDnsZonesReplicaAge(ToString(replicaAge), ToString(maxAcceptableReplicaAge)));

    TDnsZoneYpObject result;
    if (replicaAge < maxAcceptableReplicaAge) {
        INFRA_LOG_INFO(NEventlog::TZonesManagerGetDnsZoneYpObjectFromReplica());

        result.YpTimestamp = *DnsZonesReplica_->GetYpTimestamp(replicaSnapshot);
        if (auto lookupResult = DnsZonesReplica_->GetByKey<TDnsZoneReplicaObject>(zoneId, replicaSnapshot);
            lookupResult.Defined())
        {
            Y_ENSURE(lookupResult->Objects.size() == 1);
            result.Zone = lookupResult->Objects.front();
        }
    } else {
        INFRA_LOG_INFO(NEventlog::TZonesManagerGetDnsZoneYpObjectFromYP());

        const TRetryOptions retryOptions =
            TRetryOptions()
                .WithCount(3)
                .WithSleep(TDuration::MilliSeconds(500));

        result.YpTimestamp = *DoWithRetry<ui64, NYP::NClient::TResponseError>(
            [&] {
                // TODO log
                return YpClient_->GenerateTimestamp().GetValue(YpClient_->Options().Timeout() * 2);
            },
            [logFrame](const NYP::NClient::TResponseError& error) {
                Y_UNUSED(logFrame, error);
                // TODO log
            },
            retryOptions,
            /* throwLast */ true
        );

        NYP::NClient::TSelectorResult selectorResult =
            *DoWithRetry<NYP::NClient::TSelectorResult, NYP::NClient::TResponseError>(
                [&] {
                    return YpClient_->GetObject<NYP::NClient::TDnsZone>(
                        zoneId,
                        {"/meta/id", "/spec", "/status", "/labels"},
                        result.YpTimestamp,
                        NYP::NClient::TGetObjectOptions()
                            .SetIgnoreNonexistent(true)
                    ).GetValue(YpClient_->Options().Timeout() * 2);
                },
                [logFrame](const NYP::NClient::TResponseError& error) {
                    Y_UNUSED(logFrame, error);
                    // TODO log error
                },
                retryOptions,
                /* throwLast */ true
            );

        if (!selectorResult.Values().value_payloads().empty()) {
            NYP::NClient::TDnsZone& zone = result.Zone.ConstructInPlace();
            selectorResult.Fill(
                zone.MutableMeta()->mutable_id(),
                zone.MutableSpec(),
                zone.MutableStatus(),
                zone.MutableLabels());
        }
    }

    INFRA_LOG_INFO(MakeZonesManagerGetDnsZoneYpObjectSuccessEvent(result));

    return result;
} catch (...) {
    INFRA_LOG_ERROR(NEventlog::TZonesManagerGetDnsZoneYpObjectFailure(CurrentExceptionMessage()));
    throw;
}

void TZonesManager::CreateZone(const TZone& zone, NInfra::TLogFramePtr logFrame) try {
    INFRA_LOG_INFO(MakeZonesManagerCreateZoneEvent(zone));
    const TDnsZoneYpObject dnsZoneYpObject = GetDnsZoneYpObject(zone.GetId(), logFrame);
    const TZoneStateEntry state = StateCoordinator_->OnCreateZone(zone, dnsZoneYpObject.Zone, logFrame);
    UpdateZoneData(zone, dnsZoneYpObject, std::move(state), logFrame);
    INFRA_LOG_INFO(NEventlog::TZonesManagerCreateZoneSuccess());
} catch (...) {
    INFRA_LOG_ERROR(NEventlog::TZonesManagerCreateZoneFailure(CurrentExceptionMessage()));
    throw;
}

void TZonesManager::RemoveZone(const TZoneId& zoneId, NInfra::TLogFramePtr logFrame) try {
    INFRA_LOG_INFO(NEventlog::TZonesManagerRemoveZone(zoneId));

    const auto [ypTimestamp, ypZoneData] = GetDnsZoneYpObject(zoneId, logFrame);

    const TZoneStateEntry state = StateCoordinator_->OnRemoveZone(zoneId, ypZoneData, logFrame);

    if (ypZoneData.Defined()) {
        NYP::NClient::TTransactionPtr transaction =
            NYP::NClient::CreateTransactionFactory(*YpClient_)->CreateTransaction(ypTimestamp);
        transaction->RemoveObject<NYP::NClient::TDnsZone>(zoneId).GetValue(transaction->ClientOptions().Timeout() * 2);
        const ui64 commitTimestamp =
            transaction->Commit().GetValue(transaction->ClientOptions().Timeout() * 2);

        INFRA_LOG_INFO(NEventlog::TZonesManagerRemoveZoneSuccess(commitTimestamp));
    } else {
        INFRA_LOG_INFO(NEventlog::TZonesManagerRemoveZoneSuccess());
    }
} catch (...) {
    INFRA_LOG_ERROR(NEventlog::TZonesManagerRemoveZoneFailure(CurrentExceptionMessage()));
    throw;
}

void TZonesManager::UpdateZone(const TZone& zone, NInfra::TLogFramePtr logFrame) try {
    INFRA_LOG_INFO(MakeZonesManagerUpdateZoneEvent(zone));
    const TDnsZoneYpObject dnsZoneYpObject = GetDnsZoneYpObject(zone.GetId(), logFrame);
    TZoneStateEntry state = StateCoordinator_->OnUpdateZone(zone, dnsZoneYpObject.Zone, logFrame);
    UpdateZoneData(zone, dnsZoneYpObject, std::move(state), logFrame);
    INFRA_LOG_INFO(NEventlog::TZonesManagerUpdateZoneSuccess());
} catch (...) {
    INFRA_LOG_ERROR(NEventlog::TZonesManagerUpdateZoneFailure(CurrentExceptionMessage()));
    throw;
}

TVector<TZone> TZonesManager::ListZonesForService(
    const TServiceType& serviceType,
    NInfra::TLogFramePtr logFrame
) const try {
    INFRA_LOG_INFO(NEventlog::TZonesManagerListZonesForService(serviceType));

    if (!StateCoordinator_->HasService(serviceType)) {
        INFRA_LOG_ERROR(NEventlog::TUnknownServiceType(serviceType));
        throw TUnknownServiceType(serviceType);
    }

    const TVector<TDnsZoneYpObject> zones = ListAllZones(logFrame);
    INFRA_LOG_INFO(NEventlog::TZonesManagerListAllZonesDataResult(zones.size()));

    TVector<TZone> result;
    ui32 dynamicZonesNumber = 0;
    for (const auto& [ypTimestamp, zone] : zones) {
        // logFrame->LogEvent(TLOG_DEBUG, MakeZoneDataEvent(zoneData)); TODO
        if (StateCoordinator_->IsZoneReadyForService(
                serviceType,
                zone->Meta().id(),
                zone->Labels()["state"]))
        {
            ++dynamicZonesNumber;
            result.emplace_back(*zone);
        }
    }

    TMaybe<ui32> staticZonesNumber;
    if (StateCoordinator_->NeedStaticZones(serviceType)) {
        staticZonesNumber = Options_.StaticZones.size();
        result.reserve(result.size() + Options_.StaticZones.size());
        std::copy(Options_.StaticZones.begin(), Options_.StaticZones.end(), std::back_inserter(result));
    }

    INFRA_LOG_INFO(NEventlog::TZonesManagerListZonesForServiceSuccess(
        result.size(),
        dynamicZonesNumber,
        staticZonesNumber.Defined(),
        staticZonesNumber.GetOrElse(0)));

    return result;
} catch (...) {
    INFRA_LOG_ERROR(NEventlog::TZonesManagerListZonesForServiceFailure(CurrentExceptionMessage()));
    throw;
}

void TZonesManager::ReopenLogs() {
    ZonesStatesUpdateLogger_.ReopenLog();
    ReplicasLogger_.ReopenLog();
}

void TZonesManager::UpdateZonesStates(NInfra::TLogFramePtr logFrame) {
    INFRA_LOG_INFO(NEventlog::TZonesStateManagerUpdateZonesStates());

    try {
        TVector<TDnsZoneYpObject> zones = ListAllZones(logFrame);

        std::partition(zones.begin(), zones.end(), [](const TDnsZoneYpObject& zoneObject) {
            return zoneObject.Zone.Defined() &&
                zoneObject.Zone->Labels()["state"]["current_state"] != zoneObject.Zone->Labels()["state"]["target_state"];
        });

        std::exception_ptr exception;
        for (const auto& dnsZoneYpObject : zones) {
            const auto& [ypTimestamp, dnsZone] = dnsZoneYpObject;
            try {
                TZoneStateEntry state = StateCoordinator_->GetState(
                    dnsZone->Meta().id(),
                    dnsZone,
                    logFrame);

                if (dnsZoneYpObject.Zone.Defined() &&
                    dnsZoneYpObject.Zone->Labels()["state"] == state)
                {
                    continue;
                }

                UpdateZoneCurrentState(
                    dnsZone->Meta().id(),
                    dnsZoneYpObject,
                    state["current_state"].GetString(),
                    logFrame);
            } catch (...) {
                INFRA_LOG_ERROR(NEventlog::TZonesStateManagerUpdateZonesStatesError(CurrentExceptionMessage()));
                if (!exception) {
                    exception = std::current_exception();
                }
            }
        }
        if (exception) {
            std::rethrow_exception(exception);
        }
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TZonesStateManagerUpdateZonesStatesFailure(
            CurrentExceptionMessage(),
            TBackTrace::FromCurrentException().PrintToString()));
        throw;
    }
    INFRA_LOG_INFO(NEventlog::TZonesStateManagerUpdateZonesStatesSuccess());
}

void TZonesManager::UpdateZoneData(
    const TZone& zone,
    const TDnsZoneYpObject& dnsZoneYpObject,
    TZoneStateEntry state,
    NInfra::TLogFramePtr logFrame
) try {
    INFRA_LOG_INFO(MakeZonesManagerUpdateZoneDataEvent(zone, state));

    const auto& [ypTimestamp, ypZoneData] = dnsZoneYpObject;

    NYP::NClient::TDnsZone targetYpZoneData = zone.ConfigToYpDnsZoneObject();
    (*targetYpZoneData.MutableLabels())["state"] = std::move(state);
    SetZoneObjectAcl(targetYpZoneData);

    NYP::NClient::TTransactionPtr transaction =
        NYP::NClient::CreateTransactionFactory(*YpClient_)->CreateTransaction(ypTimestamp);
    // logFrame->LogEvent(MakeZonesManagerUpdateZoneDataValueEvent(data));
    if (ypZoneData.Defined()) {
        TVector<NYP::NClient::TSetRequest> setRequests = {
            NYP::NClient::TSetRequest("/meta/acl", targetYpZoneData.Meta().acl()),
            NYP::NClient::TSetRequest("/spec", targetYpZoneData.Spec()),
            NYP::NClient::TSetRequest("/status", targetYpZoneData.Status()),
            NYP::NClient::TSetRequest("/labels/state", NYT::NodeFromJsonValue(targetYpZoneData.Labels()["state"])),
            NYP::NClient::TSetRequest("/labels/owners", NYT::NodeFromJsonValue(targetYpZoneData.Labels()["owners"])),
        };
        transaction->UpdateObject<NYP::NClient::TDnsZone>(
            zone.GetId(),
            std::move(setRequests),
            /* remove */ {}
        ).GetValue(transaction->ClientOptions().Timeout() * 2);
    } else {
        transaction->CreateObject(targetYpZoneData).GetValue(transaction->ClientOptions().Timeout() * 2);
    }
    const ui64 commitTimestamp =
        transaction->Commit().GetValue(transaction->ClientOptions().Timeout() * 2);

    INFRA_LOG_INFO(NEventlog::TZonesManagerUpdateZoneDataSuccess(commitTimestamp));
} catch (...) {
    INFRA_LOG_ERROR(NEventlog::TZonesManagerUpdateZoneDataFailure(CurrentExceptionMessage()));
    throw;
}

void TZonesManager::UpdateZoneState(
    const TZoneId& zoneId,
    const TDnsZoneYpObject& dnsZoneYpObject,
    const TZoneStateEntry& state,
    NInfra::TLogFramePtr logFrame
) try {
    INFRA_LOG_INFO(MakeZonesManagerUpdateZoneStateEvent(zoneId, state));

    const auto& [ypTimestamp, ypZoneData] = dnsZoneYpObject;

    if (!ypZoneData.Defined()) {
        ythrow yexception()
            << "Can not update state for zone " << zoneId << ": it does not exist"
            << " (yp_timestamp: " << ypTimestamp << ")";
    }

    NYP::NClient::TTransactionPtr transaction =
        NYP::NClient::CreateTransactionFactory(*YpClient_)->CreateTransaction(ypTimestamp);
    transaction->UpdateObject<NYP::NClient::TDnsZone>(
        zoneId,
        /* set */ {
            NYP::NClient::TSetRequest("/labels/state", NYT::NodeFromJsonValue(state))
        },
        /* remove */ {}
    ).GetValue(transaction->ClientOptions().Timeout() * 2);
    const ui64 commitTimestamp =
        transaction->Commit().GetValue(transaction->ClientOptions().Timeout() * 2);

    INFRA_LOG_INFO(NEventlog::TZonesManagerUpdateZoneStateSuccess(commitTimestamp));
} catch (...) {
    INFRA_LOG_ERROR(NEventlog::TZonesManagerUpdateZoneStateFailure(CurrentExceptionMessage()));
    throw;
}

void TZonesManager::UpdateZoneCurrentState(
    const TZoneId& zoneId,
    const TDnsZoneYpObject& dnsZoneYpObject,
    const TString& currentState,
    NInfra::TLogFramePtr logFrame
) try {
    INFRA_LOG_INFO(MakeZonesManagerUpdateZoneCurrentStateEvent(zoneId, currentState));

    const auto& [ypTimestamp, ypZoneData] = dnsZoneYpObject;

    if (!ypZoneData.Defined()) {
        ythrow yexception()
            << "Can not update state for zone " << zoneId << ": it does not exist"
            << " (yp_timestamp: " << ypTimestamp << ")";
    }

    NYP::NClient::TTransactionPtr transaction =
        NYP::NClient::CreateTransactionFactory(*YpClient_)->CreateTransaction(ypTimestamp);
    transaction->UpdateObject<NYP::NClient::TDnsZone>(
        zoneId,
        /* set */ {
            NYP::NClient::TSetRequest("/labels/state/current_state", currentState)
        },
        /* remove */ {}
    ).GetValue(transaction->ClientOptions().Timeout() * 2);
    const ui64 commitTimestamp =
        transaction->Commit().GetValue(transaction->ClientOptions().Timeout() * 2);

    INFRA_LOG_INFO(NEventlog::TZonesManagerUpdateZoneCurrentStateSuccess(commitTimestamp));
} catch (...) {
    INFRA_LOG_ERROR(NEventlog::TZonesManagerUpdateZoneCurrentStateFailure(CurrentExceptionMessage()));
    throw;
}

TVector<TDnsZoneYpObject> TZonesManager::ListAllZones(
    NInfra::TLogFramePtr logFrame
) const try {
    INFRA_LOG_INFO(NEventlog::TZonesManagerListAllZones());

    auto replicaSnapshot = DnsZonesReplica_->GetReplicaSnapshot();

    TVector<TDnsZoneYpObject> result;

    const TDuration replicaAge = *DnsZonesReplica_->Age(replicaSnapshot);
    const TDuration maxAcceptableReplicaAge = TDuration::Seconds(10);
    INFRA_LOG_INFO(NEventlog::TDnsZonesReplicaAge(ToString(replicaAge), ToString(maxAcceptableReplicaAge)));
    if (replicaAge < maxAcceptableReplicaAge) {
        INFRA_LOG_INFO(NEventlog::TZonesManagerListZonesFromReplica());
        const ui64 ypTimestamp = *DnsZonesReplica_->GetYpTimestamp(replicaSnapshot);

        INFRA_LOG_INFO(NEventlog::TZonesManagerListZonesYpTimestamp(ypTimestamp));

        TDnsZonesReplica::TListOptions<TDnsZoneReplicaObject> listOptions;
        listOptions.Snapshot = replicaSnapshot;

        const auto elements = DnsZonesReplica_->ListElements(listOptions);
        result.reserve(elements.size());
        for (const auto& [key, element] : elements) {
            Y_ENSURE(element.size() == 1);
            result.push_back(TDnsZoneYpObject{
                    ypTimestamp,
                element.front().ReplicaObject.GetObject()});
        }
    } else {
        INFRA_LOG_INFO(NEventlog::TZonesManagerListZonesFromYP());

        const TRetryOptions retryOptions =
            TRetryOptions()
                .WithCount(3)
                .WithSleep(TDuration::MilliSeconds(500));

        const ui64 ypTimestamp = *DoWithRetry<ui64, NYP::NClient::TResponseError>(
            [&] {
                // TODO log
                return YpClient_->GenerateTimestamp().GetValue(YpClient_->Options().Timeout() * 2);
            },
            [logFrame](const NYP::NClient::TResponseError& error) {
                Y_UNUSED(logFrame, error);
                // TODO log
            },
            retryOptions,
            /* throwLast */ true
        );

        INFRA_LOG_INFO(NEventlog::TZonesManagerListZonesYpTimestamp(ypTimestamp));

        TString continuationToken;
        while (true) {
            const NYP::NClient::TSelectObjectsOptions selectOptions =
                NYP::NClient::TSelectObjectsOptions()
                    .SetLimit(1000)
                    .SetContinuationToken(continuationToken);
            NYP::NClient::TSelectObjectsResult selectResult =
                *DoWithRetry<NYP::NClient::TSelectObjectsResult, NYP::NClient::TResponseError>(
                    [&] {
                        return YpClient_->SelectObjects<NYP::NClient::TDnsZone>(
                            {"/meta/id", "/spec", "/status", "/labels"},
                            /* filter */ "",
                            selectOptions,
                            ypTimestamp
                        ).GetValue(YpClient_->Options().Timeout() * 2);
                    },
                    [logFrame](const NYP::NClient::TResponseError& error) {
                        Y_UNUSED(logFrame, error);
                        // TODO log error
                    },
                    retryOptions,
                    /* throwLast */ true
                );

            continuationToken = selectResult.ContinuationToken;

            for (const NYP::NClient::TSelectorResult& selectorResult : selectResult.Results) {
                auto& [_, dnsZone] = result.emplace_back(TDnsZoneYpObject{
                    ypTimestamp,
                    NYP::NClient::TDnsZone{}});
                selectorResult.Fill(
                    dnsZone->MutableMeta()->mutable_id(),
                    dnsZone->MutableSpec(),
                    dnsZone->MutableStatus(),
                    dnsZone->MutableLabels()
                );
            }

            if (selectResult.Results.size() < selectOptions.Limit()) {
                break;
            }
        }
    }

    INFRA_LOG_INFO(MakeZonesManagerListZonesResultEvent(result));

    INFRA_LOG_INFO(NEventlog::TZonesManagerListAllZonesSuccess(result.size()));
    return result;
} catch (...) {
    INFRA_LOG_INFO(NEventlog::TZonesManagerListAllZonesFailure(CurrentExceptionMessage()));
    throw;
}

void TZonesManager::SetZoneObjectAcl(NYP::NClient::TDnsZone& zone) const {
    zone.MutableMeta()->clear_acl();

    if (Options_.ZoneObjectCreateOptions.Acl.empty()) {
        return;
    }

    zone.MutableMeta()->mutable_acl()->Reserve(Options_.ZoneObjectCreateOptions.Acl.size());
    for (const auto& ace : Options_.ZoneObjectCreateOptions.Acl) {
        *zone.MutableMeta()->add_acl() = ace;
    }
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NDynamicZones
