#include "client.h"

#include <infra/libs/yp_dns/dynamic_zones/zones_manager_service/api/api.grpc.pb.h>
#include <infra/libs/yp_dns/dynamic_zones/zones_manager_service/service_iface/api.h>

#include <infra/libs/service_iface/reply.h>
#include <infra/libs/service_iface/str_iface.h>

#include <contrib/libs/grpc/include/grpc++/channel.h>
#include <contrib/libs/grpc/include/grpc++/create_channel.h>

#include <util/generic/guid.h>

namespace NYpDns::NDynamicZones {

////////////////////////////////////////////////////////////////////////////////

class TRawZonesManagerServiceClient::TImpl {
public:
    TImpl(TZonesManagerServiceClientOptions options)
        : Options_(std::move(options))
        , Channel_(grpc::CreateChannel(Options_.Address, grpc::InsecureChannelCredentials()))
        , Stub_(NApi::TZonesManagerService::NewStub(Channel_).release())
    {
    }

    void ListZones(NInfra::TRequestPtr<NApi::TReqListZones> request, NInfra::TReplyPtr<NApi::TRspListZones> reply) const {
        GrpcRequest(request, reply, &NApi::TZonesManagerService::Stub::ListZones);
    }

    void CreateZone(NInfra::TRequestPtr<NApi::TReqCreateZone> request, NInfra::TReplyPtr<NApi::TRspCreateZone> reply) const {
        GrpcRequest(request, reply, &NApi::TZonesManagerService::Stub::CreateZone);
    }

    void RemoveZone(NInfra::TRequestPtr<NApi::TReqRemoveZone> request, NInfra::TReplyPtr<NApi::TRspRemoveZone> reply) const {
        GrpcRequest(request, reply, &NApi::TZonesManagerService::Stub::RemoveZone);
    }

private:
    template <typename TRequest, typename TReply, typename TCallback>
    void GrpcRequest(NInfra::TRequestPtr<TRequest> request, NInfra::TReplyPtr<TReply> reply, TCallback&& callback) const {
        TReply result;

        grpc::ClientContext context;
        context.set_deadline(std::chrono::system_clock::now() + std::chrono::milliseconds(Options_.Timeout.MilliSeconds()));
        for (const auto& [key, value] : request->Attributes()) {
            context.AddMetadata(key, value);
        }

        if (grpc::Status status = ((*Stub_).*callback)(&context, request->Get(), &result); !status.ok()) {
            ythrow yexception() << "Code: " << static_cast<ui64>(status.error_code()) << "\nMessage: " << status.error_message();
        }

        for (const auto& [key, value] : context.GetServerInitialMetadata()) {
            reply->SetAttribute(TString{key.data(), key.size()}, TString{value.data(), value.size()});
        }
        for (const auto& [key, value] : context.GetServerTrailingMetadata()) {
            reply->SetAttribute(TString{key.data(), key.size()}, TString{value.data(), value.size()});
        }

        reply->Set(result);
    }

private:
    const TZonesManagerServiceClientOptions Options_;

    std::shared_ptr<grpc::Channel> Channel_;
    THolder<NApi::TZonesManagerService::Stub> Stub_;
};

////////////////////////////////////////////////////////////////////////////////

TRawZonesManagerServiceClient::TRawZonesManagerServiceClient(TZonesManagerServiceClientOptions options)
    : Impl_(MakeHolder<TImpl>(std::move(options)))
{
}

TRawZonesManagerServiceClient::~TRawZonesManagerServiceClient() = default;

void TRawZonesManagerServiceClient::ListZones(NInfra::TRequestPtr<NApi::TReqListZones> request, NInfra::TReplyPtr<NApi::TRspListZones> reply) {
    Impl_->ListZones(request, reply);
}

void TRawZonesManagerServiceClient::CreateZone(NInfra::TRequestPtr<NApi::TReqCreateZone> request, NInfra::TReplyPtr<NApi::TRspCreateZone> reply) {
    Impl_->CreateZone(request, reply);
}

void TRawZonesManagerServiceClient::RemoveZone(NInfra::TRequestPtr<NApi::TReqRemoveZone> request, NInfra::TReplyPtr<NApi::TRspRemoveZone> reply) {
    Impl_->RemoveZone(request, reply);
}

////////////////////////////////////////////////////////////////////////////////

class TZonesManagerServiceClient::TImpl {
public:
    TImpl(TZonesManagerServiceClientOptions options)
        : RawClient_(std::move(options))
    {
    }


    NApi::TRspListZones ListZones(TString serviceType) {
        NApi::TReqListZones request;
        request.set_service_type(std::move(serviceType));

        NInfra::TAttributes attributes = MakeRequestAttributes();

        NApi::TRspListZones response;
        NInfra::TAttributes responseAttributes;
        RawClient_.ListZones(
            NInfra::RequestPtr<NInfra::TProtoRequest<NApi::TReqListZones>>("", std::move(request), std::move(attributes)),
            NInfra::ReplyPtr<NInfra::TProtoReply<NApi::TRspListZones>>(response, responseAttributes)
        );

        return response;
    }

    NApi::TRspCreateZone CreateZone(TZoneConfig zoneConfig) {
        NApi::TReqCreateZone request;
        *request.mutable_zone()->mutable_config() = std::move(zoneConfig);

        NInfra::TAttributes attributes = MakeRequestAttributes();

        NApi::TRspCreateZone response;
        NInfra::TAttributes responseAttributes;
        RawClient_.CreateZone(
            NInfra::RequestPtr<NInfra::TProtoRequest<NApi::TReqCreateZone>>("", std::move(request), std::move(attributes)),
            NInfra::ReplyPtr<NInfra::TProtoReply<NApi::TRspCreateZone>>(response, responseAttributes)
        );

        return response;
    }

    NApi::TRspRemoveZone RemoveZone(TZoneId zoneId) {
        NApi::TReqRemoveZone request;
        request.set_zone_id(std::move(zoneId));

        NInfra::TAttributes attributes = MakeRequestAttributes();

        NApi::TRspRemoveZone response;
        NInfra::TAttributes responseAttributes;
        RawClient_.RemoveZone(
            NInfra::RequestPtr<NInfra::TProtoRequest<NApi::TReqRemoveZone>>("", std::move(request), std::move(attributes)),
            NInfra::ReplyPtr<NInfra::TProtoReply<NApi::TRspRemoveZone>>(response, responseAttributes)
        );

        return response;
    }

private:
    NInfra::TAttributes MakeRequestAttributes() const {
        return {{
            {"request-id", CreateGuidAsString()},
        }};
    }

private:
    TRawZonesManagerServiceClient RawClient_;
};

////////////////////////////////////////////////////////////////////////////////

TZonesManagerServiceClient::TZonesManagerServiceClient(TZonesManagerServiceClientOptions options)
    : Impl_(MakeHolder<TImpl>(std::move(options)))
{
}

TZonesManagerServiceClient::~TZonesManagerServiceClient() = default;

NApi::TRspListZones TZonesManagerServiceClient::ListZones(TString serviceType) const {
    return Impl_->ListZones(std::move(serviceType));
}

NApi::TRspCreateZone TZonesManagerServiceClient::CreateZone(TZoneConfig zoneConfig) const {
    return Impl_->CreateZone(std::move(zoneConfig));
}

NApi::TRspRemoveZone TZonesManagerServiceClient::RemoveZone(TZoneId zoneId) const {
    return Impl_->RemoveZone(std::move(zoneId));
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NDynamicZones
