#include "service.h"

#include <infra/libs/yp_dns/dynamic_zones/zones_manager_service/eventlog/make_events.h>

#include <infra/libs/yp_dns/dynamic_zones/error.h>

#include <infra/contrib/pdns/power_dns/dnsname.hh>

#include <util/generic/yexception.h>

namespace NYpDns::NDynamicZones {

////////////////////////////////////////////////////////////////////////////////

TZonesManagerServiceOptions::TZonesManagerServiceOptions(
    const TZonesManagerServiceConfig& zonesManagerServiceConfig,
    IZonesStateCoordinatorPtr stateCoordinator
)
    : DefaultYpClustersForNewZone(
        zonesManagerServiceConfig.GetDefaultYpClustersForNewZone().begin(),
        zonesManagerServiceConfig.GetDefaultYpClustersForNewZone().end()
    )
    , AdditionalOwnersForNewZone(
        zonesManagerServiceConfig.GetAdditionalOwnersForNewZone().begin(),
        zonesManagerServiceConfig.GetAdditionalOwnersForNewZone().end()
    )
    , ZonesManagerOptions(zonesManagerServiceConfig.GetZonesManagerConfig())
    , StateCoordinator(stateCoordinator)
    , LoggerConfig(zonesManagerServiceConfig.GetLoggerConfig())
{
    Y_ENSURE(!DefaultYpClustersForNewZone.empty());
}

////////////////////////////////////////////////////////////////////////////////

TZonesManagerService::TZonesManagerService(TZonesManagerServiceOptions options)
    : Options_(std::move(options))
    , Logger_(Options_.LoggerConfig)
    , ZonesManager_(Options_.ZonesManagerOptions, Options_.StateCoordinator)
{
}

void TZonesManagerService::Start() {
    NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
    INFRA_LOG_INFO(NEventlog::TZonesManagerServiceStart());
    ZonesManager_.Start(logFrame);
}

void TZonesManagerService::ReopenLogs() {
    Logger_.ReopenLog();
    ZonesManager_.ReopenLogs();
}

void TZonesManagerService::ListZones(NInfra::TRequestPtr<NApi::TReqListZones> request, NInfra::TReplyPtr<NApi::TRspListZones> reply) {
    NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
    INFRA_LOG_INFO(MakeListZonesRequestEvent(request->Get(), request->Attributes()));

    const NApi::TReqListZones& req = request->Get();
    const TString& serviceType = req.service_type();

    NApi::TRspListZones rsp;
    try {
        TVector<TZone> zones = ZonesManager_.ListZonesForService(serviceType, logFrame);

        rsp.set_status(NApi::TRspListZones::OK);
        rsp.mutable_zones()->Reserve(zones.size());
        for (TZone& zone : zones) {
            *rsp.add_zones()->mutable_config() = std::move(*zone.MutableConfig());
        }
    } catch (const TUnknownServiceType& error) {
        rsp.set_status(NApi::TRspListZones::UNKNOWN_SERVICE_TYPE);
        rsp.mutable_zones()->Clear();
    }

    for (const NApi::TZone& zone : rsp.zones()) {
        INFRA_LOG_INFO(MakeListZoneResponseEvent(zone));
    }
    INFRA_LOG_INFO(MakeListZonesResponseEvent(rsp));

    reply->Set(rsp);
}

void TZonesManagerService::CreateZone(NInfra::TRequestPtr<NApi::TReqCreateZone> request, NInfra::TReplyPtr<NApi::TRspCreateZone> reply) {
    NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
    INFRA_LOG_INFO(MakeCreateZoneRequestEvent(request->Get(), request->Attributes()));

    NApi::TReqCreateZone req = request->Get();

    NApi::TRspCreateZone rsp;
    try {
        ValidateZoneConfig(req.zone().config());
    } catch (...) {
        INFRA_LOG_ERROR(NEventlog::TConfigValidationError(CurrentExceptionMessage()));

        rsp.set_status(NApi::TRspCreateZone::ERROR);
        rsp.set_message(CurrentExceptionMessage());

        INFRA_LOG_INFO(MakeCreateZoneResponseEvent(rsp));
        return;
    }

    FillMissingFieldsWithDefaults(*req.mutable_zone()->mutable_config());
    INFRA_LOG_INFO(MakeConfigAfterFillingMissingFieldsWithDefaultsEvent(req.zone().config()));

    try {
        ZonesManager_.CreateZone(req.zone().config(), logFrame);
        rsp.set_status(NApi::TRspCreateZone::OK);
        rsp.set_message(TStringBuilder() << "Zone \"" << req.zone().config().name() << "\" has been successfully created");
    } catch (...) {
        rsp.set_status(NApi::TRspCreateZone::ERROR);
        rsp.set_message(CurrentExceptionMessage());
    }

    INFRA_LOG_INFO(MakeCreateZoneResponseEvent(rsp));

    reply->Set(rsp);
}

void TZonesManagerService::RemoveZone(NInfra::TRequestPtr<NApi::TReqRemoveZone> request, NInfra::TReplyPtr<NApi::TRspRemoveZone> reply) {
    NInfra::TLogFramePtr logFrame = Logger_.SpawnFrame();
    INFRA_LOG_INFO(MakeRemoveZoneRequestEvent(request->Get(), request->Attributes()));

    const NApi::TReqRemoveZone& req = request->Get();

    NApi::TRspRemoveZone rsp;
    try {
        ZonesManager_.RemoveZone(req.zone_id(), logFrame);
        rsp.set_status(NApi::TRspRemoveZone::OK);
        rsp.set_message(TStringBuilder() << "Zone \"" << req.zone_id() << "\" has been successfully removed");
    } catch (...) {
        rsp.set_status(NApi::TRspRemoveZone::ERROR);
        rsp.set_message(CurrentExceptionMessage());
    }

    INFRA_LOG_INFO(MakeRemoveZoneResponseEvent(rsp));

    reply->Set(rsp);
}

////////////////////////////////////////////////////////////////////////////////

void TZonesManagerService::ValidateZoneConfig(const TZoneConfig& zoneConfig) const {
    try {
        DNSName(zoneConfig.GetName());
    } catch (...) {
        ythrow yexception() << "Invalid zone name: " << CurrentExceptionMessage();
    }

    Y_ENSURE(!zoneConfig.GetPrimaryNameserver().empty(), "PrimaryNameserver must be set");

    Y_ENSURE(!zoneConfig.GetNameservers().empty(), "List of nameservers must not be empty");

    Y_ENSURE(!zoneConfig.HasSelectRecordSetMode() || zoneConfig.GetSelectRecordSetMode() == ESelectRecordSetMode::MERGE,
        "SelectRecordSetMode must be set to MERGE");
}

void TZonesManagerService::FillMissingFieldsWithDefaults(TZoneConfig& zoneConfig) const {
    zoneConfig.SetSelectRecordSetMode(ESelectRecordSetMode::MERGE);

    if (zoneConfig.GetYPClusters().empty()) {
        for (const TString& cluster : Options_.DefaultYpClustersForNewZone) {
            zoneConfig.AddYPClusters(cluster);
        }
    }

    for (const TString& owner : Options_.AdditionalOwnersForNewZone) {
        zoneConfig.AddOwners(owner);
    }
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NYpDns::NDynamicZones
